(ns hello.interceptor
  (:require [io.pedestal.interceptor.helpers :as h]
            [io.pedestal.interceptor.chain :as c]
            [clojure.core.async :as async :refer [<! >! <!! chan alt! go go-loop onto-chan sliding-buffer]]))


(defn handler [context r]
  (try
    (-> context
        (assoc :request r)
        (c/execute)
        (:response))
    (catch Exception e
      (hash-map :io.pedestal.interceptor.chain/error (ex-data e)))))


(defmulti processor (fn [type _] type))

(defmethod processor :sequence
  [_ context]
  (let [response (mapv #(handler context %) (:request context))]
    (-> context
        (assoc :response response)
        (c/terminate))))


(defmethod processor :sequence-until
  [_ context]
  (let [response (reduce (fn [acc r]
                           (let [res (handler context r)]
                             (if (contains? res :io.pedestal.interceptor.chain/error)
                               (reduced (conj acc res))
                               (conj acc res)))
                           ) [] (:request context))]
    (-> context
        (assoc :response response)
        (c/terminate))))


(defn async-handler [context r identifier]
  (async/go
    (let [t-v 2000
          exec-ch (async/thread (handler context r))
          [v rch] (async/alts! [exec-ch (async/timeout t-v)])]
      (if (= rch exec-ch)
        (do
          [identifier v])
        (do
          [identifier (hash-map :io.pedestal.interceptor.chain/error
                                (ex-info "Timeout happened" {:request r})
                                )])))))



(defmethod processor :parallel
  [_ context]
  (let [request-list (:request context)
        response (->> (doall (map (fn [r order]
                                    (async-handler context r order)
                                    ) request-list (range)))
                      (async/merge)
                      (async/take (count request-list))
                      (async/into [])
                      (async/<!!)
                      (sort-by first)
                      (mapv second))]
    (-> context
        (assoc :response response)
        (c/terminate))))



(def sequence-interceptor (h/before (partial processor :sequence)))
(def sequence-until-error-interceptor (h/before (partial processor :sequence-until)))
(def parallel-thread-interceptor (h/before (partial processor :parallel)))