(ns dosql.core
  (:require
    [clojure.spec :as s]
    [spec-model.core :as sm]
    [dosql.core-impl :as impl]
    [dosql.compiler.core :as cc]
    [dosql.clj.fail :as f]
    [dosql.core-spec :as ds]
    [dosql.impl.param-spec-genarator :as ps]
    [dosql.selector :as sel]
    [dosql.jdbc-io :as io]
    [clojure.java.jdbc :as jdbc]
    [dosql.file-reader :as fr]))


(defn validate-input!
  [req-m]
  (if (s/valid? :dosql.core/user-input req-m)
    req-m
    (f/fail (s/explain-data :dosql.core/user-input req-m))))



(defn read-file
  ([file-name] (read-file file-name nil #_(imp/new-root-node)))
  ([file-name pc]
   (-> (fr/read-file file-name)
       (cc/do-compile file-name))))


(defn select-name [tms req-m]
  (let [name (:dosql.core/name req-m)
        group (:dosql.core/group req-m)
        name (if group
               (sel/select-name-for-groups tms group name)
               name)]
    (sel/select-name-by-name-coll tms name)))




(defn get-sql-statement
  [tms]
  (let [p (comp (filter #(contains? ds/dml (:dosql.core/dml %)))
                (map :dosql.core/sql)
                (filter (fn [v] (if (< 1 (count v))
                                  true false)))
                (map first)
                (map #(clojure.string/replace % #":\w+" "?")))]
    (into [] p (vals tms))))



(defn select-spec [tms req-m]
  (->> (select-name tms req-m)
       (map :dosql.core/spec)
       (ps/as-merge-spec)))



(defn check-input [req-m]
  (s/conform :dosql.core/user-input req-m)
  )

(defn pull
  "Read or query value from database. It will return as model map
   ds: datasource
   "
  [ds tms req-m]
  (if-let [r (f/failed? (validate-input! req-m))]
    r
    (-> (hash-map :dosql.core/ds ds
                  :dosql.core/op (or (:dosql.core/op req-m)
                                     :dosql.core/pull)
                  :dosql.core/user-request (s/conform :dosql.core/user-input req-m)
                  :dosql.core/tms tms)
        (impl/execute))))



(defn push!
  "Create, update or delete value in database. DB O/P will be run within transaction. "
  [ds tms req-m]
  (if-let [r (f/failed? (validate-input! req-m))]
    r
    (let [m (io/global-info-m tms [])
          {:keys [isolation read-only? rollback?]} m]
      ; (clojure.pprint/pprint (s/conform :dosql.core/user-input req-m))
      (jdbc/with-db-transaction
        [t-conn ds
         :isolation isolation
         :read-only? read-only?]
        (let [context (hash-map :dosql.core/ds t-conn
                                :dosql.core/user-request (s/conform :dosql.core/user-input req-m)
                                :dosql.core/op (or (:dosql.core/op req-m)
                                                   :dosql.core/push)
                                :dosql.core/tms tms
                                )
              result (impl/execute context)]
          (when rollback?
            (jdbc/db-set-rollback-only! t-conn))

          result))
      )))



(defn assoc-default-param [ds tms req-m]
  (if-let [r (f/failed? (validate-input! req-m))]
    r
    (let [context (hash-map :dosql.core/ds ds
                            :dosql.core/user-request (s/conform :dosql.core/user-input req-m)
                            :dosql.core/op (or (:dosql.core/op req-m)
                                               :dosql.core/push)
                            :dosql.core/tms tms)]
      (impl/relation-data context))))


(defn get-spec
  [tms]
  (ps/gen-spec tms))


(defn get-spec-str
  ([tms ns-name]
   (let [ns-identifier (ps/filename-as-keyword
                         (or (get-in tms [:_global_ :dosql.core/file-name]) "nofound.clj"))]
     (->> (ps/gen-spec tms)
          (sm/as-file-str (or ns-name ns-identifier)))))
  ([tms]
   (get-spec-str tms nil)))

