(ns dosql.selector
  (:require
    [dosql.clj.common :as cc]
    [dosql.clj.fail :as f]
    [clojure.set]))


(defn validate-name!
  [tm name-coll]
  (let [skey (into #{} (keys tm))
        op-key (into #{} name-coll)]
    (if (clojure.set/superset? skey op-key)
      tm
      (->> (clojure.set/difference op-key skey)
           (first)
           (str "Name not found ")
           (f/fail)))))


(defn validate-model!
  [tm-coll]
  (let [model-coll (mapv :dosql.core/model tm-coll)
        m (distinct model-coll)]
    (if (not= (count model-coll)
              (count m))
      (f/fail (str "Selecting duplicate model " model-coll))
      tm-coll)))


(defn filter-join-key-coll
  [join model-coll]
  (->> join
       (filter (fn [[_ _ rel d-table _ nr]]
                 (if (= rel :spec-model.core/rel-n-n)
                   (some #{(first nr)} model-coll)
                   (some #{d-table} model-coll))))
       (into [])))


(defn filter-join-key
  [coll]
  (let [model-key-coll (mapv :dosql.core/model coll)
        p (comp
            (cc/xf-skip-type #(= :dosql.core/dml-call (:dosql.core/dml %)))
            (map #(update-in % [:spec-model.core/join] filter-join-key-coll model-key-coll)))]
    (transduce p conj [] coll)))


(defn is-reserve?
  [tms coll]
  (if (->> (clojure.set/intersection
             (into #{} coll)
             (get-in tms [:_global_ :dosql.core/reserve-name]))
           (not-empty))
    true
    false))

;;;;;;;;;;;;;;;;;;;;;;;;;;;; Selecting impl ;;;;;;;;;;;;;;;;;;;;;;;;;;


(defn select-name-by-name-coll
  "Return list module "
  [tms name-coll]
  (let [name-key-coll (cc/as-sequential name-coll)
        tm-map (select-keys tms name-key-coll)]
    (cond
      (or (nil? tms)
          (empty? tms))
      (f/fail " Source is empty ")
      (f/failed? tms)
      tms
      (or (nil? tm-map)
          (empty? tm-map))
      (f/fail (format " %s Name not found" (str name-key-coll)))
      (is-reserve? tms name-key-coll)
      (vals tm-map)
      :else
      (f/try-> tm-map
               (validate-name! name-key-coll)
               (cc/select-values name-key-coll)
               (validate-model!)
               (filter-join-key)))))


(defn select-name-for-groups
  [tms gname name-coll]
  (let [name-set (into #{} (cc/as-sequential name-coll))
        p (if name-coll
            (comp (filter #(= (:dosql.core/group %) gname))
                  (filter #(contains? name-set (:dosql.core/name %))))
            (comp (filter #(= (:dosql.core/group %) gname))))
        t (into [] p (vals tms))
        w (sort-by :dosql.core/index t)]
    (into [] (map :dosql.core/name) w)))


(defn do-result1
  [m req-m]
  (condp = (:dosql.core/op req-m)
    :dosql.core/pull-sequence-one
    (-> m
        (assoc :dosql.core/model (:dosql.core/name m))
        (assoc :dosql.core/result #{:dosql.core/result-single :dosql.core/result-array})
        (assoc :dosql.core/dml :dosql.core/dml-select))
    m))


(defn init-db-seq-op
  [tm-coll req-m]
  (mapv (fn [m] (do-result1 m req-m)) tm-coll))



(defn select-name [context]
  (let [tms (:dosql.core/tms context)

        [_ name] (get-in context [:dosql.core/user-request :dosql.core/name])
        group (get-in context [:dosql.core/user-request :dosql.core/group])
        name (if group
               (select-name-for-groups tms group name)
               name)]
    (select-name-by-name-coll tms name)))
