(ns draconic.ui.binding
  {:doc "This namespace is primarily intended to make writing UI logic idomatic and generic for Clojure. Contrasting with frameworks like Reagent and Om, there is no underlying implimentation assumed by this ns at the abstraction level. Node initialization, lifecycle management, and markup is best done using the facilities of the frameworks themselves.

  The general design goal is for an application developer to be able to write UI bindings for the business domain in a generic and portable way, that can then be applied to each platform's spcific UIs using a unified and sensical interface.

  An additional design goal is to make the API for dealing with UI frameworks simpler and more consistant (both within frameworks and between them). UI elements are all functionally similar; a text box or button doesn't behave differently between platforms, so why can't we write code that treats them as the same things?

  "})

(defprotocol Atomic-Node
  "Base set of functions to be extended on a per-node basis, which lets the given node be used by the atom-like functions in draconic/ui."
  (-ui-get-state [this] "Gets the main data from this node, which will be different depending on the node in question. A TextField, for instance, gets a String, while an HBox gets a vector of its children. Used by fx.atomic/deref.")
  (-ui-set-state! [this newval] "The second arg is set as the node's new state.")
  )

(defprotocol Selector-Node
  "Participants in this protocol have a fixed list of several possible states. These should also extend Atomic-Node, which which the state can be got and set. Rendering, etc are functions of the target framework."
  (-ui-get-options [this] "Returns a vector of the current options.")
  (-ui-set-options! [this newopts] "Takes the node and a vector of new options."))


(defprotocol Container-Node
  "Participants in this protocol have nodes below them in the user-viewable scene graph (ie, while controlsfx's Rating is arguably a container full of buttons, it only counts as one node). For simple conatiners like VBox, using Atomic-Node's facilities might be sufficient, but as many containers have multiple compartments or sections it is desierable to have an abstraction that fits neatly over these more elaborate use cases. Additionally, JavaFX natively uses ObservableList to store multiple children. This class is nifty if you want to hang a listener off of the node's contents, but communicating with them in Clojure is clunky at best. Much better to communicate in terms of vectors, seperating the listener setting and removing into its own function set."
  (-ui-get-all-children [this] "Gets the children within the parent. Returns a map of vectors, with at least one :main and any number of other names denoting seperate areas within the node. A BorderPane, for example, has :top :bottom :left :right and :main, whilst a split pane has :side and :main and an HBox has just :main. Nodes with only one child (like a ScrollPane) generate maps looking like {:main [the-single-node]}. While a bit overmuch for these spcifically, a bit of verbosity in the name of a unified API seems worthwhile.")
  (-ui-set-children! [this node-map] "Sets the children within the parent. Takes a map of the same type that is returned with -ui-get-all-children.")
  (-ui-set-children-listener! [this content-section listener-fn] "Sets a change listener on the underlying node's observable list. Content-section is a keyword corrisponding to a location found in the given node's -ui-get-children map.")
  (-ui-remove-children-listener! [this content-section] "Removes all change listeners on a section's observable list.")
  )


(defprotocol Validating-Node
  "Nodes that participate in this protocol take a spec, which validates all data going into and out of the node."
  (-ui-get-state-spec [this] "Gets the validator spec set on the node. The spec will be a namespace-qualified keyword (unnamed specs aren't supported), and all data going into or out of the spec will be conformed to this spec.")
  (-ui-set-state-spec [this newspec] "Sets the validator spec set on the node. Newspec must be a namespace qualified keyword. All data going into or out of the spec will be conformed to the provided spec."))


