(ns com.edocu.help.cache.redis
  (:use com.edocu.help.cache.protocols)
  (:require [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [clojure.core.async :as async :refer [<! put! chan go close! thread]]
            [com.edocu.help.db.redis :as redis]
            [pandect.algo.sha1 :refer [sha1]]
            [com.edocu.help.sentry :as sentry])
  (:import [org.redisson.api LocalCachedMapOptions
                             RLocalCachedMap
                             LocalCachedMapOptions$EvictionPolicy
                             RFuture]
           [java.util.concurrent TimeUnit
                                 CompletableFuture]
           [java.util.function BiFunction]
           [io.netty.util.concurrent Future
                                     FutureListener]
           [java.util UUID]))

(defn- <update-cache!
  "Update data in cache and return result of update"
  [^RLocalCachedMap cache cache-key cache-delay-channel >result]
  (go
    (let [new-data (<! @cache-delay-channel)]
      (if (some? new-data)
        (thread
          (log/trace "update-cache")
          (try
            (.fastPut cache cache-key new-data)
            (catch Exception e
              (sentry/put-in-mdc {:cache-key cache-key
                                  :value     new-data})
              (log/error e "<update-cache!")))
          (put!
            >result
            new-data
            (fn [_] (close! >result))))
        (close! >result)))))

(defrecord Cache [redisson-client ^LocalCachedMapOptions options cache cache-name]
  component/Lifecycle
  (start [this]
    (log/trace "Starting Cache")
    (assoc this
      :cache
      (delay (try
               (.getLocalCachedMap
                 (redis/client redisson-client)
                 cache-name
                 options)
               (catch Exception e
                 (log/error e "creating cache redisson map"))))))
  (stop [this]
    (log/trace "Stopping Cache")
    (.destroy @cache)
    (dissoc this :cache))

  ICache
  (<from-cache [this cache-key cache-delay-channel]
    (println "AAAAAAA" @cache)
    (let [>result (chan)
          ^RFuture fut (.getAsync @cache cache-key)]
      (.addListener
        fut
        (proxy [FutureListener] []
          (operationComplete [^Future future]
            (try
              (if (.isSuccess future)
                (if-let [cached (.getNow future)]
                  (do
                    (log/trace "<from-cache")
                    (put!
                      >result
                      cached
                      (fn [_] (close! >result))))
                  (<update-cache!
                    @cache
                    cache-key
                    cache-delay-channel
                    >result))
                (<update-cache!
                  @cache
                  cache-key
                  cache-delay-channel
                  >result))
              (catch Exception e
                (sentry/put-in-mdc {:cache-key cache-key})
                (log/error e "<from-cache")
                (<update-cache!
                  @cache
                  cache-key
                  cache-delay-channel
                  >result))))))
      >result))

  (<invalidate-cache [this cache-key]
    (let [>result (chan)]
      (thread
        (try
          (.fastRemove @cache cache-key)
          (catch Exception e
            (sentry/put-in-mdc {:cache-key cache-key})
            (log/error e "<invalidate-cache"))
          (finally
            (close! >result))))
      >result)))

(defn ->cache
  "Return component of Cache"
  [^LocalCachedMapOptions options cache-name]
  (map->Cache {:options    options
               :cache-name cache-name}))