(ns com.edocu.communication.kafka.core
  (:use com.edocu.communication.protocols)
  (:require [taoensso.timbre :as timbre]
            [clj-kafka [zk :as kafka-zk]]
            [clj-kafka.new.producer :as kafka-producer]
            [clj-kafka.consumer.zk :as kafka-consumer]
            [clj-kafka.core :as kafka-core]
            [com.edocu.configuration.core :as config]
            [clojure.core.async :refer [>!!]]))

(defrecord Director [])
(def ->Director (memoize ->Director))

(def producer-promise (promise))
(defrecord Communicator [consumer_config])

;; Protocols implementation

(def communication-factory-impl
  {:create->Communicator (fn [_ consumer_config]
                           (when-not (realized? producer-promise)
                             (deliver
                               producer-promise
                               (kafka-producer/producer
                                 {"bootstrap.servers" (kafka-zk/broker-list
                                                        (kafka-zk/brokers
                                                          (config/kafka-zookeeper)))}
                                 (kafka-producer/byte-array-serializer)
                                 (kafka-producer/byte-array-serializer))))
                           (Communicator.
                             (zipmap
                               (map name (keys consumer_config))
                               (vals consumer_config))))})

(def message-management-impl
  {:route-message! (fn [_ ^String topic message]
                     (timbre/debug "route-message!:" "topic:" topic "message:" message)
                     (kafka-producer/send
                       @producer-promise
                       (kafka-producer/record
                         topic
                         (.getBytes (:topic message))
                         (message-body message))))

   :send-message!  (fn [communicator ^String topic message]
                     (timbre/debug "send-message!:" "topic:" topic "message:" message)
                     (route-message!
                       communicator
                       (config/cbr-topic)
                       (create->SendMessage
                         topic
                         message)))})

(def topic-management-impl
  {:register-topics!   (fn [communicator ^String topics]
                         (timbre/debug ":register-topic!" "topics:" topics)
                         (send-message!
                           communicator
                           (config/register-topic)
                           {:topic topics}))

   :subscribe-to-topic (fn [comminicator ^String topic callback_chan]
                         (future
                           (kafka-core/with-resource
                             [c (kafka-consumer/consumer
                                  (merge
                                    (config/kafka-zookeeper)
                                    (:consumer_config comminicator)))]
                             kafka-consumer/shutdown
                             (doseq [{:keys [key value]} (kafka-consumer/messages c topic)]
                               (>!! callback_chan
                                    (create->ReceiveMessage
                                      (String. key)
                                      value))))))})

(def error-management-impl
  {:report-malformed-message! (fn [communicator message]
                                (send-message!
                                  communicator
                                  (config/malformed-message-topic)
                                  message))

   :report-service-error!     (fn [communicator message]
                                (send-message!
                                  communicator
                                  (config/malformed-message-topic)
                                  message))})

;; mix implementations

(extend Director
  ICommunicationFactory
  communication-factory-impl)

(extend Communicator
  IMessageManagement
  message-management-impl

  ITopicManagement
  topic-management-impl

  IErrorsManagement
  error-management-impl)