(ns cljs.com.edocu.communication.kafka.core
  (:require
    [cljs.nodejs :as nodejs]
    [cljs.core.async :as async :refer [promise-chan put! <! close! chan]]
    [taoensso.timbre :refer-macros (debug error)]
    [com.edocu.communication.protocols :as prot]
    [com.edocu.communication.config :refer [default-partition-count default-group-id]]
    [cljs.com.edocu.communication.kafka.config :as config]
    [cljs.com.edocu.communication.kafka.edocu-config :as edocu-config])
  (:require-macros [cljs.core.async.macros :refer [go]]))

(defonce kafka (nodejs/require "kafka-node"))
(defonce kafka-producer (.-Producer kafka))
(defonce kafka-consumer-group (.-ConsumerGroup kafka))
(defonce kafka-client (.-Client kafka))
(defonce keyed-message (.-KeyedMessage kafka))

(defonce ^:private
         <producer (delay
                     (let [result (promise-chan)
                           client (kafka-client.
                                    (config/connection)
                                    "cljs-kafka")
                           producer (kafka-producer.
                                      client)]
                       (.on producer
                            "ready"
                            (fn []
                              (debug "Kafka producer ready")
                              (put! result producer)))
                       (.on producer
                            "error"
                            (fn [err]
                              (error "cannot create kafka producer. err:" err)
                              (close! result)))
                       result)))

(defn <kafka-producer
  "Return configured kafka producer"
  []
  @<producer)

(defrecord Communicator [stop_check consumer_config]
  prot/IMessageManagement
  (send-message! [_ topic message]
    (go
      (if-let [producer (<! (<kafka-producer))]
        (try
          (let [km (keyed-message.
                     (prot/->kafka-topic topic)
                     (.stringify
                       js/JSON
                       (clj->js message)))
                payloads [{:topic     (prot/->kafka-topic
                                        (edocu-config/cbr-topic))
                           :messages  [km]
                           :partition (rand-int default-partition-count)}]]
            (.send producer (clj->js payloads)
                   (fn [err data]
                     (if err
                       (error "send-message!."
                              "topic:" topic
                              "message:" message
                              "error:" err)
                       (debug "send-message!. data:" (pr-str data))))))
          (catch :default err
            (error "send-message!."
                   "topic:" topic
                   "message:" message
                   "error:" err)))
        (error "send-message!. Kafka producer is not ready"))))

  (deliver-message! [_ topic original_topic message]
    (go
      (if-let [producer (<! (<kafka-producer))]
        (try
          (let [km (keyed-message.
                     (prot/->kafka-topic original_topic)
                     (.stringify
                       js/JSON
                       (clj->js message)))
                payloads [{:topic     (prot/->kafka-topic
                                        topic)
                           :messages  [km]
                           :partition (rand-int default-partition-count)}]]
            (.send producer (clj->js payloads)
                   (fn [err data]
                     (if err
                       (error "deliver-message!."
                              "topic:" topic
                              "original_topic:" original_topic
                              "message:" message
                              "error:" err)
                       (debug "send-message!. data:" (pr-str data))))))
          (catch :default err
            (error "send-message!."
                   "topic:" topic
                   "message:" message
                   "error:" err)))
        (error "send-message!. Kafka producer is not ready"))))

  prot/ITopicManagement
  (register-topics! [this topics]
    (go
      (if-let [producer (<! (<kafka-producer))]
        (try
          (if (seq topics)
            (let [topics_str (clojure.string/join "," topics)]
              (prot/send-message!
                this
                (edocu-config/register-topic)
                {:topic topics_str})))
          (catch :default err
            (error "register-topics!"
                   "topics:" topics
                   "error:" err)))
        (error "register-topics!. Kafka producer is not ready"))))

  (subscribe-to-topic [this topic]
    (let [result (chan)
          kafka-topic (prot/->kafka-topic topic)
          options {:host                 (config/connection)
                   :groupId              (:group.id consumer_config)
                   :protocol             ["roundrobin"]
                   :fromOffset           "earliest"
                   :autoCommitIntervalMs 100}
          consumer (kafka-consumer-group.
                     (clj->js options)
                     kafka-topic)]
      (.on consumer
           "message"
           (fn [message]
             (let [{:keys [key] :as msg} (js->clj message :keywordize-keys true)]
               (if (some? msg)
                 (put! result (assoc
                                msg
                                :key (.toString key "utf-8")
                                :value (js->clj
                                         (.parse js/JSON (:value msg))
                                         :keywordize-keys true)))))))
      (.on consumer
           "error"
           (fn [err]
             (error "subscribe-to-topic. this:" (pr-str this)
                    "topic:" topic
                    "error:" err)
             (close! result)))
      (.on consumer
           "offsetOutOfRange"
           (fn [err]
             (error "subscribe-to-topic. this:" (pr-str this)
                    "topic:" topic
                    "error:" err)
             (close! result)))
      result)))
