(ns com.edocu.communication.protocols
  (:require [cheshire.core :as json]
            [clojure.core.async :refer [chan go go-loop <! >! buffer]]
            [taoensso.timbre :as timbre]))


(defprotocol IMessage
  (parsed-message-body [message] "Return parsed message")
  (message-body [message] "Return raw message body"))

(defrecord Message [^String topic body]
  Object
  (toString [this] (pr-str this))

  IMessage
  (parsed-message-body [_]
    (go
      (json/parse-cbor @body true)))

  (message-body [_]
    @body))

(defn create->SendMessage [^String topic body]
  (->Message
    topic
    (delay (json/generate-cbor body))))

(defn create->ReceiveMessage [^String topic body]
  (->Message
    topic
    (delay body)))

(defprotocol IMessageManagement
  (route-message! [this ^String topic ^IMessage message] "Send message to topic")
  (send-message! [this ^String topic message] "Send message to routing topic for routing it to destination topic"))

(defprotocol ITopicManagement
  (register-topics! [this ^String topics] "Register new topics in router")
  (subscribe-to-topic [this ^String topic callback_chan] "Return IEventStream subscribed to topic"))

(defprotocol IErrorsManagement
  (send-malformed-message-report! [this message] "Report malformed message to system topic")
  (send-service-error-report! [this message] "Report service error"))

(defprotocol ICommunicationFactory
  (create->Communicator [this consumer_config] "Return new commnicator which implement IMessageManagement ITopicManagement IErrorsManagement"))

(defrecord ElementType [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord Organization [_id]
  Object
      (toString [this] (pr-str this)))

(defrecord Action [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord User [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord MessageHeaders [^ElementType element_type
                           ^Organization organization
                           ^Action action
                           ^User user]
  Object
  (toString [this] (pr-str this)))

(defrecord EventMessage [^MessageHeaders headers
                         body]
  Object
  (toString [this] (pr-str this)))

(defrecord MalformedEventMessageReport [^String original_topic
                                        ^EventMessage original_message
                                        ^String reporting_service
                                        description]
  Object
  (toString [this] (pr-str this)))

(defrecord ServiceErrorMessageReport [^String service
                                      ^EventMessage original_message
                                      ^String reporting_service
                                      request
                                      respond]
  Object
  (toString [this] (pr-str this)))

(defn construct-message [topic body]
  (let [[_ element_type organization action user] (clojure.string/split topic #"\.")
        result (->EventMessage
                 (->MessageHeaders
                   (->ElementType element_type)
                   (->Organization organization)
                   (->Action action)
                   (->User user))
                 body)]
    (timbre/trace "construct-message:" result)
    result))