(ns com.edocu.communication.protocols
  (:require [clojure.core.async :refer [chan go go-loop <! >! buffer]]
            [taoensso.timbre :as timbre]))


(defrecord Message [^String topic body]
  Object
  (toString [this] (pr-str this)))


(defprotocol IMessageManagement
  (route-message! [this ^String topic ^IMessage message] "Send message to topic")
  (send-message! [this ^String topic message] "Send message to routing topic for routing it to destination topic"))

(defprotocol ITopicManagement
  (register-topics! [this ^String topics] "Register new topics in router")
  (subscribe-to-topic [this ^String topic callback_chan] "Return IEventStream subscribed to topic"))

(defprotocol IErrorsManagement
  (send-malformed-message-report! [this message] "Report malformed message to system topic")
  (send-service-error-report! [this message] "Report service error"))

(defprotocol ICommunicationFactory
  (create->Communicator [this consumer_config] "Return new commnicator which implement IMessageManagement ITopicManagement IErrorsManagement"))

(defrecord Type [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord Organization [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord Action [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord User [_id]
  Object
  (toString [this] (pr-str this)))

(defrecord MessageHeaders [^Type _type
                           ^Organization organization
                           ^Action action
                           ^User user]
  Object
  (toString [this] (pr-str this)))

(defrecord EventMessage [^String topic
                         ^MessageHeaders headers
                         body]
  Object
  (toString [this] (pr-str this)))

(defrecord MalformedEventMessageReport [^String original_topic
                                        ^EventMessage original_message
                                        ^String reporting_service
                                        description]
  Object
  (toString [this] (pr-str this)))

(defrecord ServiceErrorMessageReport [^String service
                                      ^EventMessage original_message
                                      ^String reporting_service
                                      request
                                      respond]
  Object
  (toString [this] (pr-str this)))

(defn
  ^{:doc "Return parsed topic to message headers"}
  default-topic-parser [topic]
  (let [[_ type organization action user] (re-seq
                                            #"([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_\-@\.]+)"
                                            topic)]
    (->MessageHeaders
      (->Type type)
      (->Organization organization)
      (->Action action)
      (->User user))))