(ns com.edocu.communication.core
  (:use com.edocu.communication.protocols)
  (:require [com.edocu.communication.kafka.core :as kafka]
            [com.edocu.communication.zeromq.core :as zeromq]
            [com.edocu.configuration.core :as edocu-config]
            [taoensso.timbre :as timbre]
            [clojure.core.async :refer [chan go go-loop <! >!]]
            [environ.core :refer [env]]
            [clojure.core.memoize :as memo]))

(defrecord Director [])

;TODO Tempolary send message to ZeroMQ and Kafka. Later only Kafka
(defrecord Communicator [stop_check zmq kf]
  IMessageManagement

  (send-message! [_ topic message]
    (zeromq/send-message!
      zmq
      (->zeromq-topic topic)
      message)
    (send-message!
      kf
      (->kafka-topic topic)
      message))

  ITopicManagement
  (register-topics! [_ topics]
    (zeromq/register-topic!
      zmq
      (->zeromq-topic topics))
    (register-topics!
      kf
      (->kafka-topic topics)))

  (subscribe-to-topic [_ topic callback_chan]
    (subscribe-to-topic
      kf
      (->kafka-topic topic)
      callback_chan))

  IErrorsManagement
  (send-malformed-message-report! [_ message]
    (zeromq/report-malformed-message!
      zmq
      message)
    (send-malformed-message-report!
      kf
      message))

  (send-service-error-report! [_ message]
    (zeromq/report-service-error!
      zmq
      message)
    (send-malformed-message-report!
      kf
      message)))

(extend Director
  ICommunicationFactory
  {:create->Communicator (fn [_ consumer_config]
                           (let [stop_check (memo/ttl
                                              (fn []
                                                (= "true" (env :stop)))
                                              :ttl/threshold 15000)]
                             (->Communicator
                               stop_check
                               (zeromq/->Communication
                                 (edocu-config/zeromq-router-address)
                                 (edocu-config/zeromq-publisher-address))
                               (kafka/->Communicator
                                 stop_check
                                 consumer_config))))})

(defn process-message-on
  ([group_id topic callback]
    (process-message-on
      default-topic-parser
      ->EventMessage
      group_id topic callback))
  ([topic-parser construct-message group_id topic callback]
   (let [comm (create->Communicator
                (->Director)
                {:group.id group_id})
         c (chan 1024)]
     (register-topics! comm topic)
     (go-loop [msg (<! c)]
       (when msg
         (timbre/debug "process-message-on:" "topic:" topic "group_id:" group_id "received message:" msg)
         (try
           (let [msg_topic (:key msg)] (callback (construct-message msg_topic (topic-parser msg_topic) (:value msg))))
           (catch Exception e
             (timbre/error "process-message-on:" "topic:" topic "group_id:" group_id "received message:" msg "error:" e)))
         (if-not ((:stop_check comm))
           (recur (<! c)))))
     (subscribe-to-topic comm
                         topic
                         c)
     comm)))

(defn report-service-error! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "service-error-report"})]
    (send-service-error-report! comm report)))

(defn report-malformed-message! [report]
  (let [comm (create->Communicator
               (->Director)
               {:group.id "malformed-message-report"})]
    (send-malformed-message-report! comm report)))
