(ns com.edocu.communication.kafka.core
  (:use com.edocu.communication.protocols)
  (:require [taoensso.timbre :as timbre]
            [com.edocu.configuration.core :as edocu-config]
            [com.edocu.communication.kafka.config :as config]
            [clojure.core.async :refer [>! go]]
            [franzy.serialization.serializers :as serializers]
            [franzy.serialization.deserializers :as deserializers]
            [franzy.serialization.json.serializers :as json-serializers]
            [franzy.serialization.json.deserializers :as json-deserializers]
            [franzy.clients.producer.client :as producer]
            [franzy.clients.consumer.client :as consumer]
            [franzy.clients.producer.protocols :refer :all]
            [franzy.clients.consumer.protocols :refer :all]
            [franzy.clients.producer.types :as pt]
            [franzy.clients.consumer.callbacks :as callbacks]
            [franzy.clients.consumer.defaults :as cd]
            [franzy.admin.topics :as kafka-topics]))

(defn default-key-serializer []
  (serializers/string-serializer))

(defn default-value-serializer []
  (json-serializers/json-serializer))

(defn default-key-deserializer []
  (deserializers/string-deserializer))

(defn default-value-deserializer []
  (json-deserializers/json-deserializer {:key-fn true}))

(def producer (delay (try
                       (let [pc {:bootstrap.servers (edocu-config/kafka-brokers)}]
                         (producer/make-producer
                           pc
                           (default-key-serializer)
                           (default-value-serializer)))
                       (catch Exception e
                         (timbre/error "Cannot create kafka producer. error:" e)
                         nil))))

(defrecord Communicator [stop_check consumer_config])

(def message-management-impl
  {:route-message! (fn [_ topic message]
                     (timbre/debug "route-message!:" "topic:" topic "message:" message)
                     (let [producer_record (pt/->ProducerRecord
                                             (->kafka-topic topic)
                                             (rand-int config/default-partition-count)
                                             (:topic message)
                                             (:body message))]
                       (if @producer
                         (send-async! @producer producer_record))))

   :send-message!  (fn [communicator topic message]
                     (timbre/debug "send-message!:" "topic:" topic "message:" message)
                     (route-message!
                       communicator
                       (edocu-config/cbr-topic)
                       (->Message
                         topic
                         message)))})

(def topic-management-impl
  {:register-topics!   (fn [communicator topics]
                         (timbre/debug ":register-topic!" "topics:" topics)
                         (send-message!
                           communicator
                           (edocu-config/register-topic)
                           {:topic topics}))

   :subscribe-to-topic (fn [comminicator topic callback_chan]
                         (go
                           (let [kafka_topic (->kafka-topic topic)
                                 cc {:bootstrap.servers       (edocu-config/kafka-brokers)
                                     :group.id                (or
                                                                (get-in comminicator [:consumer_config :group.id])
                                                                config/default-group-id)
                                     :auto.offset.reset       :earliest
                                     :enable.auto.commit      true
                                     :auto.commit.interval.ms 100}
                                 rebalance_listener (callbacks/consumer-rebalance-listener (fn [_])
                                                                                           (fn [_]))
                                 options (cd/make-default-consumer-options {:rebalance-listener-callback rebalance_listener})
                                 zk_utils (config/make-zk-utils)]
                             (loop [topic_exists? (kafka-topics/topic-exists? zk_utils kafka_topic)
                                    test_count 0]
                               (when (or (not topic_exists?)
                                         (> 6 test_count))
                                 (Thread/sleep 500)
                                 (recur (kafka-topics/topic-exists? zk_utils kafka_topic)
                                        (inc test_count))))
                             (if (kafka-topics/topic-exists? zk_utils kafka_topic)
                               (with-open [c (consumer/make-consumer
                                               cc
                                               (default-key-deserializer)
                                               (default-value-deserializer)
                                               options)]
                                 (timbre/trace "subscribe-to-topic. topic:" kafka_topic)
                                 (subscribe-to-partitions! c [kafka_topic])
                                 (loop [cr (poll! c)]
                                   (doseq [r cr]
                                     (if-not (nil? r) (>! callback_chan r)))
                                   (if-not ((:stop_check comminicator))
                                     (recur (poll! c))
                                     (timbre/trace "Stop subcription on topic:" topic))))
                               (timbre/error "cannot subcribe on topic:" kafka_topic)))))})

(def error-management-impl
  {:send-malformed-message-report! (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (edocu-config/malformed-message-topic)
                                       message))

   :send-service-error-report!     (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (edocu-config/service-error-topic)
                                       message))})

(extend Communicator
  IMessageManagement
  message-management-impl

  ITopicManagement
  topic-management-impl

  IErrorsManagement
  error-management-impl)