(ns com.edocu.communication.protocols
  (:require
    #?(:clj [clojure.spec :as s]
       :cljs [cljs.spec :as s])
    #?(:clj
            [clojure.spec.gen :as gen]
       :cljs [cljs.spec.impl.gen :as gen])))

(def ^:const KAFKA_WILDCARD "_ANY_")
(def ^:const AT "@")
(def ^:const KAFKA_AT "_AT_")
(def ^:const EDOCU_WILDCARD "*")

;---------------------- Protocols ---------------------------

(defprotocol IMessageManagement
  (send-message! [this topic message] "Send message to routing topic for routing it to destination topic"))

(defprotocol ITopicManagement
  (register-topics! [this topics] "Register new topics in router")
  (subscribe-to-topic [this topic] "Return IEventStream subscribed to topic"))

(defprotocol IErrorsManagement
  (send-malformed-message-report! [this message] "Report malformed message to system topic")
  (send-service-error-report! [this message] "Report service error"))

(defprotocol ICommunicationFactory
  (create->Communicator [this consumer_config] "Return new commnicator which implement IMessageManagement ITopicManagement IErrorsManagement"))

;---------------------- Topic ---------------------------

(defn- maybe-replace-random-section-with-edocu-wildcard [topic_sections]
  (if (> (rand-int 10) 5)
    topic_sections
    (map (fn [section]
           (if (> (rand-int 10) 6)
             EDOCU_WILDCARD
             section))
         topic_sections)))

(defn- ->normal-topic [topic_sections]
  (clojure.string/join "." (map (fn [section]
                                  (if (> (count section) 45)
                                    (subs section 0 45)
                                    section))
                                topic_sections)))

(defn- ->invalid-topic [topic_sections]
  (->normal-topic
    (if (> (rand-int 10) 5)
      (rest topic_sections)
      (reduce (fn [tmp section]
                (conj tmp
                      (if (> (rand-int 10) 5)
                        (str "+/" section "@\\")
                        section)))
              []
              topic_sections))))

(defn- maybe-create-user-with-email [topic_sections]
  (if (> (rand-int 10) 5)
    (conj (pop (vec topic_sections)) "user@user.com")
    topic_sections))

(defn- topic-section []
  (gen/such-that #(not= % "")
                 (gen/string-alphanumeric)))

(def service-topic-section topic-section)
(def type-topic-section topic-section)
(def organization-topic-section topic-section)
(def action-topic-section topic-section)
(def user-topic-section topic-section)

(def topic-gen #(gen/fmap (fn [topic_sections]
                            (let [final_topic_sections (-> topic_sections
                                                           maybe-create-user-with-email)]
                              (if (> (rand-int 10) 5)
                                (->normal-topic final_topic_sections)
                                (->invalid-topic final_topic_sections))))
                          (gen/tuple
                            (service-topic-section)
                            (type-topic-section)
                            (organization-topic-section)
                            (action-topic-section)
                            (user-topic-section))))

(s/def ::topic (s/with-gen
                 (s/and string?
                        #(<= (count %) 249)
                        #(re-matches #"([a-zA-Z0-9\._\-@])+" %))
                 topic-gen))

(defn default-topic-split [topic]
  (let [sections (rest (re-find
                         #"([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_-]+)\.([a-zA-Z0-9_\-@\.]+)"
                         topic))]
    (if (seq sections)
      (vec sections))))

(s/fdef default-topic-split
        :args (s/cat :topic ::topic)
        :ret (s/or :invalid-topic nil?
                   :result (s/coll-of string?
                                      :kind vector?
                                      :count 5)))

(def edocu-topic-gen #(gen/fmap (fn [topic_sections]
                                  (let [final_topic_sections (-> topic_sections
                                                                 maybe-replace-random-section-with-edocu-wildcard
                                                                 maybe-create-user-with-email)]
                                    (->normal-topic final_topic_sections)))
                                (gen/tuple
                                  (service-topic-section)
                                  (type-topic-section)
                                  (organization-topic-section)
                                  (action-topic-section)
                                  (user-topic-section))))

(s/def ::edocu-topic (s/with-gen
                       (s/and string?
                              #(<= (count %) 249)
                              #(not (clojure.string/includes? % KAFKA_WILDCARD)))
                       edocu-topic-gen))

(def kafka-topic-gen #(gen/fmap (fn [topic_sections]
                                  (->normal-topic topic_sections))
                                (gen/tuple
                                  (service-topic-section)
                                  (type-topic-section)
                                  (organization-topic-section)
                                  (action-topic-section)
                                  (user-topic-section))))

(s/def ::kafka-topic (s/with-gen
                       (s/and string?
                              #(<= (count %) 249)
                              #(re-matches #"([a-zA-Z0-9\\._\\-])+" %))
                       kafka-topic-gen))

(defn ->kafka-topic [topic]
  (-> topic
      (clojure.string/replace EDOCU_WILDCARD KAFKA_WILDCARD)
      (clojure.string/replace AT KAFKA_AT)))

(s/fdef ->kafka-topic
        :args (s/cat :topic ::edocu-topic)
        :ret ::kafka-topic)

;---------------------- Records ---------------------------


(defrecord Message [topic body]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::body map?)

(s/def ::Message (s/keys :req-un [::topic ::body]))

(s/def ::_id string?)

(defrecord Type [_id]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::_type (s/keys :req-un [::_id]))

(defrecord Organization [_id]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::organization (s/keys :req-un [::_id]))

(defrecord Action [_id]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::action (s/keys :req-un [::_id]))

(defrecord User [_id]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::user (s/keys :req-un [::_id]))

(defrecord MessageHeaders [_type
                           organization
                           action
                           user]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::headers (s/keys :req-un [::_type
                                  ::organization
                                  ::action
                                  ::user]))

(defrecord EventMessage [topic
                         headers
                         body]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(s/def ::event-message (s/keys :req-un [::topic
                                        ::headers]))

(defrecord MalformedEventMessageReport [original_topic
                                        original_message
                                        reporting_service
                                        description]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(defrecord ServiceErrorMessageReport [service
                                      original_message
                                      reporting_service
                                      request
                                      respond]
  #?(:clj Object)
  #?(:clj (toString [this] (pr-str this))))

(defn- topic->MessageHeaders [_ type organization action user]
  (->MessageHeaders
    (->Type type)
    (->Organization organization)
    (->Action action)
    (->User user)))

(s/fdef topic->MessageHeaders
        :args (s/cat :service string?
                     :type string?
                     :organization string?
                     :action string?
                     :user string?)
        :ret ::headers
        :fn (s/and #(= (->Type (-> % :args :type))
                       (-> % :ret :_type))
                   #(= (->Organization (-> % :args :organization))
                       (-> % :ret :organization))
                   #(= (->Action (-> % :args :action))
                       (-> % :ret :action))
                   #(= (->User (-> % :args :user))
                       (-> % :ret :user))))

;------------------------ API ----------------------------

(defn default-topic-parser
  "Return parsed topic to message headers"
  [topic]
  (if-let [correct_topic (default-topic-split topic)]
    (apply topic->MessageHeaders correct_topic)))

(s/fdef default-topic-parser
        :args (s/cat :topic ::topic)
        :ret (s/or :invalid-topic nil?
                   :result ::headers))