(ns clj.com.edocu.communication.kafka.core
  (:use com.edocu.communication.protocols)
  (:require [clojure.tools.logging :as log]
            [clojure.spec.alpha :as s]
            [clojure.spec.test.alpha :as stest]
            [clojure.spec.gen.alpha :as gen]
            [com.edocu.communication.protocols :as prot]
            [com.edocu.configuration.core :as edocu-config]
            [clj.com.edocu.communication.kafka.config :as config]
            [clojure.core.async :refer [>! go chan close!]]
            [franzy.serialization.serializers :as serializers]
            [franzy.serialization.deserializers :as deserializers]
            [franzy.serialization.json.serializers :as json-serializers]
            [franzy.serialization.json.deserializers :as json-deserializers]
            [franzy.clients.producer.client :as producer]
            [franzy.clients.consumer.client :as consumer]
            [franzy.clients.producer.protocols :as producer-prot :refer :all]
            [franzy.clients.consumer.protocols :refer :all]
            [franzy.clients.producer.types :as pt]
            [franzy.clients.consumer.callbacks :as callbacks]
            [franzy.clients.consumer.defaults :as cd]
            [franzy.admin.topics :as kafka-topics]
            [com.edocu.help.sentry :as sentry]
            [com.edocu.communication.config :refer [default-partition-count default-group-id]]))

(defn default-key-serializer []
  (serializers/string-serializer))

(defn default-value-serializer []
  (json-serializers/json-serializer))

(defn default-key-deserializer []
  (deserializers/string-deserializer))

(defn default-value-deserializer []
  (json-deserializers/json-deserializer {:key-fn true}))

(def producer (delay (let [pc {:bootstrap.servers (edocu-config/kafka-brokers)}]
                       (try
                         (producer/make-producer
                           pc
                           (default-key-serializer)
                           (default-value-serializer))
                         (catch Exception e
                           (sentry/put-in-mdc {:pc pc})
                           (log/error e "Cannot create kafka producer.")
                           nil)))))

;Mocking: generated Producer return `m` for send-async!
(s/def ::producer (s/with-gen
                    #(satisfies? producer-prot/FranzyProducer %)
                    #(s/gen #{(reify producer-prot/FranzyProducer
                                (send-async! [_ m]
                                  m))})))

(defn ->kafka-producer []
  @producer)

(s/fdef ->kafka-producer
        :args (s/cat)
        :ret ::producer)

#_(stest/instrument `->kafka-producer {:stub #{`->kafka-producer}})

(defrecord Communicator [stop_check consumer_config])
(s/def ::stop_check (s/with-gen
                      fn?
                      #(s/gen #{(fn [] false) (fn [] true)})))
(s/def ::group.id (s/and
                    string?
                    #(not-empty %)))
(s/def ::consumer_config (s/keys :req-un [::group.id]))
(s/def ::Communicator (s/with-gen
                        (s/keys :req-un [::stop_check ::consumer_config])
                        #(s/gen (set (map (fn [_]
                                            (->Communicator
                                              (-> (s/gen ::stop_check)
                                                  gen/sample
                                                  first)
                                              {:group.id (-> (s/gen ::group.id)
                                                             gen/sample
                                                             first)}))
                                          (range 10))))))

;------------------------ Implementation -----------------------------


(defn impl-send-message! [_ topic message]
  (log/debug "send-message!:" "topic:" topic "message:" message)
  (let [producer_record (pt/->ProducerRecord
                          (->kafka-topic (edocu-config/cbr-topic))
                          (rand-int default-partition-count)
                          topic
                          message)]
    (if (->kafka-producer)
      (try
        (send-async! (->kafka-producer) producer_record)
        producer_record
        (catch Exception e
          (sentry/put-in-mdc {:topic   topic
                              :message message})
          (log/error e "send-message!.")
          :error)))))

(s/fdef impl-send-message!
        :args (s/cat :communicator ::Communicator
                     :topic ::prot/edocu-topic
                     :message map?)
        :ret (s/or :send map?
                   :error (s/and keyword?
                                 #(= :error %)))
        :fn (s/and #(= (->kafka-topic (edocu-config/cbr-topic)) (-> % :ret second :topic))
                   #(int? (-> % :ret second :partition))
                   #(= (-> % :ret second :key) (-> % :args :topic))
                   #(= (-> % :ret second :value) (-> % :args :message))))

(defn impl-deliver-message! [_ topic original_topic message]
  (log/debug "send-message!:" "topic:" topic "message:" message)
  (let [producer_record (pt/->ProducerRecord
                          (->kafka-topic topic)
                          (rand-int default-partition-count)
                          original_topic
                          message)]
    (if (->kafka-producer)
      (try
        (send-async! (->kafka-producer) producer_record)
        producer_record
        (catch Exception e
          (sentry/put-in-mdc {:topic   topic
                              :message message})
          (log/error e "send-message!.")
          :error)))))

(s/fdef impl-deliver-message!
        :args (s/cat :communicator ::Communicator
                     :topic ::prot/edocu-topic
                     :original_topic ::prot/edocu-topic
                     :message map?)
        :ret (s/or :send map?
                   :error (s/and keyword?
                                 #(= :error %))))

(defn impl-register-topics! [communicator topics]
  (log/debug ":register-topic!" "topics:" topics)
  (if (seq topics)
    (let [topics_str (clojure.string/join "," topics)]
      (send-message!
        communicator
        (edocu-config/register-topic)
        {:topic topics_str}))))

(s/fdef impl-register-topics!
        :args (s/cat :communicator ::Communicator
                     :topics (s/coll-of ::prot/edocu-topic))
        :ret (s/or :send map?
                   :error (s/and keyword?
                                 #(= :error %)))
        :fn (s/and #(= (->kafka-topic (edocu-config/cbr-topic)) (-> % :ret second :topic))
                   #(int? (-> % :ret second :partition))
                   #(= (-> % :ret second :key) (edocu-config/register-topic))
                   #(= (-> % :ret second :value) {:topic (clojure.string/join "," (-> % :args :topics))})))

(defmacro if-topic-exists? [zk_utils kafka_topic & body]
  `(do
     (loop [topic_exists?# (kafka-topics/topic-exists? ~zk_utils ~kafka_topic)
            test_count# 0]
       (when (or (not topic_exists?#)
                 (> 6 test_count#))
         (Thread/sleep 500)
         (recur (kafka-topics/topic-exists? ~zk_utils ~kafka_topic)
                (inc test_count#))))
     (if (kafka-topics/topic-exists? ~zk_utils ~kafka_topic)
       ~@body
       (do
         (sentry/put-in-mdc {:kafka_topic ~kafka_topic})
         (log/error "cannot subcribe on topic.")))))

(defn impl-subscribe-to-topic [comminicator topic]
  (let [callback_chan (chan 1024)]
    (go
      (let [kafka_topic (->kafka-topic topic)
            cc {:bootstrap.servers       (edocu-config/kafka-brokers)
                :group.id                (or
                                           (get-in comminicator [:consumer_config :group.id])
                                           default-group-id)
                :auto.offset.reset       :earliest
                :enable.auto.commit      true
                :auto.commit.interval.ms 100}
            rebalance_listener (callbacks/consumer-rebalance-listener (fn [_])
                                                                      (fn [_]))
            options (cd/make-default-consumer-options {:rebalance-listener-callback rebalance_listener})
            zk_utils (config/make-zk-utils)]
        (try
          (if-topic-exists? zk_utils kafka_topic
                            (with-open [c (consumer/make-consumer
                                            cc
                                            (default-key-deserializer)
                                            (default-value-deserializer)
                                            options)]
                              (log/debug "subscribe-to-topic. topic:" kafka_topic)
                              (subscribe-to-partitions! c [kafka_topic])
                              (loop [cr (poll! c)]
                                (doseq [r cr]
                                  (if-not (nil? r) (>! callback_chan r)))
                                (if-not ((:stop_check comminicator))
                                  (recur (poll! c))
                                  (log/debug "Stop subcription on topic:" topic)))))
          (catch Exception e
            (sentry/put-in-mdc {:topic       topic
                                :kafka_topic kafka_topic
                                :cc          cc})
            (log/error e "subscribe-to-topic.")))))
    callback_chan))

(s/fdef impl-subscribe-to-topic
        :args (s/cat :communicator ::Communicator
                     :topic ::prot/edocu-topic)
        :ret #(satisfies? clojure.core.async.impl.protocols/ReadPort %))

(def message-management-impl
  {:send-message! #'impl-send-message!
   :deliver-message! #'impl-deliver-message!})

(def topic-management-impl
  {:register-topics!   #'impl-register-topics!
   :subscribe-to-topic #'impl-subscribe-to-topic})

(def error-management-impl
  {:send-malformed-message-report! (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (edocu-config/malformed-message-topic)
                                       message))

   :send-service-error-report!     (fn [communicator message]
                                     (send-message!
                                       communicator
                                       (edocu-config/service-error-topic)
                                       message))})

(extend Communicator
  IMessageManagement
  message-management-impl

  ITopicManagement
  topic-management-impl

  IErrorsManagement
  error-management-impl)