(ns com.edocu.cache.core
  (:require [clojure.core.cache :as core-cache]
            [selmeci.clojure.lang.redis-persistent-map :as redis-map]
            [clojure.core.async :as async :refer [go chan go-loop >! <!]]))

(defn hit-command [cache_ref cache_key]
  (fn []
    (dosync
      (commute
        cache_ref
        core-cache/hit
        cache_key))))

(defn miss-command [cache_ref cache_key new_respond]
  (fn []
    (dosync
      (commute
        cache_ref
        core-cache/miss
        cache_key 
        new_respond))))

(defn through-command [cache_ref cache_key new_value]
  (fn []
    (dosync
      (commute
        cache_ref
        (fn [cache]
          (core-cache/through (fn [& _] new_value)
                              cache
                              cache_key))))))

(defn ^{:doc "Retur ref with LRU cache in Redis Map"} create->lru-cache [redis_cfg lock_key prefix threshold]
  (ref (let [cache_map (redis-map/create->RedisPersistentMap 
                         redis_cfg 
                         lock_key
                         prefix)]
         (core-cache/lru-cache-factory
                      cache_map
                      :threshold threshold))))

(defn run-cache-changer [cache_manager_chan]
  (go-loop [cmd (<! cache_manager_chan)]
           (if-not (nil? cmd)
             (do
               (cmd )
               (recur (<! cache_manager_chan))))))

(defn with-cache 
  ([cache_ref cache_manager_chan command]
    (go (>! cache_manager_chan command)))
  ([cache_ref cache_manager_chan cache_key update_fn]
    (if-let [respond (core-cache/lookup @cache_ref cache_key)]
      (do
        (go (>! cache_manager_chan (hit-command cache_ref cache_key)))
        respond)
      (do
        (let [new_value (update_fn )]
          (go (>! cache_manager_chan (miss-command cache_ref cache_key new_value)))
          new_value)))))



