(ns com.edocu.configuration.core
  (:require [environ.core :as e]
            [com.edocu.configuration.etcd :as etcd]
            [taoensso.timbre :as timbre]
            [clojurewerkz.urly.core :as urly]))

(def ^:const ^String BASE_URL_PATH "base-url")
(def ^:const ^String EMAIL_PREFIX "email/")
(def ^:const ^String SMTP_PATH (str EMAIL_PREFIX "smtp"))
(def ^:const ^String NOTIFICATION_EMAIL_PATH (str EMAIL_PREFIX "addresses/notifications"))

(def ^:const BASE_URL_DEFAULT "https://edocu.service.dev.edocu.local")
(def ^:const SMTP_DEFAULT "")
(def ^:const NOTIFICATION_EMAIL_DEFAULT "notifications@edocu.eu")

;############################ Base URL ###############################

(def base-url-promise (promise ))

(defn base-url []
  (if-not (realized? base-url-promise)
    (do
      (deliver base-url-promise
               (etcd/get-config-for
                 BASE_URL_PATH
                 BASE_URL_DEFAULT))
      (timbre/trace "base-url" @base-url-promise)))
  @base-url-promise)

;######################################################################

;############################## SMTP ##################################

(def smtp-promise (promise ))

(defn- parse-user-authentification [user_info]
  (if user_info
    (let [[user pass] (clojure.string/split user_info #":")
          result {}
          result (if user (assoc result :user user) result)
          result (if pass (assoc result :pass pass) result)]
      result)
    {}))

(defn- parse-protocol [protocol]
  (case protocol
    "smtps" {:ssl :y}
    "smtpd" {:tls :y}
    {}))

(defn- parse-smtp-config [smtp_url]
  (let [urly_map (urly/as-map (urly/url-like smtp_url))]
    (merge urly_map
           (parse-user-authentification (:user-info urly_map))
           (parse-protocol (:protocol urly_map)))))

(defn smtp []
  (if-not (realized? smtp-promise)
    (do
      (deliver smtp-promise
               (parse-smtp-config
                 (etcd/get-config-for
                   SMTP_PATH
                   SMTP_DEFAULT)))
      (timbre/trace "smtp" @smtp-promise)))
  @smtp-promise)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-promise (promise ))

(defn notifications-email []
  (if-not (realized? notifications-email-promise)
    (do
      (deliver notifications-email-promise
               (etcd/get-config-for
                 NOTIFICATION_EMAIL_PATH
                 NOTIFICATION_EMAIL_DEFAULT))
      (timbre/trace "notifications-email" @notifications-email-promise)))
  @notifications-email-promise)

;######################################################################
