(ns com.edocu.configuration.core
  (:require [com.edocu.configuration.etcd :as etcd]
            [taoensso.timbre :as timbre]
            [clojurewerkz.urly.core :as urly]
            [cheshire.core :as json]))

(def ^:const ^String BASE_URL_PATH "base-url")
(def ^:const ^String EMAIL_PREFIX "email/")
(def ^:const ^String SMTP_PATH (str EMAIL_PREFIX "smtp"))
(def ^:const ^String NOTIFICATION_EMAIL_PATH (str EMAIL_PREFIX "addresses/notifications"))
(def ^:const ^String KAFKA_PATH "kafka/")
(def ^:const ^String KAFKA_BROKERS_PATH (str KAFKA_PATH "brokers"))
(def ^:const ^String KAFKA_CONFIG_PATH (str KAFKA_PATH "config"))
(def ^:const ^String REDIS_CONFIG_PATH "redis/config")

(def ^:const ^String REDIS_HOST "redis.db.dev.edocu.local")
(def ^:const ^String BASE_URL_DEFAULT "https://edocu.service.dev.edocu.local")
(def ^:const ^String SMTP_DEFAULT "")
(def ^:const ^String NOTIFICATION_EMAIL_DEFAULT "notifications@edocu.eu")
(def ^:const KAFKA_BROKER [{:host "kafka.mq.dev.edocu.local" :port 9092}])
(def ^:const KAFKA_BROKERS_DEFAULT (json/generate-string
                                     KAFKA_BROKER))
(def ^:const KAFKA_CONFIG_DEFAULT (json/generate-string
                                    {:redis-conf        {:host       REDIS_HOST
                                                         :max-active 5
                                                         :timeout    1000
                                                         :group-name "kafka"}
                                     :bootstrap-brokers KAFKA_BROKER}))
(def ^:const REDIS_CONFIG_DEFAULT (json/generate-string
                                    {:pool {}
                                     :spec {:host REDIS_HOST
                                            :port 6379
                                            :db   10}}))

;############################ Base URL ###############################

(def base-url-promise (promise))

(defn base-url []
  (if-not (realized? base-url-promise)
    (do
      (deliver base-url-promise
               (etcd/get-config-for
                 BASE_URL_PATH
                 BASE_URL_DEFAULT))
      (timbre/trace "base-url" @base-url-promise)))
  @base-url-promise)

;######################################################################

;############################## SMTP ##################################

(def smtp-promise (promise))

(defn- parse-user-authentification [user_info]
  (if user_info
    (let [[user pass] (clojure.string/split user_info #":")
          result {}
          result (if user (assoc result :user user) result)
          result (if pass (assoc result :pass pass) result)]
      result)
    {}))

(defn- parse-protocol [protocol]
  (case protocol
    "smtps" {:ssl :y}
    "smtpd" {:tls :y}
    {}))

(defn- parse-smtp-config [smtp_url]
  (let [urly_map (urly/as-map (urly/url-like smtp_url))]
    (merge urly_map
           (parse-user-authentification (:user-info urly_map))
           (parse-protocol (:protocol urly_map)))))

(defn smtp []
  (if-not (realized? smtp-promise)
    (do
      (deliver smtp-promise
               (parse-smtp-config
                 (etcd/get-config-for
                   SMTP_PATH
                   SMTP_DEFAULT)))
      (timbre/trace "smtp" @smtp-promise)))
  @smtp-promise)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-promise (promise))

(defn notifications-email []
  (if-not (realized? notifications-email-promise)
    (do
      (deliver notifications-email-promise
               (etcd/get-config-for
                 NOTIFICATION_EMAIL_PATH
                 NOTIFICATION_EMAIL_DEFAULT))
      (timbre/trace "notifications-email" @notifications-email-promise)))
  @notifications-email-promise)

;######################################################################

;######################## Kafka Brokers ###############################

(def kafka-brokers-promise (promise))

(defn kafka-brokers []
  (when-not (realized? kafka-brokers-promise)
    (deliver kafka-brokers-promise
             (json/parse-string
               (etcd/get-config-for
                 KAFKA_BROKERS_PATH
                 KAFKA_BROKERS_DEFAULT)
               true))
    (timbre/trace "kafka-brokers" @kafka-brokers-promise))
  @kafka-brokers-promise)

;######################################################################

;######################### Kafka Config ###############################

(def kafka-config-promise (promise))

(defn kafka-config [group_name]
  (when-not (realized? kafka-config-promise)
    (deliver kafka-config-promise
             (assoc-in (json/parse-string
                         (etcd/get-config-for
                           KAFKA_CONFIG_PATH
                           KAFKA_CONFIG_DEFAULT)
                         true)
                       [:redis-conf :group-name]
                       group_name))
    (timbre/trace "kafka-config" @kafka-config-promise))
  @kafka-config-promise)

;######################################################################

;######################### Redis Config ###############################

(def redis-config-promise (promise))

(defn redis-config []
  (when-not (realized? redis-config-promise)
    (deliver redis-config-promise
             (json/parse-string
               (etcd/get-config-for
                 REDIS_CONFIG_PATH
                 REDIS_CONFIG_DEFAULT)
               true))
    (timbre/trace "redis-config-promise" @redis-config-promise))
  @redis-config-promise)

;######################################################################