(ns com.edocu.configuration.persistence.mongodb
  (:require [com.edocu.configuration.core :refer [mongodb-replicas-uri mongodb-db env]]
            [clojure.tools.logging :as log]
            [environ.core :as e])
  (:import [com.mongodb ReadPreference WriteConcern ConnectionString]
           [com.mongodb.async.client MongoClients]
           [org.bson Document]
           [clojure.lang IPersistentMap Named Keyword Ratio]
           [java.util List ArrayList Date Set]))

(defn- build-url-address []
  (clojure.string/join
    ","
    (map
      (fn [{:keys [uri port]}]
        (format "%s:%d" uri port))
      (mongodb-replicas-uri))))

(defn replicas-uri []
  (ConnectionString.
    (if (and (e/env :mongodb-username)
             (e/env :mongodb-password))
      (format "mongodb://%s:%s@%s/?replicaSet=eDocu&authSource=%s&w=2"
              (e/env :mongodb-username)
              (e/env :mongodb-password)
              (build-url-address)
              (mongodb-db))
      (format "mongodb://%s/?replicaSet=eDocu&w=2" (build-url-address)))))

(def ^:private db-delay (delay (let [client (MongoClients/create (replicas-uri))
                                     database (.getDatabase client (mongodb-db))]
                                 database)))

(defn db []
  @db-delay)

(defprotocol ConvertFromDocument
  (document->hashmap [input] "Converts given Document instance to a piece of Clojure data"))

(extend-protocol ConvertFromDocument
  nil
  (document->hashmap [input] input)

  Object
  (document->hashmap [input] input)

  List
  (document->hashmap [^List input]
    (vec (map #(document->hashmap %) input)))

  ArrayList
  (document->hashmap [^ArrayList input]
    (vec (map #(document->hashmap %) input)))

  Document
  (document->hashmap [^Document input]
    (reduce (fn [m ^String k]
              (assoc m (keyword k) (document->hashmap (.get input k))))
            {}
            (.keySet input))))

(defprotocol ConvertToDocument
  (hashmap->document [input] "Converts given piece of Clojure data to Document MongoDB Java async driver uses"))

(extend-protocol ConvertToDocument
  nil
  (hashmap->document [_]
    nil)

  String
  (hashmap->document [^String input]
    input)

  Boolean
  (hashmap->document [^Boolean input]
    input)

  Date
  (hashmap->document [^Date input]
    input)

  Ratio
  (hashmap->document [^Ratio input]
    (double input))

  Keyword
  (hashmap->document [^Keyword input] (name input))

  Named
  (hashmap->document [^Named input] (name input))

  IPersistentMap
  (hashmap->document [^IPersistentMap input]
    (let [o (Document.)]
      (doseq [[k v] input]
        (.put o (hashmap->document k) (hashmap->document v)))
      o))

  List
  (hashmap->document [^List input] (map hashmap->document input))

  Set
  (hashmap->document [^Set input] (map hashmap->document input))

  Document
  (hashmap->document [^Document input] input)

  Object
  (hashmap->document [input]
    input))

(defmacro ->async-collection
  ([collection_name]
   `(->async-collection ~collection_name ReadPreference/secondaryPreferred))
  ([collection_name read_preference]
   `(doto (.getCollection
            (db)
            ~collection_name)
      (.withReadPreference (~read_preference))
      (.withWriteConcern (WriteConcern. 2 5000)))))


