(ns com.edocu.configuration.core
  (:use com.edocu.configuration.constants)
  (:require [clojure.tools.logging :as log]
            [com.edocu.configuration.etcd :as etcd]
            [cheshire.core :as json]
            [environ.core :as e]))

(def ENV (delay
           (let [result (or
                          (e/env :env)
                          :dev)]
             (log/info "Setting up in " result)
             result)))

(defn env []
  @ENV)

;############################# Defaults ################################

(def MONGODB_REPLICAS_URI_DEFAULT (delay
                                    (case (env)
                                      ("prod" "prod_unsecure") [{:uri "mongodb.1.db.prod.edocu.local" :port 27017}
                                                                {:uri "mongodb.2.db.prod.edocu.local" :port 27017}
                                                                {:uri "mongodb.3.db.prod.edocu.local" :port 27017}]
                                      "k8s-prod" [{:uri "mongodb1" :port 27017}
                                                  {:uri "mongodb2" :port 27017}
                                                  {:uri "mongodb3" :port 27017}]
                                      [{:uri "mongodb1.db.dev.edocu.local" :port 27017}
                                       {:uri "mongodb2.db.dev.edocu.local" :port 27017}
                                       {:uri "mongodb3.db.dev.edocu.local" :port 27017}])))
(def MONGODB_DB_DEFAULT (delay
                          (case (env)
                            ("prod" "prod_unsecure" "k8s-prod") "eDocu-production"
                            "eDocu-development")))


(defn with-default
  ([path default]
   (with-default
     path
     default
     identity))
  ([path default data_parser]
   (if-let [data (etcd/get-config-for
                   path)]
     (do
       (log/debug "with-default. path:" path "data:" data)
       (data_parser data))
     (do
       (log/debug "with-default. path:" path "default:" default)
       default))))

;############################ Base URL ###############################

(def base-url-atom (atom nil))

(defn base-url []
  (when-not @base-url-atom
    (reset! base-url-atom
            (with-default
              BASE_URL_PATH
              BASE_URL_DEFAULT))
    (log/trace "base-url" @base-url-atom))
  @base-url-atom)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-atom (atom nil))

(defn notifications-email []
  (when-not @notifications-email-atom
    (reset! notifications-email-atom
            (with-default
              NOTIFICATION_EMAIL_PATH
              NOTIFICATION_EMAIL_DEFAULT))
    (log/trace "notifications-email" @notifications-email-atom))
  @notifications-email-atom)

;######################################################################

;###################### Zookeepers Servers ############################

(def zookeepers-servers-atom (atom nil))

(defn zookeepers-servers []
  (when-not @zookeepers-servers-atom
    (reset! zookeepers-servers-atom
            (with-default
              ZOOKEPER_SERVERS_PATH
              ZOOKEPER_SERVERS_DEFAULT
              #(json/parse-string % true)))
    (log/trace "zookeepers-servers" @zookeepers-servers-atom))
  @zookeepers-servers-atom)

;######################################################################

;######################## Kafka Brokers ###############################

(def kafka-brokers-atom (atom nil))

(defn kafka-brokers []
  (when-not @kafka-brokers-atom
    (reset! kafka-brokers-atom
            (with-default
              KAFKA_BROKERS_PATH
              KAFKA_BROKERS_DEFAULT
              #(json/parse-string % true)))
    (log/trace "kafka-brokers" @kafka-brokers-atom))
  @kafka-brokers-atom)

;######################################################################

;######################## ZEROMQ ROUTER ###############################

(def zeromq-router-address-atom (atom nil))

(defn zeromq-router-address []
  (when-not @zeromq-router-address-atom
    (reset! zeromq-router-address-atom
            (with-default
              ZEROMQ_ROUTER_ADDRESS_PATH
              ZEROMQ_ROUTER_ADDRESS_DEFAULT))
    (log/trace "zeromq-router-address" @zeromq-router-address-atom))
  @zeromq-router-address-atom)

;######################################################################

;###################### ZEROMQ PUBLISHER ##############################

(def zeromq-publisher-address-atom (atom nil))

(defn zeromq-publisher-address []
  (when-not @zeromq-publisher-address-atom
    (reset! zeromq-publisher-address-atom
            (with-default
              ZEROMQ_PUBLISHER_ADDRESS_PATH
              ZEROMQ_PUBLISHER_ADDRESS_DEFAULT))
    (log/trace "zeromq-publisher-address" @zeromq-publisher-address-atom))
  @zeromq-publisher-address-atom)

;######################################################################

;######################### Redis Config ###############################

(def redis-config-atom (atom nil))

(defn redis-config
  ([]
   (redis-config REDIS_DB))
  ([db]
   (when-not @redis-config-atom
     (let [host (with-default
                  REDIS_HOST_PATH
                  REDIS_HOST)
           result {:pool {}
                   :spec {:host (or
                                  (e/env :redis)
                                  host)
                          :port 6379
                          :db   db}}]
       (reset! redis-config-atom
               result))
     (log/trace "redis-config" @redis-config-atom))
   @redis-config-atom))

;######################################################################

;############################ CBR topis ###############################

(def cbr-topic-atom (atom nil))

(defn cbr-topic []
  (when-not @cbr-topic-atom
    (reset! cbr-topic-atom
            (with-default
              CBR_TOPIC_PATH
              CBR_TOPIC_DEFAULT))
    (log/trace "cbr-topic" @cbr-topic-atom))
  @cbr-topic-atom)

;######################################################################

;########################## Register topis ############################

(def register-topic-atom (atom nil))

(defn register-topic []
  (when-not @register-topic-atom
    (reset! register-topic-atom
            (with-default
              REGISTER_TOPIC_PATH
              REGISTER_TOPIC_DEFAULT))
    (log/trace "register-topic" @register-topic-atom))
  @register-topic-atom)

;######################################################################

;########################## Register topis ############################

(def service-error-topic-atom (atom nil))

(defn service-error-topic []
  (when-not @service-error-topic-atom
    (reset! service-error-topic-atom
            (with-default
              SERVICE_ERROR_TOPIC_PATH
              SERVICE_ERROR_TOPIC_DEFAULT))
    (log/trace "service-error-topic" @service-error-topic-atom))
  @service-error-topic-atom)

;######################################################################

;########################## Register topis ############################

(def malformed-message-topic-atom (atom nil))

(defn malformed-message-topic []
  (when-not @malformed-message-topic-atom
    (reset! malformed-message-topic-atom
            (with-default
              MALFORMED_MESSAGE_TOPIC_PATH
              MALFORMED_MESSAGE_TOPIC_DEFAULT))
    (log/trace "malformed-message-topic" @malformed-message-topic-atom))
  @malformed-message-topic-atom)

;######################################################################

;##################### MongoDB Replicas URI ###########################

(def mongodb-replicas-uri-delay (delay
                                  (let [result (with-default
                                                 MONGODB_REPLICAS_URI_PATH
                                                 @MONGODB_REPLICAS_URI_DEFAULT
                                                 #(json/parse-string % true))]
                                    (log/trace "mongodb-replicas-uri" result)
                                    result)))

(defn mongodb-replicas-uri []
  @mongodb-replicas-uri-delay)

;######################################################################

;############################ MongoDB DB ##############################

(def mongodb-db-delay (delay
                        (let [result (with-default
                                       MONGODB_DB_PATH
                                       @MONGODB_DB_DEFAULT)]
                          (log/trace "mongodb-db" result)
                          result)))

(defn mongodb-db []
  @mongodb-db-delay)

;######################################################################

;########################### Integration ##############################

(def integration-env-atom (atom nil))

(defn integration-env []
  (when-not @integration-env-atom
    (reset! integration-env-atom
            (with-default
              INTEGRATION_ENV_PATH
              INTEGRATION_ENV))
    (log/trace "integration-env" @integration-env-atom))
  @integration-env-atom)

;######################################################################

(defn ELEMENT_STRUCTURE []
  (case (env)
    ("prod" "prod_unsecure") "http://editors.service.prod.edocu.local/api/v1/elementtypes"
    "k8s-prod" "http://editors/api/v1/elementtypes"
    "http://editors.service.dev.edocu.local/api/v1/elementtypes"))

(defn ELEMENTS_API []
  (case (env)
    ("prod" "prod_unsecure") "http://elements-api.service.prod.edocu.local"
    "k8s-prod" "http://elements-api"
    "http://elements-api.service.dev.edocu.local"))
