(ns com.edocu.configuration.core
  (:use com.edocu.configuration.constants)
  (:require [clojure.tools.logging :as log]
            [com.edocu.configuration.etcd :as etcd]
            [cheshire.core :as json]
            [environ.core :as e]))

(def ENV (delay
           (let [result (or
                          (e/env :env)
                          :dev)]
             (log/info "Setting up in " result)
             result)))

(defn env []
  @ENV)

;############################# Defaults ################################

(def MONGODB_REPLICAS_URI_DEFAULT (delay
                                    (case (env)
                                      "k8s-prod" [{:uri "mongo-0.mongo" :port 27017}
                                                  {:uri "mongo-1.mongo" :port 27017}
                                                  {:uri "mongo-2.mongo" :port 27017}]
                                      [{:uri "mongodb1" :port 27017}
                                       {:uri "mongodb2" :port 27017}
                                       {:uri "mongodb3" :port 27017}])))
(def MONGODB_DB_DEFAULT (delay
                          (case (env)
                            ("prod" "prod_unsecure" "k8s-prod") "eDocu-production"
                            "eDocu-development")))


(defn with-default
  ([path default]
   (with-default
     path
     default
     identity))
  ([path default data_parser]
   (if-let [data (etcd/get-config-for
                   path)]
     (do
       (log/debug "with-default. path:" path "data:" data)
       (data_parser data))
     (do
       (log/debug "with-default. path:" path "default:" default)
       default))))

;############################ Base URL ###############################

(def base-url-delay (delay
                      (let [result (with-default
                                     BASE_URL_PATH
                                     BASE_URL_DEFAULT)]
                        (log/trace "base-url" result)
                        result)))

(defn base-url []
  @base-url-delay)

;######################################################################

;##################### Notifications Email ############################

(def notifications-email-delay (delay
                                 (let [result (with-default
                                                NOTIFICATION_EMAIL_PATH
                                                NOTIFICATION_EMAIL_DEFAULT)]
                                   (log/trace "notifications-email" result)
                                   result)))

(defn notifications-email []
  @notifications-email-delay)

;######################################################################

;###################### Zookeepers Servers ############################

(def zookeepers-servers-delay (delay
                                (let [result (with-default
                                               ZOOKEPER_SERVERS_PATH
                                               ZOOKEPER_SERVERS_DEFAULT
                                               #(json/parse-string % true))]
                                  (log/trace "zookeepers-servers" result)
                                  result)))

(defn zookeepers-servers []
  @zookeepers-servers-delay)

;######################################################################

;######################## Kafka Brokers ###############################

(def kafka-brokers-delay (delay
                           (let [result (with-default
                                          KAFKA_BROKERS_PATH
                                          KAFKA_BROKERS_DEFAULT
                                          #(json/parse-string % true))]
                             (log/trace "kafka-brokers" result)
                             result)))

(defn kafka-brokers []
  @kafka-brokers-delay)

;######################################################################

;######################## ZEROMQ ROUTER ###############################

(def zeromq-router-address-delay (delay
                                   (let [result (with-default
                                                  ZEROMQ_ROUTER_ADDRESS_PATH
                                                  ZEROMQ_ROUTER_ADDRESS_DEFAULT)]
                                     (log/trace "zeromq-router-address" result)
                                     result)))

(defn zeromq-router-address []
  @zeromq-router-address-delay)

;######################################################################

;###################### ZEROMQ PUBLISHER ##############################

(def zeromq-publisher-address-delay (delay
                                      (let [result (with-default
                                                     ZEROMQ_PUBLISHER_ADDRESS_PATH
                                                     ZEROMQ_PUBLISHER_ADDRESS_DEFAULT)]
                                        (log/trace "zeromq-publisher-address" result)
                                        result)))

(defn zeromq-publisher-address []
  @zeromq-publisher-address-delay)

;######################################################################

;######################### Redis Config ###############################

(def redis-config-delay (delay
                          (let [host (with-default
                                       REDIS_HOST_PATH
                                       REDIS_HOST)
                                result {:pool {}
                                        :spec {:host (or
                                                       (e/env :redis)
                                                       host)
                                               :port 6379
                                               :db   REDIS_DB}}]
                            (log/trace "redis-config" result)
                            result)))

(defn redis-config
  ([]
   @redis-config-delay)
  ([db]
   (assoc-in
     @redis-config-delay
     [:spec :db]
     db)))

;######################################################################

;############################ CBR topis ###############################

(def cbr-topic-delay (delay
                       (let [result (with-default
                                      CBR_TOPIC_PATH
                                      CBR_TOPIC_DEFAULT)]
                         (log/trace "cbr-topic" result)
                         result)))

(defn cbr-topic []
  @cbr-topic-delay)

;######################################################################

;########################## Register topis ############################

(def register-topic-delay (delay
                            (let [result (with-default
                                           REGISTER_TOPIC_PATH
                                           REGISTER_TOPIC_DEFAULT)]
                              (log/trace "register-topic" result)
                              result)))

(defn register-topic []
  @register-topic-delay)

;######################################################################

;########################## Register topis ############################

(def service-error-topic-delay (delay
                                 (let [result (with-default
                                                SERVICE_ERROR_TOPIC_PATH
                                                SERVICE_ERROR_TOPIC_DEFAULT)]
                                   (log/trace "service-error-topic" result)
                                   result)))

(defn service-error-topic []
  @service-error-topic-delay)

;######################################################################

;########################## Register topis ############################

(def malformed-message-topic-delay (delay
                                     (let [result (with-default
                                                    MALFORMED_MESSAGE_TOPIC_PATH
                                                    MALFORMED_MESSAGE_TOPIC_DEFAULT)]
                                       (log/trace "malformed-message-topic" result)
                                       result)))

(defn malformed-message-topic []
  @malformed-message-topic-delay)

;######################################################################

;##################### MongoDB Replicas URI ###########################

(def mongodb-replicas-uri-delay (delay
                                  (let [result (with-default
                                                 MONGODB_REPLICAS_URI_PATH
                                                 @MONGODB_REPLICAS_URI_DEFAULT
                                                 #(json/parse-string % true))]
                                    (log/trace "mongodb-replicas-uri" result)
                                    result)))

(defn mongodb-replicas-uri []
  @mongodb-replicas-uri-delay)

;######################################################################

;############################ MongoDB DB ##############################

(def mongodb-db-delay (delay
                        (let [result (or
                                       (e/env :mongodb-db)
                                       (with-default
                                         MONGODB_DB_PATH
                                         @MONGODB_DB_DEFAULT))]
                          (log/trace "mongodb-db" result)
                          result)))

(defn mongodb-db []
  @mongodb-db-delay)

;######################################################################

;########################### Integration ##############################

(def integration-env-delay (delay
                             (let [result (with-default
                                            INTEGRATION_ENV_PATH
                                            INTEGRATION_ENV)]
                               (log/trace "integration-env" result)
                               result)))

(defn integration-env []
  @integration-env-delay)

;######################################################################

(defn ELEMENT_STRUCTURE []
  "http://editors/api/v1/elementtypes")

(defn ELEMENTS_API []
  "http://elements-api")

(defn JOB_SCHEDULE_API []
  "http://job-schedule")