(ns com.edocu.help.rest
  (:require [cheshire.core :as json]
            [com.edocu.users.protocols :as users]
            [clojure.core.async :refer [go >!]]
            [taoensso.timbre :as timbre]
            [clojure.core.memoize :as core-memo])
  (:import [java.io PipedInputStream PipedOutputStream]))

(defn is-know-content-type? [ctx supported]
  (some (set (map
               clojure.string/trim
               (clojure.string/split
                 (or
                   (get-in ctx [:request :headers "content-type"])
                   (get-in ctx [:request :headers "Content-type"])
                   (get-in ctx [:request :headers "Content-Type"])
                   "")
                 #";")))
        supported))

(defmacro know-content-type? [known]
  `(fn [ctx#]
     (if (#{:patch :post :put} (get-in ctx# [:request :request-method]))
       (is-know-content-type? ctx# ~known)
       true)))

(def memo-lazy->User
  (core-memo/ttl
    (fn [user_uid]
      (users/lazy->User (users/->Users) user_uid))
    :ttl/threshold 30000))

(defn unknow-user []
  (memo-lazy->User users/OPEN_USER_UID))

(defn construct-user [ctx]
  (if-let [user_uid (or
                      (get-in ctx [:request :headers "uid"])
                      (get-in ctx [:request :headers "UID"])
                      (get-in ctx [:request :headers "Uid"])
                      (get-in ctx [:request :headers "x-credential-username"])
                      (get-in ctx [:request :headers "X-Credential-Username"]))]
    (memo-lazy->User user_uid)
    (unknow-user)))

(defn malformed? [ctx checker_fn]
  (let [user (construct-user ctx)]
    (timbre/trace "Malformation check" "user:" user)
    (if (#{:patch :post :put} (get-in ctx [:request :request-method]))
      (try
        (let [body (json/parse-string (slurp (get-in ctx [:request :body])) true)]
          [(and
             (not (nil? (:uid user)))
             (checker_fn
               body))
           {:body body
            :user user}])
        (catch Exception e
          (timbre/error "Malformed input:" (get-in ctx [:request :body]) "error:" e)
          true))
      [(nil? (:uid user)) {:user user}])))

(defn options
  ([user]
   (options user {}))
  ([user others?]
   (merge-with merge
               {:timeout 10000
                :headers {"uid"    (:uid user)
                          "Accept" "application/json"}}
               others?)))

(defprotocol RESTLinks
  (links [this] "Return set of links"))

(defn method-to-privilege [ctx]
  ((get-in ctx [:request :request-method]) {:get    :read
                                            :head   :read
                                            :post   :update
                                            :put    :update
                                            :patch  :update
                                            :delete :update}))

(defn #^PipedInputStream create-response-stream [write_fn]
  (let [in_stream (PipedInputStream.)
        out_stream (PipedOutputStream. in_stream)]
    (future (write_fn out_stream))
    in_stream))
