(ns com.edocu.users.core
  (:use [com.edocu.users.protocols])
  (:require [com.edocu.users.http :as http]
            [com.edocu.help.memo :as memo :refer :all]
            [taoensso.timbre :as timbre]
            [clojure.core.async :refer [go chan <! <!! timeout alts! thread]]
            [com.edocu.users.configuration :as config])
  (:import [com.edocu.users.protocols 
            Users
            User
            OpenUser]))

(def global-impl
  {:lazy->User   (fn [_ uid]
                   (let [factory (if (= OPEN_USER_UID uid)
                                   map->OpenUser
                                   map->User)
                         user (factory {:uid uid})]
                     (timbre/trace "lazy->User" "uid" uid)
                     (init-memo-storage user)))

   :create->User (fn [_ author uid cn sn mail mobile preferredLanguage password]
                   (thread
                     (timbre/trace "create->User" "uid" uid
                                   "cn" cn
                                   "sn" sn
                                   "mail" mail
                                   "mobile" mobile
                                   "preferredLanguage" preferredLanguage)
                     (let [user (->User uid cn sn mail mobile preferredLanguage password)
                           [created? _] (alts! [(http/create->User author user) (timeout config/RESPONSE_TIME)])]
                       (when created?
                         (init-memo-storage user)))))})

(def privilege-impl
  {:element-types-with-privileges-in-organization (fn [this organization privileges]
                                                    (with-memo
                                                      this
                                                      [:element-types-with-privileges-in-organization]
                                                      (partial )))

   :privileges-in-element (fn
                            ([this edocu_element]
                              (privileges-in-element this edocu_element nil))
                            ([this edocu_element attribute_filter?]
                              (with-memo
                                this
                                [:privileges-in-element
                                 [(:element_hash edocu_element) attribute_filter?]]
                                (partial http/privileges-in-element
                                         this
                                         edocu_element
                                         attribute_filter?))))
   
   :privileges-in-organization (fn 
                                 ([this organization ^String element_type]
                                   (privileges-in-organization this organization element_type nil))
                                 ([this organization ^String element_type attribute_filter?]
                                   (with-memo
                                     this
                                     [:privileges-in-organization 
                                      [(:uid organization) element_type attribute_filter?]]
                                     (partial http/privileges-in-organization
                                              this
                                              organization
                                              element_type
                                              attribute_filter?))))
   
   :organizations (fn [this]
                    (with-memo
                      this
                      :organizations
                      (partial http/organizations
                               this)))
   
   :belongs-to-security-group (fn 
                                ([this ^String security_group_id]
                                  (with-memo
                                    this
                                    (format "sg_%s" security_group_id)
                                    (partial http/belongs-to-security-group
                                             this
                                             security_group_id)))
                                ([this ^String organization_id ^String title]
                                  (with-memo
                                    this
                                    (format "sg_%s_%s" organization_id title)
                                    (partial http/belongs-to-security-group
                                             this
                                             organization_id
                                             title))))})

(def open-privilege-impl
  (assoc privilege-impl
    :organizations
    (fn [_]
      (go {:respond (delay ["Public"])}))))

(extend Users
  Global
  global-impl)

(extend User
  Privilege
  privilege-impl)

(extend OpenUser
  Privilege
  open-privilege-impl)
