(ns com.edocu.users.protocols
  (:require [clojure.spec :as s]))

(def ^:const ^String OPEN_USER_UID "open")

(s/def ::uid string?)
(s/def ::element-type string?)
(s/def ::attribute keyword)

;---------------------------------------------------- Users ---------------------------------------------------------

(defrecord Users [])

(def ->Users (memoize ->Users))

(s/def ::cn (s/nilable string?))
(s/def ::sn (s/nilable string?))
(def email-regex #"^[a-zA-Z0-9._%+-]+@[a-zA-Z0-9.-]+\.[a-zA-Z]{2,63}$")
(s/def ::mail (s/with-gen
                (s/nilable (s/and string?
                                  #(re-matches email-regex %)))
                #(s/gen #{nil "test1@test.test" "test2@test.test"})))
(s/def ::mobile (s/nilable string?))
(s/def ::preferredLanguage (s/nilable string?))
(s/def ::password (s/nilable string?))

(s/def ::user (s/keys :req-un [::uid]
                      :opt-un [::cn ::sn ::mail ::mobile ::preferredLanguage ::password]))

(defrecord User [uid cn sn mail mobile preferredLanguage password]
  Object
  (toString [this] (pr-str (dissoc this :password))))

(defrecord OpenUser [uid cn sn mail mobile preferredLanguage password]
  Object
  (toString [this] (pr-str (dissoc this :password))))

;------------------------------------------------ Privileges ------------------------------------------------------

(s/def ::type-privilege #{:read :update :create})
(s/def ::attribute-privilege #{:read :update})

(s/def ::type (s/coll-of ::type-privilege :kind set?))
(s/def ::attributes (s/or
                      :with-filter (s/coll-of ::attribute :kind set?)
                      :without-filter (s/map-of keyword?
                                                (s/coll-of ::attribute-privilege :kind set?))))

(s/def ::privileges (s/keys :req-un [::type ::attributes]))

(defrecord Privileges [type attributes]
  Object
  (toString [this] (pr-str this)))

;----------------------------------------------- Organization ----------------------------------------------------

(s/def ::organization (s/keys :req-un [::uid]))
(s/def ::organizations (s/coll-of ::uid :kind vector?))

;----------------------------------------------- eDocu element ----------------------------------------------------

(s/def ::element_hash string?)
(s/def ::edocu-element (s/keys :req-un [::element_hash]))

;----------------------------------------------- Element Type ----------------------------------------------------

(s/def ::id string?)
(s/def ::element-type (s/keys :req-un [::id]))
(s/def ::element_types (s/coll-of ::element-type :kind vector?))

;------------------------------------------------ Protocols ------------------------------------------------------

(defprotocol UsersDirector
  (lazy->User [director uid] "Return lazy instantion of User")
  (create->User [director author uid cn sn mail mobile preferredLanguage password] "Return channel with user object if created"))

(s/fdef create->User
        :args (s/cat :director any?
                     :author ::user
                     :cn ::cn
                     :sn ::sn
                     :mail ::mail
                     :mobile ::mobile
                     :preferredLanguage ::preferredLanguage
                     :password ::password))

(defprotocol Privilege
  (organizations [this] "Return channel with user's organizations")
  (element-types-with-privileges-in-organization [this organization privilege] "Return channel with only with element types with required privilege")
  (privileges-in-organization
    [this organization element_type]
    [this organization element_type attribute_filter?] "Return channel with user privilege in organization for element type")
  (privileges-in-element
    [this edocu_element]
    [this edocu_element attribute_filter?] "Return channel with user privileges in element")
  (belongs-to-security-group
    [this security_group_id]
    [this organization_id title] "Return channel with true if user belongs to organization. Otherwise false"))

(s/fdef organizations
        :args (s/cat :this ::user))

(s/fdef privileges-in-organization
        :args (s/or
                :without-filter (s/cat :this ::user
                                       :edocu_element ::edocu-element)
                :with-filter (s/cat :this ::user
                                    :organization ::organization
                                    :element_type ::element-type
                                    :attribute_filter? ::type-privilege)))

(s/fdef privileges-in-element
        :args (s/or
                :without-filter (s/cat :this ::user
                                       :edocu_element ::edocu-element)
                :with-filter (s/cat :this ::user
                                    :edocu_element ::edocu-element
                                    :attribute_filter? ::type-privilege)))

(s/fdef belongs-to-security-group
        :args (s/or
                :by-security-group (s/cat :this ::user
                                          :security_group_id ::uid)
                :by-organization (s/cat :this ::user
                                        :organization_id ::uid
                                        :title string?)))

(s/fdef element-types-with-privileges-in-organization
        :args (s/cat :this ::user
                     :organization ::organization
                     :privilege ::type-privilege))