(ns com.edocu.elements.type.protocols
  (:require [clojure.spec :as s]
            [clojure.core.async :as async]))

(s/def ::response-channel #(satisfies? clojure.core.async.impl.protocols/ReadPort %))

(defprotocol Global
  (lazy->ElementType [director type_id] "Return lazy ElementType object"))

(defprotocol TypeManager
  (structure [element_type] "Return channel with element type structure")
  (file-attributes [element_type] "Return channel with set of file attributes")
  (text-attributes [element_type] "Return channel with set of text attributes")
  (dropdown-attributes [element_type] "Return channel with set of dropdown attributes")
  (text-area-attributes [element_type] "Return channel with set of textarea attributes")
  (date-time-attributes [element_type] "Return channel with set of datetime attributes")
  (date-attributes [element_type] "Return channel with set of date attributes"))

(defrecord ElementTypes [])

(def ->ElementTypes (memoize ->ElementTypes))

(s/def ::type_id string?)

(s/def ::element-type (s/and
                        (s/keys :req-un [::type_id])
                        #(satisfies? TypeManager %)))

(defrecord ElementType [type_id]
  Object
  (toString [this] (pr-str this)))

;;--------------------------- Element Structure -----------------------------

(s/def ::name string?)
(s/def ::required boolean?)
(s/def ::type string?)
(s/def ::order number?)
(s/def ::key string?)
(s/def ::modifiable boolean?)
(s/def ::values (s/coll-of string?))
(s/def ::markdown boolean?)
(s/def ::multiple boolean?)
(s/def ::query string?)
(s/def ::attribute (s/keys :req-un [::required ::type ::order ::key]
                           :opt-un [::values ::markdown ::multiple ::query ::modifiable]))
(s/def ::attributes (s/map-of keyword? ::attribute))
(s/def ::structure (s/keys :req-un [::name ::attributes]))

;;---------------------------------------------------------------------------

(s/def ::element-types #(satisfies? Global %))

(s/fdef lazy->ElementType
        :args (s/cat :director ::element-types :type_id ::type_id)
        :ret ::element-type)

(s/fdef structure
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef file-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef text-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef dropdown-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef text-area-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef date-time-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)

(s/fdef date-attributes
        :args (s/cat :element_type ::element-type)
        :ret ::response-channel)