(ns selmeci.clojure.lang.redis-persistent-map
  (:require [taoensso.carmine :as car :refer (wcar)]
            [liberator.representation])
  (:import [clojure.lang MapEntry]
           [liberator.representation RingResponse]
           [java.util.AbstractMap$SimpleEntry]))

(defn set-command [server_ops k v]
  (wcar server_ops (car/set k (car/freeze v))))

(defn del-command [server_ops k]
  (wcar server_ops (car/del k)))

(defprotocol RedisServerOps
  (server-ops [this])
  (key-prefix [this]))

(deftype RedisPersistentMap [server_ops key_prefix]
  RedisServerOps
  (server-ops [this]
    server_ops)
  (key-prefix [this]
    key_prefix)
  
  clojure.lang.IFn
  (invoke [this k]
    (wcar server_ops (car/get (car/key key_prefix k))))
  
  (invoke [this k not_found]
    (if-let [v (wcar server_ops (car/get (car/key key_prefix k)))]
      v
      not_found))
  
  clojure.lang.IPersistentMap
  (assoc [this k v]
    (set-command server_ops (car/key key_prefix k) v)
    this)
  (assocEx [this k v]
    (if (= 1 (wcar server_ops (car/exists (car/key key_prefix k))))
      (throw (java.lang.RuntimeException. "Key already present"))
      (set-command server_ops (car/key key_prefix k) v))
    this)
  (without [this k]
    (del-command server_ops (car/key key_prefix k))
    this)
  
  java.lang.Iterable
  (iterator [this]
    (.iterator
      (map (fn [k] 
             (java.util.AbstractMap$SimpleEntry. (keyword (second (clojure.string/split k #":")))
                                                 (wcar server_ops (car/get k))))
           (wcar server_ops (car/keys (car/key key_prefix "*"))))))
  
  clojure.lang.Associative
  (containsKey [this k]
    (= 1 (wcar server_ops (car/exists (car/key key_prefix k)))))
  (entryAt [this k]
    (MapEntry. k (wcar server_ops (car/get (car/key key_prefix k)))))
  
  clojure.lang.IPersistentCollection
  (count [_]
    (clojure.core/count (wcar server_ops (car/keys (car/key key_prefix "*")))))
  (cons [this [k v]]
    (set-command server_ops (car/key key_prefix k) v)
    this)
  (empty [this]
    (let [rm_keys (wcar server_ops (car/keys (car/key key_prefix "*")))
          rm (partial apply car/del rm_keys)]
      (wcar server_ops (rm)))
    this)
  (equiv [this o]
    (and (isa? (class o) RedisPersistentMap)
         (.equiv server_ops (server-ops o))
         (= key_prefix (:key_prefix o))))
  
  clojure.lang.Seqable
  (seq [this]
    (.seq 
      (map (fn [k] 
             (java.util.AbstractMap$SimpleEntry. (keyword (second (clojure.string/split k #":")))
                                                 (wcar server_ops (car/get k))))
           (wcar server_ops (car/keys (car/key key_prefix "*"))))))
  
  clojure.lang.ILookup
  (valAt [this k]
    (wcar server_ops (car/get (car/key key_prefix k))))
  (valAt [this k not_found]
    (if-let [v (wcar server_ops (car/get (car/key key_prefix k)))]
      v
      not_found)))
