(ns com.edocu.communication.zeromq.core
  (:require [clojure.tools.logging :as log]
            [com.keminglabs.zmq-async.core :refer [register-socket! create-context initialize!]]
            [clojure.core.async :refer [>! <! go chan sliding-buffer close!]]
            [cheshire.core :as json]))

(def ^:const MALFORMED_MESSAGE_TOPIC "system.errors.message.malformed")
(def ^:const SERVICE_ERROR_TOPIC "system.errors.service")
(def ^:const CHANNEL_REGISTRATION_TOPIC "system.router.register-topic")

(defrecord Communication [router_address publisher_address])

(defprotocol MessageManagement
  (send-message! [this topic message] "Send message to chanel with topic"))

(defprotocol TopicManagement
  (register-topic! [this topic] "Register new channel with topic in router")
  (subscribe-to-topic [this topic callback_chan] "Subscribe to topic with callback for received messages"))

(defprotocol ErrorsManagement
  (report-malformed-message! [this message] "Report malformed message to system topic channel")
  (report-service-error! [this message] "Report service error during execution of business rules"))

(def message-management-impl
  {:send-message! (fn [this topic message]
                    (log/trace "send-message!:" "topic:" topic "message:" message)
                    (let [router-in (chan )] 
                      (register-socket! {:in router-in
                                         :socket-type :dealer
                                         :configurator (fn [socket]
                                                         (.connect socket (:router_address this)))})
                      (go
                        (>! router-in [topic (json/generate-string message {:escape-non-ascii true})])
                        (close! router-in))))})

(def topic-management-impl
  {:register-topic! (fn [this topic]
                      (log/trace "register-topic!:" topic)
                      (send-message! this CHANNEL_REGISTRATION_TOPIC {:topic topic}))
   
   :subscribe-to-topic (fn [this topic callback_chan]
                         (log/trace "subscribe-to-topic" topic)
                         (let [subscriber-out (chan (sliding-buffer 1024))
                               subscriber-socket (doto (create-context (str (java.util.UUID/randomUUID)))
                                                   (initialize!))] 
                           (register-socket! {:out subscriber-out
                                              :socket-type :sub
                                              :context subscriber-socket
                                              :configurator (fn [socket]
                                                              (.connect socket (:publisher_address this))
                                                              (.subscribe socket (.getBytes topic)))})
                           (go
                             (while true
                               (let [[_ msg_topic body] (<! subscriber-out)
                                     msg_topic (clojure.string/split (String. msg_topic) #"\.")
                                     body (json/parse-string (String. body) true)
                                     result [msg_topic body]]
                                 (log/trace "receive sub message:" "topic:" msg_topic "message" result)
                                 (>! callback_chan result))))))})

(def errors-management-impl
  {:report-malformed-message! (fn [this message]
                                (log/trace "report-malformed-message!:" message)
                                (send-message! this MALFORMED_MESSAGE_TOPIC message))
   :report-service-error! (fn [this message]
                            (log/trace "report-service-error!:" message)
                            (send-message! this SERVICE_ERROR_TOPIC message))})

(extend Communication
  MessageManagement
  message-management-impl
  
  TopicManagement
  topic-management-impl
  
  ErrorsManagement
  errors-management-impl)