(ns etcd-clojure.core
  (:use [etcd-clojure.util])
  (:require [clj-http.client :as http])
  (:require [cheshire.core :refer :all])
  (:refer-clojure :exclude [list get set]))

(def ^:private endpoint (atom "http://127.0.0.1:2379"))

(def ^:private api-version "v2")

(defn ^:private base-url
  []
  (str @endpoint "/" api-version))

(def ^:dynamic *throw-on-error* true)

(defn connect!
  ([etcd-server-host] (connect! etcd-server-host 2379))
  ([etcd-server-host port]
     [(reset! endpoint (format "http://%s:%s" etcd-server-host port))]))

(defn api-get-in
  [response ks]
  (let [error-code (clojure.core/get response "errorCode")]
    ;; 100 shows that key does not exist, nill will be returned from
    ;;  NE-path in response
    (if (and error-code (not= error-code 100))
      (if *throw-on-error*
        (throw (ex-info (clojure.core/get response "message")
                        response))
        response)
    (get-in response ks))))

(defn version
  "Gets the etcd server version"
  []
    (get-json http/get (str @endpoint "/version")))

(defn set
  "Sets a vaue to key, optional param :ttl"
  [key val & {:keys [ttl prev-value prev-index prev-exist wait]}]
  (let
      [params (compose-params
                :ttl ttl
                :val val
                :prev-value prev-value
                :prev-index prev-index
                :prev-exist prev-exist)
        url (str (base-url) "/keys/" key)]
    (api-get-in (send-json http/put url {:form-params params}) ["node" "value"])))

(defn get
  "Gets a value"
  [key & {:keys [wait callback]}]
  (let [url (str (base-url) "/keys/" key)]
    (if (nil? wait)
      (api-get-in (get-json http/get url) ["node" "value"])
      (let [f (future (api-get-in (get-json http/get (str url "?" (compose-query-string {:wait wait}))) ["node" "value"]))]
        (when-done f #(callback %)) f))
    ))

(defn delete
  "Deletes a value"
  [key & {:keys [prev-value prev-index prev-exist]}]
  (let [query-string (compose-query-string {:prevValue prev-value :prevIndex prev-index})
        url (str (base-url) "/keys/" key)]
    (if (empty? [prev-value prev-index prev-exist])
      (api-get-in (send-json http/delete url) ["node" "key"])
      (api-get-in (send-json http/delete (str url "?" query-string)) ["node" "key"]))))

(defn delete-dir
  "Deletes a dir"
  [key]
  (api-get-in (send-json http/delete (str (base-url) "/keys/" key "?dir=true")) ["node" "key"]))

(defn delete-dir-recur
  "Deletes a directory recursively"
  [key]
  (api-get-in (send-json http/delete (str (base-url) "/keys/" key "?dir=true&recursive=true")) ["node" "key"]))

(defn create-in-order
  "Creates in order"
  [key value]
  (let [url (str (base-url) "/keys/" key)]
    (api-get-in (send-json http/post url {:form-params {:value value}}) ["node" "value"])))

(defn list
  "Lists the content of a directory"
  [key & {:keys [recursive] :or {recursive false}}]
  (let [url (str (base-url) "/keys/" key "?recursive=" recursive)]
    (map #(assoc {} :key (clojure.core/get % "key") :value (clojure.core/get % "value"))
         (api-get-in (get-json http/get url) ["node" "nodes"]))))

(defn list-in-order
  "Lists the content of a directory recursively and in order"
  [key]
  (let [url (str (base-url) "/keys/" key "?recursive=true&sorted=true")]
    (map #(clojure.core/get % "value") (api-get-in (get-json http/get url) ["node" "nodes"]))))

(defn create-dir
  "Creates a dir"
  [key & {:keys [ttl]}]
  (let [params (compose-params
                :ttl ttl
                :dir true)
        url (str (base-url) "/keys/" key)]
    (api-get-in (send-json http/put url {:form-params params}) ["node" "key"])))

(defn stats
  "Leader stats"
  []
  (get-json http/get (base-url) "/stats/leader"))

(defn self-stats
  "Self Stats"
  []
  (get-json http/get (base-url) "/stats/self"))

(defn store-stats
  "Store stats"
  []
  (get-json http/get (base-url) "/stats/store"))

(defn members
  "Machines"
  []
  (get-json http/get (base-url) "/members"))