(ns org.httpkit.client
  (:refer-clojure :exclude [get])
  (:require [clojure.string :as str]
            [com.edocu.cache.core :as edocu-cache]
            [clojure.core.cache :as cache]
            [clojure.core.async :refer [>! <! <!! go go-loop chan close!]]
            [pandect.core :as pandect]
            [taoensso.timbre :as timbre]
            [cheshire.core :as json])
  (:use [clojure.walk :only [prewalk]])
  (:import [org.httpkit.client HttpClient IResponseHandler RespListener
            IFilter RequestConfig]
           [org.httpkit HttpMethod PrefixThreadFactory HttpUtils]
           [java.util.concurrent ThreadPoolExecutor LinkedBlockingQueue TimeUnit]
           [java.net URI URLEncoder]
           [org.httpkit.client SslContextFactory MultipartEntity]
           javax.xml.bind.DatatypeConverter))

;;;; Utils

(defn- utf8-bytes    [s]     (.getBytes         (str s) "utf8"))
(defn url-encode     [s]     (URLEncoder/encode (str s) "utf8"))
(defn- base64-encode [bytes] (DatatypeConverter/printBase64Binary bytes))

(defn- basic-auth-value [basic-auth]
  (let [basic-auth (if (string? basic-auth)
                     basic-auth
                     (str (first basic-auth) ":" (second basic-auth)))]
    (str "Basic " (base64-encode (utf8-bytes basic-auth)))))

(defn- prepare-request-headers
  [{:keys [headers form-params basic-auth oauth-token user-agent] :as req}]
  (merge headers
         (when form-params {"Content-Type"  "application/x-www-form-urlencoded"})
         (when basic-auth  {"Authorization" (basic-auth-value basic-auth)})
         (when oauth-token {"Authorization" (str "Bearer " oauth-token)})
         (when user-agent  {"User-Agent"    user-agent})))

(defn- prepare-response-headers [headers]
  (reduce (fn [m [k v]] (assoc m (keyword k) v)) {} headers))

;;; {:a {:b 1 :c [1 2 3]}} => {"a[b]" 1, "a[c]" [1 2 3]}
(defn- nested-param [params]            ; code copyed from clj-http
  (prewalk (fn [d]
             (if (and (vector? d) (map? (second d)))
               (let [[fk m] d]
                 (reduce (fn [m [sk v]]
                           (assoc m (str (name fk) \[ (name sk) \]) v))
                         {} m))
               d))
           params))

(defn- query-string
  "Returns URL-encoded query string for given params map."
  [m]
  (let [m (nested-param m)
        param (fn [k v]  (str (url-encode (name k)) "=" (url-encode v)))
        join  (fn [strs] (str/join "&" strs))]
    (join (for [[k v] m] (if (sequential? v)
                           (join (map (partial param k) (or (seq v) [""])))
                           (param k v))))))

(comment (query-string {:k1 "v1" :k2 "v2" :k3 nil :k4 ["v4a" "v4b"] :k5 []}))

(defn- coerce-req
  [{:keys [url method body insecure? query-params form-params multipart] :as req}]
  (let [r (assoc req
            :url (if query-params
                   (if (neg? (.indexOf ^String url (int \?)))
                     (str url "?" (query-string query-params))
                     (str url "&" (query-string query-params)))
                   url)
            :sslengine (or (:sslengine req)
                           (when (:insecure? req) (SslContextFactory/trustAnybody)))
            :method    (HttpMethod/fromKeyword (or method :get))
            :headers   (prepare-request-headers req)
            ;; :body ring body: null, String, seq, InputStream, File, ByteBuffer
            :body      (if form-params (query-string form-params) body))]
    (if multipart
      (let [entities (map (fn [{:keys [name content filename]}]
                            (MultipartEntity. name content filename)) multipart)
            boundary (MultipartEntity/genBoundary entities)]
        (-> r
            (assoc-in [:headers "Content-Type"]
                      (str "multipart/form-data; boundary=" boundary))
            (assoc :body (MultipartEntity/encode boundary entities))))
      r)))

;; thread pool for executing callbacks, since they may take a long time to execute.
;; protect the IO loop thread: no starvation
(def default-pool (let [max (.availableProcessors (Runtime/getRuntime))
                        queue (LinkedBlockingQueue.)
                        factory (PrefixThreadFactory. "client-worker-")]
                    (ThreadPoolExecutor. max max 60 TimeUnit/SECONDS queue factory)))

;;;; Cache

(def ^:private cache-ref (promise ))
(def ^:private etag-cache-ref (ref nil))

(defn initialize-cache! [redis_config threshold]
  (dosync
    (deliver cache-ref 
             (edocu-cache/create->lru-cache
               redis_config
               "#edocu_cache_lock"
               "edocu_cache"
               threshold))
    (ref-set etag-cache-ref
             (cache/lru-cache-factory 
               {} 
               :threshold threshold))
    ))

(def ^:private etag-cache-manager-promise (promise ))

(defn- etag-cache-manager []
  (if @etag-cache-ref
    (do
      (if-not (realized? etag-cache-manager-promise)
        (do
          (deliver etag-cache-manager-promise (chan ))
          (edocu-cache/run-cache-changer 
            @etag-cache-manager-promise)))
      @etag-cache-manager-promise)
    nil))

(defn- generate-request-key [url headers]
  (pandect/md5 (str "httpkit_request" url headers)))

(defn- headers-with-etag [url headers]
  (if  @etag-cache-ref
    (let [etag_key (generate-request-key url headers)]
      (if-let [etag_value (cache/lookup @etag-cache-ref etag_key)]
        (assoc headers "If-None-Match" etag_value)
        headers))
    headers))

(defn- etag [headers]
  (if-let [etag_value (or (:etag headers)
                          (clojure.core/get headers "If-None-Match"))]
    (last (re-find #"(W\/)?(\w+)" etag_value))
    nil))

(defn- miss-in-cache [url req_headers res_headers]
  (if @etag-cache-ref
    (if-let [respond_etag (etag res_headers)]
      (let [etag_key (generate-request-key url req_headers)]
        (edocu-cache/with-cache
          etag-cache-ref (etag-cache-manager ) 
          (edocu-cache/miss-command etag-cache-ref etag_key respond_etag))))))

(defn- hit-in-cache [url req_headers res_headers]
  (if @etag-cache-ref
    (let [etag_key (generate-request-key url req_headers)]
      (edocu-cache/with-cache
        etag-cache-ref (etag-cache-manager )
        (edocu-cache/hit-command etag-cache-ref etag_key)))))

(defn- deliver-resp [deliver_resp opts body headers status etag_value]
  (deliver_resp {:opts    (dissoc opts :response)
                 :body    body
                 :headers headers
                 :status  status
                 :etag etag_value}))

(defn- handle-respond-with-cache-dispatch [method url req_headers deliver_resp opts body res_headers status]
  [method status])

(defmulti ^:private handle-respond-with-cache
  #'handle-respond-with-cache-dispatch)

(defmethod ^:private handle-respond-with-cache :default
  [_ _ _ deliver_resp opts body res_headers status]
  (deliver-resp deliver_resp opts body res_headers status nil))

(defmethod ^:private handle-respond-with-cache [:get 200]
  [_ url req_headers deliver_resp opts body res_headers status]
  (miss-in-cache url req_headers res_headers)
  (deliver-resp deliver_resp opts (delay body) res_headers status (etag res_headers)))

(defmethod ^:private handle-respond-with-cache [:get 304]
  [_ url req_headers deliver_resp opts body res_headers status]
  (hit-in-cache url req_headers res_headers)
  (let [etag_value (etag req_headers)]
    (deliver-resp deliver_resp opts 
                 (delay (cache/lookup @@cache-ref etag_value) )
                 res_headers status etag_value)))

;;;; Public API

(defn max-body-filter "reject if response's body exceeds size in bytes"
  [size] (org.httpkit.client.IFilter$MaxBodyFilter. (int size)))

;;; "Get the default client. Normally, you only need one client per application. You can config parameter per request basic"
(defonce default-client (delay (HttpClient.)))

(defn request
  "Issues an async HTTP request and returns a promise object to which the value
  of `(callback {:opts _ :status _ :headers _ :body _})` or
     `(callback {:opts _ :error _})` will be delivered.

  When unspecified, `callback` is the identity

  ;; Asynchronous GET request (returns a promise)
  (request {:url \"http://www.cnn.com\"})

  ;; Asynchronous GET request with callback
  (request {:url \"http://www.cnn.com\" :method :get}
    (fn [{:keys [opts status body headers error] :as resp}]
      (if error
        (println \"Error on\" opts)
        (println \"Success on\" opts))))

  ;; Synchronous requests
  @(request ...) or (deref (request ...) timeout-ms timeout-val)

  ;; Issue 2 concurrent requests, then wait for results
  (let [resp1 (request ...)
        resp2 (request ...)]
    (println \"resp1's status: \" (:status @resp1))
    (println \"resp2's status: \" (:status @resp2)))

  Output coercion:
  ;; Return the body as a byte stream
  (request {:url \"http://site.com/favicon.ico\" :as :stream})
  ;; Coerce as a byte-array
  (request {:url \"http://site.com/favicon.ico\" :as :byte-array})
  ;; return the body as a string body
  (request {:url \"http://site.com/string.txt\" :as :text})
  ;; Try to automatically coerce the output based on the content-type header, currently supports :text :stream, (with automatic charset detection)
  (request {:url \"http://site.com/string.txt\" :as :auto})

  Request options:
    :url :method :headers :timeout :query-params :form-params :as
    :client :body :basic-auth :user-agent :filter :worker-pool"
  [{:keys [client timeout filter worker-pool keepalive as follow-redirects max-redirects response]
    :as opts
    :or {client @default-client
         timeout 60000
         follow-redirects true
         max-redirects 10
         filter IFilter/ACCEPT_ALL
         worker-pool default-pool
         response (promise)
         keepalive 120000
         as :auto}}
   & [callback]]
  (let [{:keys [url method headers body sslengine]} (coerce-req opts)
        req_headers (headers-with-etag url headers)
        deliver_resp #(deliver response ;; deliver the result
                               (try ((or callback identity) %1)
                                    (catch Exception e
                                      ;; dump stacktrace to stderr
                                      (HttpUtils/printError (str method " " url "'s callback") e)
                                      ;; return the error
                                      {:opts opts :error e})))
        handler (reify IResponseHandler
                  (onSuccess [this status headers body]
                    (if (and follow-redirects
                             (#{301 302 303 307 308} status)) ; should follow redirect
                      (if (>= max-redirects (count (:trace-redirects opts)))
                        (request (assoc opts ; follow 301 and 302 redirect
                                   :url (.toString ^URI (.resolve (URI. url) ^String
                                                                  (.get headers "location")))
                                   :response response
                                   :method (if (#{301 302 303} status)
                                             :get ;; change to :GET
                                             (:method opts))  ;; do not change
                                   :trace-redirects (conj (:trace-redirects opts) url))
                                 callback)
                        (deliver_resp {:opts (dissoc opts :response)
                                       :error (Exception. (str "too many redirects: "
                                                               (count (:trace-redirects opts))))}))
                      (handle-respond-with-cache 
                        (:method opts)
                        url
                        req_headers
                        deliver_resp 
                        opts 
                        body 
                        (prepare-response-headers headers) 
                        status)))
                  (onThrowable [this t]
                    (deliver_resp {:opts opts :error t})))
        listener (RespListener. handler filter worker-pool
                                ;; only the 4 support now
                                (case as :auto 1 :text 2 :stream 3 :byte-array 4))
        cfg (RequestConfig. method req_headers body timeout keepalive)]
    (.exec ^HttpClient client url cfg sslengine listener)
    response))

(defmacro ^:private defreq [method]
  `(defn ~method
     ~(str "Issues an async HTTP " (str/upper-case method) " request. "
           "See `request` for details.")
     ~'{:arglists '([url & [opts callback]] [url & [callback]])}
     ~'[url & [s1 s2]]
     (if (or (instance? clojure.lang.MultiFn ~'s1) (fn? ~'s1))
       (request {:url ~'url :method ~(keyword method)} ~'s1)
       (request (merge ~'s1 {:url ~'url :method ~(keyword method)}) ~'s2))))

(defreq get)
(defreq delete)
(defreq head)
(defreq post)
(defreq put)
(defreq options)
(defreq patch)
(defreq propfind)
(defreq proppatch)
(defreq lock)
(defreq unlock)
(defreq report)
(defreq acl)
(defreq copy)
(defreq move)

(defrecord EDocuRespond [etag respond])

(def ^:const ^String ERROR_ETAG "error-etag")

(defn- edocu-respond-handler 
  ([method respond_channel base_url options service builder]
    (edocu-respond-handler method respond_channel base_url options service builder true))
  ([method respond_channel base_url options service builder parse_body?]
    (method base_url options
            (fn [{:keys [status headers body error etag]}]
              (go 
                (let [error_respond (->EDocuRespond ERROR_ETAG (delay (builder {})))]
                  (if (or error
                          (>= status 400))
                    (do
                      (timbre/error service "service error:" error "status:" status "url:" base_url)
                      (>! respond_channel error_respond))
                    (>! respond_channel
                        (try
                          (let [data (delay (if (and parse_body?
                                                     (or
                                                       (= 200 status)
                                                       (= 304 status)))
                                              (json/parse-string @body true)
                                              body))]
                            (->EDocuRespond etag (delay (builder @data))))
                          (catch Exception e
                            (timbre/error service "malformed respond body:" body "error:" e)
                            error_respond))))))))))

(def edocu-get (partial edocu-respond-handler get))

(def edocu-post (partial edocu-respond-handler post))
