(ns com.edocu.help.db.neo4j
  (:require [com.stuartsierra.component :as component]
            [clojure.tools.logging :as log]
            [environ.core :as e]
            [com.edocu.help.sentry :as sentry]
            [clojure.core.async :as async :refer [close! go >! >!! chan thread put!]])
  (:import [org.neo4j.driver.v1 GraphDatabase AuthTokens Session StatementResult Record Config Config$EncryptionLevel]))

(def ^:const n 128)

(defrecord Statement [client statement params default >result])

(defn- statement-run [{:keys [client statement params default >result]} >callback]
  (thread
    (with-open [^Session session (.session client)]
      (try
        (let [^StatementResult result (if params
                                        (.run session statement (zipmap
                                                                  (map name (keys params))
                                                                  (map identity (vals params))))
                                        (.run session statement))]
          (while (.hasNext result)
            (let [^Record record (.next result)]
              (>!! >result (.asMap record)))))
        (catch Exception e
          (sentry/put-in-mdc {:statement statement
                              :params    params})
          (log/error e "statement-run")
          (if (some? default)
            (>!! >result default)
            (>!! >result :error)))
        (finally
          (close! >result)
          (close! >callback))))))

(defrecord TransactionStatement [statement params])

(defprotocol INeo4J
  (<statement-run
    [this statement]
    [this statement params]
    [this statement params default] "Run statement in Neo4J with bolt")
  (<transaction-run
    [this statements] "Run statements in transaction"))

(defprotocol Client
  (client [this] "Return db client"))

(defrecord Neo4JClient [driver >statements]
  component/Lifecycle
  (start [this]
    (log/trace "Starting Neo4JClient")
    (let [>jobs (chan n)]
      (async/pipeline-async
        n
        (chan (async/sliding-buffer 8))
        statement-run
        >jobs)
      (assoc this
        :driver (delay
                  (try
                    (let [username (e/env :neo4j-username)
                          password (e/env :neo4j-password)]
                      (if (and username password)
                        (GraphDatabase/driver (e/env :neo4j-uri) (AuthTokens/basic username password))
                        (GraphDatabase/driver (e/env :neo4j-uri)
                                              (.. Config
                                                  build
                                                  (withEncryptionLevel (Config$EncryptionLevel/NONE))
                                                  toConfig))))
                    (catch Exception e
                      (sentry/put-in-mdc {:url (e/env :neo4j-uri)})
                      (log/error e "Neo4JClient")
                      nil)))
        :>statements >jobs)))
  (stop [this]
    (log/trace "Stopping Neo4JClient")
    (dissoc this :driver))

  Client
  (client [this]
    @driver)

  INeo4J
  (<statement-run [this statement]
    (<statement-run this statement nil nil))

  (<statement-run [this statement params]
    (<statement-run this statement params nil))

  (<statement-run [this statement params default]
    (let [>result (chan)]
      (put! >statements (->Statement (client this) statement params default >result))
      >result))

  (<transaction-run
    [this statements]
    (thread
      (try
        (with-open [^Session session (.session (client this))]
          (doseq [{:keys [statement params]} statements]
            (if params
              (.run session statement (zipmap
                                        (map name (keys params))
                                        (map identity (vals params))))
              (.run session statement)))
          :ok)
        (catch Exception e
          (sentry/put-in-mdc {:statements statements})
          (log/error e "<transaction-run")
          :error)))))

(defn ->neo4j-client
  "Return component for neo4j client"
  []
  (map->Neo4JClient {}))