(ns com.edocu.help.server.nginx
  (:require [com.stuartsierra.component :as component]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor.chain :as interceptor.chain]
            [nginx.clojure.embed :as nginx-server]
            [nginx.clojure.core :as nginx]
            [clojure.core.async :as async :refer [<! put! chan go close!]]
            [com.edocu.help.handlers.protocols :as routes]))

(defprotocol IServiceMap
  (service-map [this] "Return service map for pedestal server"))

(defrecord ServiceMap [service_map routes]
  component/Lifecycle
  (start [this]
    this)
  (stop [this]
    this)

  IServiceMap
  (service-map [this]
    (assoc service_map
      ::http/routes
      (routes/routes routes))))

(defn ->service-map [service_map]
  (map->ServiceMap {:service_map service_map}))

(defn nginx-provider
  [service-map]
  (assoc service-map ::handler (fn [{:keys [uri] :as request}]
                                 (.prefetchAll request)
                                 (let [downstream (nginx/hijack! request false)
                                       ctx (nginx/get-context downstream)]
                                   (clojure.pprint/pprint ctx)
                                   (go
                                     (let [response-channel (chan)
                                           {:keys [response] :as result}
                                           (interceptor.chain/execute
                                             {:request          (assoc request
                                                                  :path-info uri
                                                                  :async-supported? true)
                                              :response-channel response-channel}
                                             (::http/interceptors service-map))]
                                       (nginx/send-response!
                                         downstream
                                         (or response
                                             (<! response-channel))))))
                                 {:status 200})))

(defn nginx-server
  [service-map server-opts]
  (let [handler (::handler service-map)
        {:keys [port]
         :or   {port 8180}} server-opts]
    {:server   :nginx
     :start-fn (fn []
                 (nginx-server/run-server
                   handler
                   {:port port})
                 :nginx)
     :stop-fn  (fn []
                 (nginx-server/stop-server)
                 :nginx)}))

(defn test?
  [service-map]
  (= :test (:env service-map)))

(defrecord Pedestal [pedestal-service-map
                     service]
  component/Lifecycle
  (start [this]
    (let [service_map (service-map pedestal-service-map)]
      (if service
        this
        (cond-> service_map
                true http/create-server
                (test? service_map) http/start
                true ((partial assoc this :service))))))
  (stop [this]
    (when (and service (not (test? (service-map pedestal-service-map))))
      (http/stop service))
    (assoc this :service nil)))

(defn ->pedestal
  []
  (map->Pedestal {}))
