(ns com.edocu.help.server.nginx
  (:require [com.stuartsierra.component :as component]
            [io.pedestal.http :as http]
            [io.pedestal.interceptor.chain :as interceptor.chain]
            [io.pedestal.interceptor.helpers :as interceptors]
            [nginx.clojure.embed :as nginx-server]
            [nginx.clojure.core :as nginx]
            [clojure.core.async :as async :refer [<! put! chan go close!]]
            [com.edocu.help.handlers.protocols :as routes]
            [io.pedestal.log :as log]
            [clojure.string :as string]
            [ring.util.response :as ring-response]))

(defn- add-prepare-response
  [interceptors]
  (conj
    interceptors
    (interceptors/after
      ::prepare-response
      (fn [{:keys [request response] :as context}]
        (let [response-headers (:headers response)
              start-time (::start-time context)
              request-time (- (System/currentTimeMillis) start-time)]
          (log/info :msg (format "%s %s take: %dms"
                                 (string/upper-case (name (:request-method request)))
                                 (:uri request)
                                 request-time))
          (log/meter ::response)
          (assoc context
            :response (assoc (into {} response)
                                     :headers (into {} response-headers))
            ::request-time request-time))))))

(defn- add-nginx-response
  [interceptors]
  (cons
    (interceptors/after
      ::nginx-response
      (fn [{:keys [response response-channel] :as context
            :or {response (ring-response/response "")}}]
        (put! response-channel response (fn [_] (close! response-channel)))
        context))
    interceptors))

(defprotocol IServiceMap
  (service-map [this] "Return service map for pedestal server"))

(defrecord ServiceMap [service_map routes]
  component/Lifecycle
  (start [this]
    this)
  (stop [this]
    this)

  IServiceMap
  (service-map [this]
    (let [svr-mp (assoc service_map ::http/routes (routes/routes routes))
          interceptors (-> (:io.pedestal.http/interceptors (http/default-interceptors svr-mp))
                           add-prepare-response
                           add-nginx-response)
          result (assoc svr-mp
                   ::http/interceptors interceptors)]
      result)))

(defn ->service-map [service_map]
  (map->ServiceMap {:service_map service_map}))

(defn nginx-provider
  [service-map]
  (assoc service-map ::handler (fn [{:keys [uri] :as request}]
                                 (.prefetchAll request)
                                 (let [start-time (System/currentTimeMillis)
                                       downstream (nginx/hijack! request false)
                                       req (assoc (into {} request)
                                             :path-info uri
                                             :body (if-let [body (:body request)]
                                                     (slurp body))
                                             :async-supported? true
                                             :headers (into {} (:headers request)))]
                                   (go
                                     (let [response-channel (chan)
                                           {:keys [response] :as result}
                                           (interceptor.chain/execute
                                             {:request          req
                                              :response-channel response-channel
                                              ::start-time start-time}
                                             (::http/interceptors service-map))]
                                       (nginx/send-response!
                                         downstream
                                         (or response
                                             (<! response-channel))))))
                                 {:status 200})))

(defn nginx-server
  [service-map server-opts]
  (let [handler (::handler service-map)
        {:keys [port]
         :or   {port 8180}} server-opts]
    {:server   :nginx
     :start-fn (fn []
                 (nginx-server/run-server
                   handler
                   {:port port})
                 :nginx)
     :stop-fn  (fn []
                 (nginx-server/stop-server)
                 :nginx)}))

(defn test?
  [service-map]
  (= :test (:env service-map)))

(defrecord Pedestal [pedestal-service-map
                     service]
  component/Lifecycle
  (start [this]
    (let [service_map (service-map pedestal-service-map)]
      (if service
        this
        (cond-> service_map
                true http/create-server
                (test? service_map) http/start
                true ((partial assoc this :service))))))
  (stop [this]
    (when (and service (not (test? (service-map pedestal-service-map))))
      (http/stop service))
    (assoc this :service nil)))

(defn ->pedestal
  []
  (map->Pedestal {}))
