
(ns ^{:author "Eduardo Julian",
      :doc "Core functionality."}
     eejp-web-dev0.core
  (:refer-clojure :exclude [load])
  (:use clojure.template)
  (:require [clj-orient.core :as oc]
            [clj-orient.graph :as og]
            [clj-orient.query :as oq]))

; [Constants]
; Visibility Levels
(def +private+ 0) ; only me and tagged users
(def +protected+ 1) ; me and friends
(def +public+ 2)
(def +anon+ 3) ; public & anonymous.

; [Global Vars]
(def ^:dynamic ^{:doc "Path to the configurations file."} *conf-path* "system.conf")
(def ^:dynamic ^{:doc "Hash-map with all the system configuration options."} *system-conf* {})

; [Utils]
(defmacro deffetch "Defines a document-fetching function."
  [sym kcluster]
  `(defn ~sym "" [item-id#]
     (if (integer? item-id#)
       (oc/load-item [(oc/get-cluster-id ~kcluster) item-id#])
       item-id#)))

(defn set-var-root! "" [var val] (alter-var-root var (fn [_] val)))

(defn get-db "" [] (og/open-graph-db! (:db-url *system-conf*) (:db-user *system-conf*) (:db-pass *system-conf*)))

(defn elog "Logs something by throwing an exception."
  [& strs] (throw (Exception. (apply str strs))))

(defn real-id "Returns a document's Id without the cluster part."
  [d] (-> d oc/get-id oc/id->vec second))

(defn ^:private ^{:doc "Logs the system configuration file."} load-system-conf! []
  (set-var-root! #'*system-conf* (read-string (slurp *conf-path*))))

(defn db-load [kcluss id] (oc/load-item (oc/get-cluster-id kcluss) id))

; Data-type Declaration
(defn- declare-records! [group]
  (oc/with-db (get-db)
    (doseq [t group]
      (try (oc/create-class! t)
        (catch com.orientechnologies.orient.core.exception.OSchemaException e nil)))))

(do-template [sym parent]
  (defn- sym [group]
    (oc/with-db (get-db)
      (doseq [t group]
        (try (oc/create-class! t parent)
          (catch com.orientechnologies.orient.core.exception.OSchemaException e nil)))))
  declare-vertices! :OGraphVertex
  declare-edges! :OGraphEdge)

(defn declare-data-types! "Declares all data-types to be used by the application."
  [{:keys [records, vertices, edges]}]
  (declare-records! records)
  (declare-vertices! vertices)
  (declare-edges! edges))

; Enhancements to clj-orient
(defn passoc!+ "Assocs only if 'v' is not null."
  ([d k v] (if v (oc/passoc! d k v) d))
  ([d k v & kvs] (reduce #(passoc!+ %1 (first %2) (second %2)) d (partition 2 (list* k v kvs)))))

; Execute some basic stuff:
(load-system-conf!)
