
(ns ^{:author "Eduardo Julian",
      :doc "Common functionality for building web pages."}
     eejp-web-dev0.html
  (:use noir.core
        (hiccup page-helpers form-helpers))
  (:require [noir.session :as session]
            [noir.response :as response]
            [noir.statuses :as wstatus]
            [noir.cookies :as cookies]
            [noir.request :as request]
            [clj-orient.core :as oc]
            [clj-orient.graph :as og]
            [clj-orient.query :as oq]
            [eejp-web-dev0.external.recaptcha :as $recaptcha]
            [eejp-web-dev0.external.email :as $email]
            [eejp-web-dev0.core :as $]
            [eejp-web-dev0.users :as $users]
            [eejp-web-dev0.comments :as $comments]
            [eejp-web-dev0.feedback :as $feedback]
            [eejp-web-dev0.messaging :as $msgs]
            [eejp-web-dev0.voting :as $vote]
            [eejp-web-dev0.logging :as $log]
            [eejp-web-dev0.files :as $files]
            [eejp-web-dev0.utils :as $utils]))

(declare user-header)

; [Globals]
(def ^:dynamic ^{:doc ""} *head* nil)
(def ^:dynamic ^{:doc ""} *footer* nil)
(def ^:dynamic ^{:doc ""} *faq* nil)

; [Constants]
(def ^{:doc ""} +css-path+ "/static/css/app.css")
(def ^{:doc ""} +js-path+ "/static/js/main.js")

(def +page-size+ 10)

(def +nationalities+
  (list ["un" "Nothing Selected"] ; Base, unselected state.
        ; A
        ["af" "Afghanistan"] ["al" "Albania"] ["ag" "Algeria"] ["an" "Andorra"] ["ao" "Angola"] ["ac" "Antigua and Barbuda"]
        ["ar" "Argentina"] ["am" "Armenia"] ["as" "Australia"] ["au" "Austria"] ["aj" "Azerbaijan"]
        ; B
        ["bf" "Bahamas, The"] ["ba" "Bahrain"] ["bg" "Bangladesh"] ["bb" "Barbados"] ["bo" "Belarus"] ["be" "Belgium"] ["bh" "Belize"]
        ["bn" "Benin"] ["bt" "Bhutan"] ["bl" "Bolivia"] ["bk" "Bosnia and Herzegovina"] ["bc" "Botswana"] ["br" "Brazil"] ["bx" "Brunei"]
        ["bu" "Bulgaria"] ["uv" "Burkina Faso"] ["bm" "Burma"] ["by" "Burundi"]
        ; C
        ["cb" "Cambodia"] ["cm" "Cameroon"] ["ca" "Canada"] ["ct" "Central African Republic"] ["cd" "Chad"] ["ci" "Chile"] ["ch" "China"]
        ["co" "Colombia"] ["cn" "Comoros"] ["cg" "Congo, Democratic Republic of the"] ["cf" "Congo, Republic of the"] ["cs" "Costa Rica"]
        ["iv" "Cote d'Ivoire"] ["hr" "Croatia"] ["cu" "Cuba"] ["cy" "Cyprus"] ["ez" "Czech Republic"]
        ; D
        ["da" "Denmark"] ["dj" "Djibouti"] ["do" "Dominica"] ["dr" "Dominican Republic"]
        ; E
        ["ec" "Ecuador"] ["eg" "Egypt"] ["es" "El Salvador"] ["ek" "Equatorial Guinea"] ["er" "Eritrea"] ["en" "Estonia"] ["et" "Ethiopia"]
        ; F
        ["fj" "Fiji"] ["fi" "Finland"] ["fr" "France"]
        ; G
        ["gb" "Gabon"] ["ga" "Gambia, The"] ["gg" "Georgia"] ["gm" "Germany"] ["gh" "Ghana"] ["gr" "Greece"] ["gt" "Guatemala"]
        ["gv" "Guinea"] ["pu" "Guinea-Bissau"] ["gy" "Guyana"]
        ; H
        ["ha" "Haiti"] ["ho" "Honduras"] ["hk" "Hong Kong"] ["hu" "Hungary"]
        ; I
        ["ic" "Iceland"] ["in" "India"] ["id" "Indonesia"] ["ir" "Iran"] ["iz" "Iraq"] ["ei" "Ireland"] ["is" "Israel"] ["it" "Italy"]
        ; J
        ["jm" "Jamaica"] ["ja" "Japan"] ["jo" "Jordan"]
        ; K
        ["kz" "Kazakhstan"] ["ke" "Kenya"] ["kr" "Kiribati"] ["kn" "Korea, North"] ["ks" "Korea, South"] ["ku" "Kuwait"] ["kg" "Kyrgyzstan"]
        ; L
        ["la" "Laos"] ["lg" "Latvia"] ["le" "Lebanon"] ["li" "Liberia"] ["ly" "Libya"] ["ls" "Liechtenstein"] ["lh" "Lithuania"]
        ["lu" "Luxembourg"]
        ; M
        ["mc" "Macau"] ["mk" "Macedonia"] ["ma" "Madagascar"] ["mi" "Malawi"] ["my" "Malaysia"] ["mv" "Maldives"] ["ml" "Mali"]
        ["mt" "Malta"] ["mr" "Mauritania"] ["mp" "Mauritius"] ["mx" "Mexico"] ["fm" "Micronesia, Federated States of"] ["md" "Moldova"]
        ["mn" "Monaco"] ["mg" "Mongolia"] ["mo" "Morocco"] ["mz" "Mozambique"]
        ; N
        ["wa" "Namibia"] ["nr" "Nauru"] ["np" "Nepal"] ["nl" "Netherlands"] ["nt" "Netherlands Antilles"] ["nz" "New Zealand"]
        ["nu" "Nicaragua"] ["ng" "Niger"] ["ni" "Nigeria"] ["no" "Norway"]
        ; O
        ["mu" "Oman"]
        ; P
        ["pk" "Pakistan"] ["pm" "Panama"] ["pp" "Papua New Guinea"] ["pa" "Paraguay"] ["pe" "Peru"] ["rp" "Philippines"] ["pl" "Poland"]
        ["po" "Portugal"] ["rq" "Puerto Rico"]
        ; Q
        ["qa" "Qatar"]
        ; R
        ["ro" "Romania"] ["rs" "Russia"] ["rw" "Rwanda"]
        ; S
        ["sc" "Saint Kitts and Nevis"] ["st" "Saint Lucia"] ["vc" "Saint Vincent and the Grenadines"] ["ws" "Samoa"]
        ["tp" "Sao Tome and Principe"] ["sa" "Saudi Arabia"] ["sg" "Senegal"] ["se" "Seychelles"] ["sl" "Sierra Leone"] ["sn" "Singapore"]
        ["lo" "Slovakia"] ["si" "Slovenia"] ["bp" "Solomon Islands"] ["so" "Somalia"] ["sf" "South Africa"] ["sp" "Spain"] ["ce" "Sri Lanka"]
        ["su" "Sudan"] ["ns" "Suriname"] ["sw" "Sweden"] ["sz" "Switzerland"] ["sy" "Syria"]
        ; T
        ["tw" "Taiwan"] ["ti" "Tajikistan"] ["tz" "Tanzania"] ["th" "Thailand"] ["to" "Togo"] ["tn" "Tonga"] ["td" "Trinidad and Tobago"]
        ["ts" "Tunisia"] ["tu" "Turkey"] ["tx" "Turkmenistan"] ["tv" "Tuvalu"]
        ; U
        ["ug" "Uganda"] ["up" "Ukraine"] ["tc" "United Arab Emirates"] ["uk" "United Kingdom"] ["us" "United States"] ["uy" "Uruguay"]
        ["uz" "Uzbekistan"]
        ; V
        ["nh" "Vanuatu"] ["ve" "Venezuela"] ["vm" "Vietnam"]
        ; Y
        ["ym" "Yemen"] ["yi" "Yugoslavia"]
        ; Z
        ["za" "Zambia"] ["zi" "Zimbabwe"]))
(def +nats-map+ (reduce {} conj +nationalities+))
(def ^{:doc "A 'select' widget with nationalities."} +nationalities+
  [:select {:name :nationality, :required true}
   (for [[code name] +nationalities+]
     [:option {:value code} name])])

(def +talk-types+
  {:comment "Comment on the Service",
   :question "Ask us a Question",
   :complaint "Complain about a Problem",
   :request "Request New Functionality",
   :copyright "Denounce copyright infringement by an user."})

(def ^{:doc "Default login widget."} +login+
  [:section#login (form-to [:post (str ($/*system-conf* :ssl-url) "/login")]
                     [:label "E-mail: "] [:input {:type :email, :name :email, :placeholder "john.doe@example.com", :required true}]
                     [:label "Password: "] [:input {:type :password, :name :password, :placeholder "************", :required true}]
                     [:label "Remember me?"] [:input {:type :checkbox, :name :remember-me}]
                     [:a {:href "/forgot-pass"} "Forgot password?"]
                     [:input {:type :submit, :value "Sign In"}])
   [:p "Don't have an account? We can solve that! Click " [:a {:href (str ($/*system-conf* :ssl-url) "/signup")} "here"] "."]])

(def +search-bar+
  [:form#search-bar {:method :GET, :action "/search"}
   [:input {:type :text, :name :q}]
   [:input {:type :submit, :value "Search"}]])

; [Customized Chunks]
(defn setup-head [dependencies]
  ($/set-var-root! #'*head*
    [:head [:title ($/*system-conf* :title)]
     (include-css (str ($/*system-conf* :home-url) +css-path+))
     (include-js (str ($/*system-conf* :home-url) +js-path+))
     dependencies]))

(defn setup-footer [paths]
  ($/set-var-root! #'*footer*
    [:footer
     [:section#footer-paths
      (interpose [:span.vsep] (map (fn [[p d]] [:a {:href p} d]) paths))]
     [:p#copyright (str "Copyright © " ($/*system-conf* :copyright-owner) " " (.get (java.util.Calendar/getInstance) java.util.Calendar/YEAR) ". All Rights Reserved.")]]))

(defn setup-faq [questions]
  ($/set-var-root! #'*faq*
    [:section#faq
     (for [[q a] questions]
       [:details.faq-question [:summary q] [:div a]])]))

(def +logo+ [:div#logo [:a {:href "/"} [:img {:src (str ($/*system-conf* :home-url) ($/*system-conf* :logo-path))}]]])

; [Utils]
(defn current-user [] ($users/fetch-user (session/get :user-id)))

(defn cookie-login! "Log-in using the cookie values stored on the user's computer."
  []
  (if (not (empty? (cookies/get :email)))
    (render [:post "/login"] {:email (cookies/get :email), :password (cookies/get :password), :remember-me true})))

(defn add-flashes!
  "Adds all the flash notifications as hidden .flash-notif input fields.
The JS part will then query them and do as needed."
  [] (for [[k v] (merge {:user-id (session/get :user-id)}
                        (session/flash-get))]
       [:input.flash-notif {:type :hidden, :name k, :value v}]))

(defpartial base-layout [& contents]
  (do (cookie-login!) nil)
  *head*
  [:body
   (add-flashes!)
   (user-header (session/get :user-name))
   [:section#content
    +logo+
    contents]
   *footer*])
(def layout base-layout)
(defn set-layout "Function for setting the global layout fn."
  [f]
  (let [lay (comp base-layout f)]
    ($/set-var-root! #'layout lay)
    lay))

(defmacro require-login "Only present the page if the user is logged in. Else, return to main page."
  [& body]
  `(if (session/get :user-id)
     (do ~@body)
     (do (session/flash-put! {:error :require-login})
       (response/redirect "/"))))

(defn logged-in? "" [] (session/get :user-id))

(defn- verification-url "" [email code] (str ($/*system-conf* :home-url) "/verify?email=" email "&code=" code))

(def +expire-week+ (* 60 60 24 7))
(defn expire-val "Set expiration for cookie value."
  [val secs] {:path "/", :value val, :expires secs})

(defmacro defpage+
  "Enhanced defpage with automatic map destructuring, data transformation from string to primitive value,
exception logging & freshly allocated *db* instance."
  [route args & body]
  (let [tagged (filter #(-> % meta :tag (= 'data)) args)
        tlet `(let [~@(mapcat (fn [t] (let [t (with-meta t nil)] `[~t (if (not (empty? ~t)) (read-string ~t))])) tagged)]
                ~@body)
        body (if (empty? tagged) body (list tlet))]
    `(defpage ~route {:keys ~(vec (map #(with-meta % nil) args))}
       (oc/with-db ($/get-db)
         (try ~@body
           (catch ~'Exception e# ($log/log e#) (throw e#)))))))

; [Partials]
(defpartial table-layout
  [props row-size & items]
  (let [items (mapcat (fn [x] (if (seq? x) x (list x))) items)
        unevens (take-last (rem (count items) row-size) items)]
    [:table props
     (map (fn [r] [:tr (map (fn [i] [:td i]) r)])
          (concat (partition row-size items) (list unevens)))]))

(defpartial table-form [props row-size m-a & items] (form-to m-a (apply table-layout props row-size items)))

(defpartial t-row [& items] [:tr (map (fn [i] [:td i]) items)])

(defpartial menu
  [menu-id mdesc]
  [:nav {:id menu-id}
   [:ul
    (for [[label data] mdesc]
      [:li (if (string? data)
             [:a {:href data} label]
             (list [:a {:href "#"} label]
                   (menu nil data)))])]])

(defpartial prev-next
  [base-url nres page]
  [:div#prev-next
   (if (> page 1) [:a#prev {:href (str base-url "page=" (dec page))} "Previous"])
   (if (> (- nres (* +page-size+ page)) 0)
     [:a#next {:href (str base-url "page=" (inc page))} "Next"])])

(defpartial user-header
  [user-name]
  [:header
   (if user-name
     [:section#user-tab
      [:p user-name]
      [:div#user-options
       [:a {:href "/profile"} "View Profile"]
       [:a {:href "/logout"} "Log out"]]]
     +login+)])

(defpartial render-message [{:keys [title text date other-id other-name mode msg-id]}]
  [:details [:summary (str date ": " title)]
   [:p "Sender: " (if (= mode :received) other-name (session/get :user-name))]
   [:p "Receiver: " (if (= mode :received) (session/get :user-name) other-name)]
   [:textarea {:value text, :readonly true, :wrap :soft, :cols 80, :rows 4}]
   [:a {:href (str "/send-message/" other-id "?reply-for=" msg-id)} "Reply Message"]])

(defpartial comment-box [kcluss id]
  (if (logged-in?)
    [:div.comment-box {:_kcluss (str kcluss), :_id id}
     [:textarea {:name :text, :cols 50, :rows 4}]
     [:a {:href "#"} "Post Comment"]]))

(defpartial comments-section [item kcluss id]
  [:section.comments [:h2 "Comments"]
   (comment-box kcluss id)
   (for [item ($comments/get-comments item)
         :let [id ($/real-id item)
               {:keys [first-name last-name]} (oc/doc->map (og/get-vertex item :out))
               {:keys [text date]} (oc/doc->map item)]]
     [:article.comment
      (str "By: " first-name " " last-name)
      (str "On: " date)
      text])])

(defpartial votes-section [item kcluss id]
  (let [[ups downs] ($vote/get-votes item)]
    [:div.votes {:_kcluss (str kcluss), :_id id}
     [:p "Up Votes: " [:span.votes-up ups]] [:p "Down Votes: " [:span.votes-down downs]]
     [:p "Total Votes: " [:span.votes-total (+ ups downs)]] [:p "Ratio: " [:span.votes-ratio (if (= ups downs 0) 0 (double (/ ups (+ ups downs))))]]
     [:a.up-vote-btn {:href "#"} "Up Vote"] [:a.down-vote-btn {:href "#"} "Down Vote"]]))

(defpartial upload-form [kcluss id accept]
  [:form.upload-form {:method :POST, :enctype "multipart/form-data", :action "/upload-file",
                      :_kcluss (str kcluss), :_id id}
   [:input.file-field {:name :file, :type :file, :size 30, :accept accept}]
   [:div.file-name "Name: "]
   [:div.file-mime "Type: "]
   [:div.file-size "Size: "]
   [:input.upload-btn {:type :button, :value "Upload"}]
   [:progress {:value "0%", :max 100} "0%"]])

; Server-side logic
(defpage+ [:post "/signup"] [email password repassword user-name first-name last-name birthday nationality
                             recaptcha_challenge_field, recaptcha_response_field _user-ip]
  (if (true? ($recaptcha/verify-captcha _user-ip recaptcha_challenge_field recaptcha_response_field))
    (if (= password repassword)
      (when-let [user ($users/sign-up email password user-name first-name last-name birthday nationality)]
        ($email/send-verification-email email (verification-url email (oc/pget user :verification-code)))
        (session/flash-put! {:notif :verif-email-sent})
        (response/redirect "/"))
      (do (session/flash-put! {:error :pass-repass})
        (response/redirect "/pass-repass")))
    (do (session/flash-put! {:error :captcha-error})
      (response/redirect "/captcha-error"))
    ))

(defpage+ [:post "/login"] [email password remember-me]
  (if-let [user ($users/login email password)]
    (do (session/put! :user-id (-> user oc/get-id oc/id->vec second))
      (session/put! :user-name (oc/pget user :user-name))
      (when remember-me
        (cookies/put! :email (expire-val email +expire-week+))
        (cookies/put! :password (expire-val password +expire-week+))))
    (session/flash-put! {:error :login-error}))
  (response/redirect "/"))

(defpage+ [:get "/verify"] [email code]
  (if-let [user ($users/verify-user email code)]
    (session/flash-put! {:notif :verif-success})
    (session/flash-put! {:notif :verif-error}))
  (response/redirect "/"))

(defpage+ [:get "/logout"] []
  (session/clear!)
  (cookies/put! :email "")
  (cookies/put! :password "")
  (response/redirect "/"))

(defpage+ [:post "/change-pass"] [email password repassword]
  (if (= password repassword)
    (do ($users/change-password email password)
      (session/flash-put! {:notif :pass-changed})
      (response/redirect "/"))
    (do (session/flash-put! {:error :pass-repass})
      (response/redirect "/change-pass"))))

(defpage+ [:post "/forgot-pass"] [email]
  (if-let [code ($users/password-change-code email)]
    (do ($email/send-password-change-email email (str ($/*system-conf* :home-url) "/change-pass?email=" email "&change-code=" code))
      (session/flash-put! {:notif :password-change-email-sent}))
    (session/flash-put! {:error :no-user-email}))
  (response/redirect "/"))

(defpage+ [:post "/feedback/:type"] [type text]
  ($feedback/store-feedback type (session/get :user-id) text)
  (response/redirect "/feedback/thankyou"))

(defpage+ [:post "/send-message/:receiver-id"] [receiver-id msg-title msg]
  ($msgs/send-message (session/get :user-id) receiver-id msg-title msg)
  (session/flash-put! {:notif :message-sent})
  (response/redirect (str "/send-message/" receiver-id)))

(defpage+ [:post "/upload-file"] [file, ^data kcluss, ^data id]
  (spit "file-log" (str file " " kcluss " " id "\n"
                        (type file) " " (type kcluss) " " (type id)))
  (let [user ($users/fetch-user (session/get :user-id))
        target ($/db-load kcluss id)
        {:keys [filename content-type size tempfile]} file
        file ($files/create-file filename content-type size
                                 (-> tempfile java.io.FileInputStream. $utils/read-input-stream))]
    ($files/associate-media target file)
    (oc/save! (og/link! user $files/+UPLOADS+ file))
    ""))

; HTML Structure
(defpage+ "/signup" []
  (layout
    (table-form nil 2 [:post "/signup"]
      [:label "E-mail: "] [:input {:type :email, :name :email, :placeholder "john.doe@example.com", :required true}]
      [:label "Password: "] [:input {:type :password, :name :password, :placeholder "************", :required true,
                                     :pattern ".{6,20}", :title "The password must have between 6 and 20 characters."}]
      [:label "Retype Password: "] [:input {:type :password, :name :repassword, :placeholder "************", :required true,
                                            :pattern ".{6,20}", :title "The password must have between 6 and 20 characters."}]
      [:label "Username: "] [:input {:type :text, :name :user-name, :placeholder "JohnnyD", :required true}]
      [:label "First Name: "] [:input {:type :text, :name :first-name, :placeholder "John", :required true}]
      [:label "Last Name: "] [:input {:type :text, :name :last-name, :placeholder "Doe", :required true}]
      [:label "Birthday: "] [:input {:type :date, :name :birthday, :required true}]
      [:label "Nationality: "] +nationalities+
      [:br] $recaptcha/+widget+
      [:br] [:input {:type :submit, :value "Sign Up!"}]
      )))

(defpage+ "/forgot-pass" []
  (layout
    [:p "We'll send you a special code to your e-mail so you can change your password."]
    (table-form nil 2 [:post "/forgot-pass"]
      [:label "E-mail: "] [:input {:type :email, :name :email, :placeholder "john.doe@example.com", :required true}]
      [:input {:type :submit, :value "Send Code"}])
    ))

(defpage+ "/change-pass" [email change-code]
  (if ($users/allow-password-change? email change-code)
    (layout [:p "Please write the new password you want."]
      (table-form nil 2 [:post "/change-pass"]
        [:p] [:input {:type :hidden, :name :email, :value email, :required true}]
        [:label "Password:"] [:input {:type :password, :name :password, :placeholder "********", :required true}]
        [:label "Retype Password:"] [:input {:type :password, :name :repassword, :placeholder "********", :required true}]
        [:input {:type :submit, :value "Send Code"}]))
    (do (session/flash-put! {:error :invalid-code})
      (response/redirect "/"))))

(defpage+ "/messages/:mode" [mode page]
  (layout
    (let [[res nres] ($msgs/get-messages (session/get :user-id) (keyword mode))]
      (concat (map render-message res)
              [(prev-next (str "/messages/" mode "?") nres page)])
      )))

(defpage+ "/send-message/:receiver-id" [^data receiver-id, ^data reply-for]
  (require-login
    (if (not= receiver-id (session/get :user-id))
      (oc/with-db ($/get-db)
        (layout
          (form-to [:post "/send-message/:receiver-id"]
            [:label "Title:"]
            [:input {:type :text, :name :msg-title, :required true,
                     :value (if reply-for (-> reply-for $msgs/fetch-msg (oc/pget :title) (->> (str "Re: "))))}]
            [:label "Message:"]
            [:textarea {:name :msg, :required true, :wrap :soft, :rows 4, :cols 80, :autofocus true}]
            [:input {:type :submit, :value "Send Message"}])))
      (do (session/flash-put! {:error :message-oneself})
        (response/redirect "/")))))

(defpage+ [:get ["/feedback/:type" :talk-type #"(comment|question|complaint|request|copyright)"]] [type]
  (layout
    (form-to [:post (str "/feedback/" type)]
      [:label "Message:"]
      [:textarea {:name :text, :required true, :wrap :soft, :rows 4, :cols 80, :autofocus true}]
      [:input {:type :submit, :value "Send Message"}])))

(defpage+ "/feedback/thankyou" []
  (layout
    [:p "Thanks for your feedback!"]
    [:p "We will take it into account while developing and improving the site :-)"]))

(defpage+ "/faq" [] (layout *faq*))

(defpage+ [:post "/comment"] [^data kcluss, ^data id, return-url, text]
  ($comments/post-comment (current-user) text (oc/load-item [(oc/get-cluster-id kcluss) id]))
  (response/redirect return-url))

; For serving user profile images.
(defpage+ "/user/:uid/img" [^data uid]
  (if-let [img (oc/pget ($users/fetch-user uid) :user-img)]
    (response/content-type (oc/pget img :content-type) (-> img (oc/pget :data) java.io.ByteArrayInputStream.))
    (response/redirect ($/*system-conf* :no-user-img))))

; For serving the static resources... Small hack to work around the fact that Noir, for some weird-ass reason, cannot find my static files... T_T
(defpage+ [:get ["/static/:path" :path #".+"]] [path]
  (let [f (java.io.File. (str "static/" path))]
    (if (.exists f) f)))

; For the different kinds of weird status pages...
(defn setup-status-pages! "Status pages must be passed as no-args fns."
  [{:keys [s404]}]
  (wstatus/set-page! 404 (s404))
  ; (wstatus/set-page! 500 (layout [:p "There was an internal error. Please try again or go to another part of the web site."]))
  ; I keep it commented because for now because I want to keep seeing the stacktraces for debugging.
  )

; Make sure signup & login are secure
(pre-route [:get "/signup"] []
  (let [req (request/ring-request)]
    (if (or (not= (req :scheme) :https) (not= (req :server-port) 8443))
      (response/redirect (str "https://" (:server-name req) ":8443" (:uri req))))))
(pre-route [:get "/login"] []
  (let [req (request/ring-request)]
    (if (or (not= (req :scheme) :https) (not= (req :server-port) 8443))
      (response/redirect (str "https://" (:server-name req) ":8443" (:uri req))))))
