(ns elasticsearch.document
  (:refer-clojure :exclude [get type update])
  (:require [elasticsearch.connection :as conn]
            [elasticsearch.schema :as s]
            [slingshot.slingshot :refer [throw+]]))

(defn make-uri [{:keys [index type id action]}]
  (->> [index type id action]
       (filter identity)
       (map #(java.net.URLEncoder/encode %))
       (clojure.string/join "/")
       (format "/%s")))

(defn request
  [conn method {:keys [index type id body] :as req} & [opts]]
  (conn/request conn method (merge-with merge
                                        (dissoc req :index :type :id)
                                        {:uri (make-uri req)}
                                        opts)))

(defn index
  "Index a document into cluster/index/type/id"
  [conn req]
  (s/validate s/index-schema req)
  ;; use POST since it will handle absent and present _id
  (request conn :post req))

(defn create
  "Create a document into cluster/index/type/id, will fail if document
  already exists"
  [conn req]
  (s/validate s/create-schema req)
  (request conn :put (assoc req :action "_create")))

(defn get
  "GET a document from a cluster/index/type/id"
  [conn req]
  (s/validate s/get-schema req)
  (request conn :get req))

(defn get-source
  "GET a document, source-only"
  [conn req]
  (s/validate s/get-schema req)
  (request conn :get (assoc req :action "_source")))

(defn update
  "Update a document"
  [conn req]
  (s/validate s/update-schema req)
  (request conn :post (assoc req :action "_update")))

(defn search
  [conn req]
  (s/validate s/search-schema req)
  (request conn :post (assoc req :action "_search")))

(defn bulk
  [conn req]
  (s/validate s/bulk-schema req)
  (let [res (request conn :post (assoc req :action "_bulk"))]
    (when (:errors res)
      ;; A response that contains an error looks like:
      ;;
      ;; {:took 17,
      ;;  :errors true,
      ;;  :items
      ;;  [{:delete
      ;;    {:_index "FOO",
      ;;     :_type "FAKE",
      ;;     :_id "NO",
      ;;     :status 400,
      ;;     :error
      ;;     {:type "invalid_index_name_exception",
      ;;      :reason "Invalid index name [FOO], must be lowercase",
      ;;      :index "FOO"}}}]}
      (let [has-error? (fn [action]
                         (:error ((comp first vals) action)))
            actions-with-errors (filter has-error? (:items res))]
        (throw+ {:type ::bulk-error
                 :count (count actions-with-errors)
                 :items actions-with-errors})))
    res))
