(ns elasticsearch.schema
  (:require [schema.core :as s]))

(def ^{:dynamic true} *validation-enabled*
  "Set to `true' to validate requests, defaults to true."
  true)

(defn validate
  "If validation is enabled, validate the object, otherwise return it"
  [schema obj]
  (if *validation-enabled*
    (s/validate schema obj)
    obj))

(def default-schema
  "Base schema"
  {
   (s/optional-key :query-params) (s/pred map?)

   ;; :conn-params for extra things that may not make sense for the
   ;; connection's entire lifecycle, but needed for a single request,
   ;; eg, {:throw-exceptions? false}.
   (s/optional-key :conn-params) (s/pred map?)
   })

(def default-doc-schema
  "Base schema for docs"
  (merge
   default-schema
   {:index s/Str
    :type s/Str}))

(def index-schema
  "Schema for a document being indexed"
  (merge
   default-doc-schema
   {
    ;; id is optional since ES will auto-generate
    (s/optional-key :id) s/Str
    :body (s/pred map?)
    }))

(def create-schema
  "Schema for a document being created (requires ID)"
  (merge
   default-doc-schema
   {:id s/Str
    :body (s/pred map?)}))

(def get-schema
  "Schema for the get-by-id API"
  (merge
   default-doc-schema
   {:id s/Str}))

(def update-schema
  "Schema for the update API"
  (merge
   default-doc-schema
   {:id s/Str
    :body {(s/optional-key :doc) (s/pred map?)
           (s/optional-key :ids) (s/pred sequential?)
           (s/optional-key :script) s/Str
           (s/optional-key :params) (s/pred map?)
           (s/optional-key :upsert) (s/pred map?)
           (s/optional-key :scripted_id) s/Str
           (s/optional-key :scripted_upsert) s/Bool}}))

(def search-schema
  (merge
   default-schema
   {(s/optional-key :index) s/Str
    (s/optional-key :type) s/Str
    (s/optional-key :body) (s/pred map?)}))

(def bulk-schema
  (merge
   default-schema
   {(s/optional-key :index) s/Str
    (s/optional-key :type) s/Str
    ;; TODO handle optionals like :delete musn't have a :source,
    ;; but the others must
    :body [{(s/enum :index :create :update :delete) (s/pred map?)}]}))
