(ns elasticsearch.indices
  (:refer-clojure :exclude [get delete flush ensure count])
  (:require
   [clojure.spec :as s]
   [elasticsearch.cluster :as cluster]
   [elasticsearch.common :refer [format-uri request]]
   [elasticsearch.connection.http :as hconn]
   [elasticsearch.connection :as conn]
   [slingshot.slingshot :refer [try+]]))

(def index-pattern
  #"[-a-z0-9._]+")

(s/def ::name
  (s/and string?
         #(re-matches index-pattern %)))

(s/def ::keyword
  (s/and keyword?
         #(re-matches index-pattern (name %))))

(s/def ::settings
  map?)

(s/def ::mappings
  map?)

(s/def ::metadata
  (s/keys :opt-un [::settings ::mappings]))

(defn create
  ([conn index]
   (create conn index {}))
  ([conn index req]
   (request conn :put (format-uri index) req)))

(defn ensure [& args]
  (try+
   (apply create args)
   (catch [:status 400] e
     (if (= (-> e :body :error :type)
            "index_already_exists_exception")
       true
       (throw e)))))

(defn exists
  ([conn idx]
   (exists conn idx {}))
  ([conn idx req]
   (try+
    (request conn :head (format-uri idx) req)
    true
    (catch [:status 404] _
      false))))

(defn exists-type
  ([conn idx type]
   (exists-type conn idx type {}))
  ([conn idx type req]
   (try+
    (request conn :head (format-uri idx "_mapping" type) req)
    true
    (catch [:status 404] _
      false))))

(defn exists-template
  ([conn name]
   (exists-template conn name {}))
  ([conn name req]
   (try+
    (request conn :head (format-uri "_template" name) req)
    true
    (catch [:status 404] _
      false))))

(defn exists-alias
  ([conn]
   (exists-alias conn ""))
  ([conn idx]
   (exists-alias conn idx ""))
  ([conn idx name]
   (exists-alias conn idx name {}))
  ([conn idx name req]
   (try+
    (request conn :head (format-uri idx "_alias" name) req)
    true
    (catch [:status 404] _
      false))))

(defn put-mapping
  ([conn type req]
   (put-mapping conn "" type req))
  ([conn index type req]
   (request conn :put (format-uri index type "_mappings") req)))

(defmulti get-field-mapping
  (fn [_ & args] (map? (last args))))

(defmethod get-field-mapping true
  ([conn fields req]
   (get-field-mapping conn "" fields req))
  ([conn index fields req]
   (get-field-mapping conn index "" fields req))
  ([conn index type fields req]
   (request conn :get (format-uri index "_mapping" type "field" fields) req)))

(defmethod get-field-mapping false
 ([conn fields]
  (get-field-mapping conn "" fields))
 ([conn index fields]
  (get-field-mapping conn index "" fields))
 ([conn index type fields]
  (request conn :get (format-uri index "_mapping" type "field" fields) {})))

(defn health
  ([conn index]
   (health conn index {}))
  ([conn index req]
   (cluster/health conn index req)))

(defn wait-for-health
  ([conn status index]
   (wait-for-health conn status index {}))
  ([conn status index req]
   (cluster/health conn index (merge req
                                     {:query-params
                                      {:wait_for_status (name status)}}))))

(defn count
  ([conn index]
   (count conn index "" {}))
  ([conn index type]
   (count conn index type {}))
  ([conn index type req]
   (request conn :get (format-uri index type "_count") req)))

(defmacro with-tmp-index [conn sym settings & body]
  `(let [~sym (format "tmp-%s" (->> (java.util.UUID/randomUUID)
                                    str (take 6) (apply str)))]
     (try
       (create ~conn ~sym {:body ~settings})
       (wait-for-health ~conn :yellow ~sym)
       ~@body
       (finally
         (delete ~conn ~sym)))))

;; Begin generated code. Everything above will be kept and anything below will be regenerated.

(defn open
  ([conn index] (open conn index nil))
  ([conn index req]
    (request conn :post (format-uri index "_open") req)))

(defmulti validate-query (fn [_ & args] (map? (last args))))

(defmethod
 validate-query
 true
 ([conn req] (validate-query conn nil req))
 ([conn index req] (validate-query conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_validate" "query") req)))

(defmethod
 validate-query
 false
 ([conn] (validate-query conn nil))
 ([conn index] (validate-query conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_validate" "query") {})))

(defmulti get-settings (fn [_ & args] (map? (last args))))

(defmethod
 get-settings
 true
 ([conn req] (get-settings conn nil req))
 ([conn index req] (get-settings conn index nil req))
 ([conn index name req]
  (request conn :get (format-uri index "_settings" name) req)))

(defmethod
 get-settings
 false
 ([conn] (get-settings conn nil))
 ([conn index] (get-settings conn index nil))
 ([conn index name]
  (request conn :get (format-uri index "_settings" name) {})))

(defmulti refresh (fn [_ & args] (map? (last args))))

(defmethod
 refresh
 true
 ([conn req] (refresh conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_refresh") req)))

(defmethod
 refresh
 false
 ([conn] (refresh conn nil))
 ([conn index] (request conn :post (format-uri index "_refresh") {})))

(defmulti get-mapping (fn [_ & args] (map? (last args))))

(defmethod
 get-mapping
 true
 ([conn req] (get-mapping conn nil req))
 ([conn index req] (get-mapping conn index nil req))
 ([conn index type req]
  (request conn :get (format-uri index "_mapping" type) req)))

(defmethod
 get-mapping
 false
 ([conn] (get-mapping conn nil))
 ([conn index] (get-mapping conn index nil))
 ([conn index type]
  (request conn :get (format-uri index "_mapping" type) {})))

(defn update-aliases
  ([conn req] (request conn :post (format-uri "_aliases") req)))

(defn delete-alias
  ([conn index name] (delete-alias conn index name nil))
  ([conn index name req]
    (request conn :delete (format-uri index "_aliases" name) req)))

(defn put-settings
  ([conn req] (put-settings conn nil req))
  ([conn index req]
    (request conn :put (format-uri index "_settings") req)))

(defn delete
  ([conn index] (delete conn index nil))
  ([conn index req] (request conn :delete (format-uri index) req)))

(defmulti flush-synced (fn [_ & args] (map? (last args))))

(defmethod
 flush-synced
 true
 ([conn req] (flush-synced conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_flush" "synced") req)))

(defmethod
 flush-synced
 false
 ([conn] (flush-synced conn nil))
 ([conn index]
  (request conn :post (format-uri index "_flush" "synced") {})))

(defmulti upgrade (fn [_ & args] (map? (last args))))

(defmethod
 upgrade
 true
 ([conn req] (upgrade conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_upgrade") req)))

(defmethod
 upgrade
 false
 ([conn] (upgrade conn nil))
 ([conn index] (request conn :post (format-uri index "_upgrade") {})))

(defmulti stats (fn [_ & args] (map? (last args))))

(defmethod
 stats
 true
 ([conn req] (stats conn nil req))
 ([conn index req] (stats conn index nil req))
 ([conn index metric req]
  (request conn :get (format-uri index "_stats" metric) req)))

(defmethod
 stats
 false
 ([conn] (stats conn nil))
 ([conn index] (stats conn index nil))
 ([conn index metric]
  (request conn :get (format-uri index "_stats" metric) {})))

(defmulti get-template (fn [_ & args] (map? (last args))))

(defmethod
 get-template
 true
 ([conn req] (get-template conn nil req))
 ([conn name req]
  (request conn :get (format-uri "_template" name) req)))

(defmethod
 get-template
 false
 ([conn] (get-template conn nil))
 ([conn name] (request conn :get (format-uri "_template" name) {})))

(defmulti segments (fn [_ & args] (map? (last args))))

(defmethod
 segments
 true
 ([conn req] (segments conn nil req))
 ([conn index req]
  (request conn :get (format-uri index "_segments") req)))

(defmethod
 segments
 false
 ([conn] (segments conn nil))
 ([conn index] (request conn :get (format-uri index "_segments") {})))

(defmulti rollover (fn [_ & args] (map? (last args))))

(defmethod
 rollover
 true
 ([conn alias req] (rollover conn alias nil req))
 ([conn alias new-index req]
  (request conn :post (format-uri alias "_rollover" new-index) req)))

(defmethod
 rollover
 false
 ([conn alias] (rollover conn alias nil))
 ([conn alias new-index]
  (request conn :post (format-uri alias "_rollover" new-index) {})))

(defn shrink
  ([conn index target] (shrink conn index target nil))
  ([conn index target req]
    (request conn :put (format-uri index "_shrink" target) req)))

(defn put-template
  ([conn name req]
    (request conn :put (format-uri "_template" name) req)))

(defmulti get-upgrade (fn [_ & args] (map? (last args))))

(defmethod
 get-upgrade
 true
 ([conn req] (get-upgrade conn nil req))
 ([conn index req]
  (request conn :get (format-uri index "_upgrade") req)))

(defmethod
 get-upgrade
 false
 ([conn] (get-upgrade conn nil))
 ([conn index] (request conn :get (format-uri index "_upgrade") {})))

(defn close
  ([conn index] (close conn index nil))
  ([conn index req]
    (request conn :post (format-uri index "_close") req)))

(defmulti clear-cache (fn [_ & args] (map? (last args))))

(defmethod
 clear-cache
 true
 ([conn req] (clear-cache conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_cache" "clear") req)))

(defmethod
 clear-cache
 false
 ([conn] (clear-cache conn nil))
 ([conn index]
  (request conn :post (format-uri index "_cache" "clear") {})))

(defmulti get (fn [_ & args] (map? (last args))))

(defmethod
 get
 true
 ([conn index req] (get conn index nil req))
 ([conn index feature req]
  (request conn :get (format-uri index feature) req)))

(defmethod
 get
 false
 ([conn index] (get conn index nil))
 ([conn index feature]
  (request conn :get (format-uri index feature) {})))

(defmulti analyze (fn [_ & args] (map? (last args))))

(defmethod
 analyze
 true
 ([conn req] (analyze conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_analyze") req)))

(defmethod
 analyze
 false
 ([conn] (analyze conn nil))
 ([conn index] (request conn :post (format-uri index "_analyze") {})))

(defn put-alias
  ([conn index name] (put-alias conn index name nil))
  ([conn index name req]
    (request conn :put (format-uri index "_aliases" name) req)))

(defmulti flush (fn [_ & args] (map? (last args))))

(defmethod
 flush
 true
 ([conn req] (flush conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_flush") req)))

(defmethod
 flush
 false
 ([conn] (flush conn nil))
 ([conn index] (request conn :post (format-uri index "_flush") {})))

(defmulti shard-stores (fn [_ & args] (map? (last args))))

(defmethod
 shard-stores
 true
 ([conn req] (shard-stores conn nil req))
 ([conn index req]
  (request conn :get (format-uri index "_shard_stores") req)))

(defmethod
 shard-stores
 false
 ([conn] (shard-stores conn nil))
 ([conn index]
  (request conn :get (format-uri index "_shard_stores") {})))

(defmulti forcemerge (fn [_ & args] (map? (last args))))

(defmethod
 forcemerge
 true
 ([conn req] (forcemerge conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_forcemerge") req)))

(defmethod
 forcemerge
 false
 ([conn] (forcemerge conn nil))
 ([conn index]
  (request conn :post (format-uri index "_forcemerge") {})))

(defmulti get-alias (fn [_ & args] (map? (last args))))

(defmethod
 get-alias
 true
 ([conn req] (get-alias conn nil req))
 ([conn index req] (get-alias conn index nil req))
 ([conn index name req]
  (request conn :get (format-uri index "_alias" name) req)))

(defmethod
 get-alias
 false
 ([conn] (get-alias conn nil))
 ([conn index] (get-alias conn index nil))
 ([conn index name]
  (request conn :get (format-uri index "_alias" name) {})))

(defmulti recovery (fn [_ & args] (map? (last args))))

(defmethod
 recovery
 true
 ([conn req] (recovery conn nil req))
 ([conn index req]
  (request conn :get (format-uri index "_recovery") req)))

(defmethod
 recovery
 false
 ([conn] (recovery conn nil))
 ([conn index] (request conn :get (format-uri index "_recovery") {})))

(defn delete-template
  ([conn name] (delete-template conn name nil))
  ([conn name req]
    (request conn :delete (format-uri "_template" name) req)))

