(ns elasticsearch.async
  (:require [cheshire.core :as json]
            [clojure.core.async
             :refer [chan go <! <!! >! >!! alts! alts!!
                     close!]
             :as async]
            [clojure.spec :as s]
            [elasticsearch.document :as doc]
            [elasticsearch.indices.managed :as im]
            [robert.bruce :refer [try-try-again *try*] :as try]))

(Thread/setDefaultUncaughtExceptionHandler
 (reify Thread$UncaughtExceptionHandler
   (uncaughtException [_ thread ex]
     (println ex "Uncaught exception on" (.getName thread)))))

(s/def ::throwable #(instance? Throwable %))

(s/def ::error (s/keys :req [::throwable]))

(defn buffering-proc
  "Takes a function f and returns an input and output channel.  Items
  put into the input channel will be buffered together until
  either (pred buf), or interval millis is exhausted, at which
  point (f buf) will be applied if (has-items? buf).

  The return value of each call to f will be put into the output
  channel."
  ([f* new-buf conj-item has-items? get-items preds interval
    queue-size-in queue-size-out]
   (let [in (if (>= queue-size-in 0)
              (chan queue-size-in)
              (chan))
         out (if (>= queue-size-out 0)
               (chan queue-size-out)
               (chan))
         f (fn [& args]
             (try
               (apply f* args)
               (catch Throwable t
                 (>!! out {::throwable t})
                 :err)))]
     (go
       (loop [buf (new-buf)]
         (let [[x ch] (alts! [in (async/timeout interval)])]
           (conj-item buf x)
           (cond
             ;; apply the buffer to f and empty it, whether...
             (or
              ;; ...interval expired
              (nil? x)
              ;; ...or at least one pred true
              (some identity (for [pred preds] (pred buf))))
             (if (has-items? buf)
               (do
                 (let [res (f (get-items buf))]
                   ;; make sure we don't send the `true` (from the >!!
                   ;; after an error) back through the response
                   ;; channel
                   (when-not (= :err res)
                     (>! out res)))
                 (recur (new-buf)))
               ;; it was empty, let's just reuse it
               (recur buf))

             ;; nothing to do, go again
             :else
             (recur buf)))))
     [in out])))

(defn bulk-buffering-proc
  ([f max-bytes max-actions interval queue-size-req queue-size-resp]
   (let [buf-init (fn []
                    (atom {:xs [] :bytes 0}))
         count+ (fn [v x]
                  (if x
                    (-> v
                        (update-in [:xs] conj x)
                        (update-in [:bytes] + (count (json/encode x))))
                    v))]
     (buffering-proc
      f
      buf-init
      (fn [buf val] (swap! buf count+ val))
      (fn [buf] (pos? (count (:xs @buf))))
      (fn [buf] (:xs @buf))
      [(fn [buf] (and max-bytes (>= (:bytes @buf) max-bytes)))
       (fn [buf] (and max-actions (>= (count (:xs @buf)) max-actions)))]
      interval
      queue-size-req
      queue-size-resp))))

(defn start-bulk-processor!
  "Returns [in out] channels to write actions to, and receive
  responses from."
  [conn index type req opts]
  (bulk-buffering-proc
   (fn [actions]
     (try-try-again
      {:sleep (:retry-delay opts 200)
       :tries (:retry-count opts 3)
       :decay try/exponential}
      #(doc/bulk conn index type (assoc req :body actions))))
   (:bulk-bytes opts)
   (:bulk-actions opts)
   (:interval opts 1000)
   (:queue-size-req opts -1)
   (:queue-size-resp opts -1)))

(defn start-rolling-bulk-processor!
  "Returns [in out] channels to write actions to, and receive
  responses from."
  [conn index type {:keys [query-params body]} opts]
  (bulk-buffering-proc
   (fn [actions]

     (let [idx (im/get-or-roll-over conn
                                    index
                                    body
                                    (:max-index-bytes opts im/MAX_INDEX_BYTES))
           req {:query-params query-params
                :body actions}]
       (try-try-again
        {:sleep (:retry-delay opts 100)
         :tries (:retry-count opts 3)}
        #(doc/bulk conn idx type req))))
   (:bulk-bytes opts)
   (:bulk-actions opts)
   (:interval opts 1000)
   (:queue-size-req opts -1)
   (:queue-size-resp opts -1)))
