(ns elasticsearch.indices.managed
  (:require [clojure.spec :as s]
            [elasticsearch.connection :as c]
            [elasticsearch.connection.http :as hconn]
            [elasticsearch.indices :as i]
            [elasticsearch.time :as time]
            [slingshot.slingshot :refer [try+ throw+]]))

(def MAX_INDEX_BYTES
  (* 1024 1024 1024))

(s/def ::max-index-bytes
  integer?)

(defn make-timestamped-name [prefix]
  (format "%s-%s" prefix (time/now-digits)))

(s/fdef newest-index-matching-pattern
          :args (s/cat :c ::c/connection
                       :pat string?)
          :ret keyword?)
(defn newest-index-matching-pattern
  [conn pat]
  (try+
   (let [resp (i/get-settings conn pat)]
     (when (not-empty resp)
       (->> resp
            (map #(vector
                   (key %)
                   (Long/parseLong
                    (get-in (val %) [:settings :index :creation_date]))))
            (sort-by second)
            reverse
            first
            first
            name)))
   (catch [:status 404] _
     ;; index doesn't exist
     )))

(s/fdef index-size-bytes
          :args (s/cat :c ::c/connection
                       :idx (s/nilable ::i/name))
          :ret int?)
(defn index-size-bytes
  [conn idx]
  (when idx
    (let [r (try+
             (c/request conn :get
                        {:uri (format "/%s/_stats/store" (name idx))})
             (catch [:status 404] _ nil))
          size (get-in r [:indices (keyword idx)
                          :primaries :store :size_in_bytes])]
      (or size 0))))

(s/fdef create-and-wait-ready
          :args (s/cat :c ::c/connection :idx ::i/name :body map?)
          :ret ::i/name)
(defn create-and-wait-for-status
  [conn idx body status]
  (try+
   (i/create conn idx {:body body})
   (i/wait-for-health conn status idx)
   (catch [:status 400] e
     (when-not (= (get-in e [:body :error :type])
                  "index_already_exists_exception")
       (throw+ e))))
  idx)

(s/fdef create-timestamped-index
          :args (s/cat :c ::c/connection
                       :p string?
                       :body ::i/metadata))
(defn create-timestamped-index
  [conn prefix body]
  (let [idx (make-timestamped-name prefix)]
    (create-and-wait-for-status conn idx body :yellow)))

(s/fdef get-or-roll-over
          :args (s/alt
                 :a1 (s/cat :c ::c/connection
                            :i ::i/name
                            :b map?)
                 :a2 (s/cat :c ::c/connection
                            :i ::i/name
                            :b map?
                            :bytes ::max-index-bytes))
          :ret keyword?)
(defn get-or-roll-over
  ([conn idx body]
   (get-or-roll-over conn idx body MAX_INDEX_BYTES))
  ([conn idx body max-bytes]
   (let [newest (newest-index-matching-pattern
                 conn (format "%s*" idx))
         newest-bytes (index-size-bytes conn newest)]
     (if (and newest (<= newest-bytes max-bytes))
       (name newest)
       (create-timestamped-index conn idx body)))))
