(ns elasticsearch.document
  (:refer-clojure :exclude [get delete count update])
  (:require
   [elasticsearch.common :refer [format-uri request]]
   [elasticsearch.connection.http :as hconn]
   [elasticsearch.connection :as conn]
   [slingshot.slingshot :refer [try+ throw+]]))

(defn index
  ([conn index type req]
   (request conn :post (format-uri index type) req))
  ([conn index type id req]
   (request conn :put (format-uri index type id) req)))

(defn exists
  ([conn idx type id]
   (exists conn idx type id {}))
  ([conn idx type id req]
   (try+
    (request conn :head (format-uri idx type id) req)
    true
    (catch [:status 404] _
      false))))

(defn bulk
  ([conn req]
   (bulk conn nil req))
  ([conn idx req]
   (bulk conn idx nil req))
  ([conn idx type req]
   (let [res (request conn :post (format-uri idx type "_bulk") req)]
     (when (:errors res)
       ;; A response that contains an error looks like:
       ;;
       ;; {:took 17,
       ;;  :errors true,
       ;;  :items
       ;;  [{:delete
       ;;    {:_index "FOO",
       ;;     :_type "FAKE",
       ;;     :_id "NO",
       ;;     :status 400,
       ;;     :error
       ;;     {:type "invalid_index_name_exception",
       ;;      :reason "Invalid index name [FOO], must be lowercase",
       ;;      :index "FOO"}}}]}
       (let [collate-errors (fn [orig-req resp]
                              (let [item (-> resp vals first)]
                                (when (:error item)
                                  (assoc item :original-item orig-req))))
             actions-with-errors (remove nil?
                                         (map collate-errors
                                              (:body req) (:items res)))]
         (throw+ {:type ::bulk-error
                  :count (clojure.core/count actions-with-errors)
                  :items actions-with-errors})))
     res)))

(defn msearch
  ([conn req]
   (msearch conn nil req))
  ([conn idx req]
   (msearch conn idx nil req))
  ([conn idx type req]
   (let [res (request conn :get (format-uri idx type "_msearch") req)]
     (when (:errors res)

       (let [has-error? (fn [action]
                          (:error ((comp first vals) action)))
             actions-with-errors (filter has-error? (:items res))]
         (throw+ {:type ::msearch-error
                  :count (clojure.core/count actions-with-errors)
                  :items actions-with-errors})))
     res)))

;; Begin generated code. Everything above will be kept and anything below will be regenerated.

(defmulti clear-scroll (fn [_ & args] (map? (last args))))

(defmethod
 clear-scroll
 true
 ([conn req] (clear-scroll conn nil req))
 ([conn scroll-id req]
  (request conn :delete (format-uri "_search" "scroll" scroll-id) req)))

(defmethod
 clear-scroll
 false
 ([conn] (clear-scroll conn nil))
 ([conn scroll-id]
  (request conn :delete (format-uri "_search" "scroll" scroll-id) {})))

(defn delete
  ([conn index type id] (delete conn index type id nil))
  ([conn index type id req]
    (request conn :delete (format-uri index type id) req)))

(defn delete-by-query
  ([conn index req] (delete-by-query conn index nil req))
  ([conn index type req]
    (request
      conn
      :post
      (format-uri index type "_delete_by_query")
      req)))

(defn delete-script
  ([conn id lang] (delete-script conn id lang nil))
  ([conn id lang req]
    (request conn :delete (format-uri "_scripts" lang id) req)))

(defn exists-source
  ([conn index type id] (exists-source conn index type id nil))
  ([conn index type id req]
    (request conn :head (format-uri index type id "_source") req)))

(defmulti field-caps (fn [_ & args] (map? (last args))))

(defmethod
 field-caps
 true
 ([conn req] (field-caps conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_field_caps") req)))

(defmethod
 field-caps
 false
 ([conn] (field-caps conn nil))
 ([conn index]
  (request conn :post (format-uri index "_field_caps") {})))

(defn get
  ([conn index type id] (get conn index type id nil))
  ([conn index type id req]
    (request conn :get (format-uri index type id) req)))

(defn get-script
  ([conn id lang] (get-script conn id lang nil))
  ([conn id lang req]
    (request conn :get (format-uri "_scripts" lang id) req)))

(defn get-source
  ([conn index type id] (get-source conn index type id nil))
  ([conn index type id req]
    (request conn :get (format-uri index type id "_source") req)))

(defn mget
  ([conn req] (mget conn nil req))
  ([conn index req] (mget conn index nil req))
  ([conn index type req]
    (request conn :post (format-uri index type "_mget") req)))

(defn reindex
  ([conn req] (request conn :post (format-uri "_reindex") req)))

(defn create
  ([conn index type id req]
    (request conn :put (format-uri index type id "_create") req)))

(defn msearch-template
  ([conn req] (msearch-template conn nil req))
  ([conn index req] (msearch-template conn index nil req))
  ([conn index type req]
    (request
      conn
      :post
      (format-uri index type "_msearch" "template")
      req)))

(defmulti mtermvectors (fn [_ & args] (map? (last args))))

(defmethod
 mtermvectors
 true
 ([conn req] (mtermvectors conn nil req))
 ([conn index req] (mtermvectors conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_mtermvectors") req)))

(defmethod
 mtermvectors
 false
 ([conn] (mtermvectors conn nil))
 ([conn index] (mtermvectors conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_mtermvectors") {})))

(defn suggest
  ([conn req] (suggest conn nil req))
  ([conn index req]
    (request conn :post (format-uri index "_suggest") req)))

(defmulti count (fn [_ & args] (map? (last args))))

(defmethod
 count
 true
 ([conn req] (count conn nil req))
 ([conn index req] (count conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_count") req)))

(defmethod
 count
 false
 ([conn] (count conn nil))
 ([conn index] (count conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_count") {})))

(defn put-script
  ([conn id lang req]
    (request conn :put (format-uri "_scripts" lang id) req)))

(defmulti reindex-rethrottle (fn [_ & args] (map? (last args))))

(defmethod
 reindex-rethrottle
 true
 ([conn req] (reindex-rethrottle conn nil req))
 ([conn task-id req]
  (request
   conn
   :post
   (format-uri "_update_by_query" task-id "_rethrottle")
   req)))

(defmethod
 reindex-rethrottle
 false
 ([conn] (reindex-rethrottle conn nil))
 ([conn task-id]
  (request
   conn
   :post
   (format-uri "_update_by_query" task-id "_rethrottle")
   {})))

(defmulti render-search-template (fn [_ & args] (map? (last args))))

(defmethod
 render-search-template
 true
 ([conn req] (render-search-template conn nil req))
 ([conn id req]
  (request conn :post (format-uri "_render" "template" id) req)))

(defmethod
 render-search-template
 false
 ([conn] (render-search-template conn nil))
 ([conn id]
  (request conn :post (format-uri "_render" "template" id) {})))

(defmulti search (fn [_ & args] (map? (last args))))

(defmethod
 search
 true
 ([conn req] (search conn nil req))
 ([conn index req] (search conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_search") req)))

(defmethod
 search
 false
 ([conn] (search conn nil))
 ([conn index] (search conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_search") {})))

(defmulti search-shards (fn [_ & args] (map? (last args))))

(defmethod
 search-shards
 true
 ([conn req] (search-shards conn nil req))
 ([conn index req] (search-shards conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_search_shards") req)))

(defmethod
 search-shards
 false
 ([conn] (search-shards conn nil))
 ([conn index] (search-shards conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_search_shards") {})))

(defmulti search-template (fn [_ & args] (map? (last args))))

(defmethod
 search-template
 true
 ([conn req] (search-template conn nil req))
 ([conn index req] (search-template conn index nil req))
 ([conn index type req]
  (request
   conn
   :post
   (format-uri index type "_search" "template")
   req)))

(defmethod
 search-template
 false
 ([conn] (search-template conn nil))
 ([conn index] (search-template conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_search" "template") {})))

(defmulti termvectors (fn [_ & args] (map? (last args))))

(defmethod
 termvectors
 true
 ([conn index type req] (termvectors conn index type nil req))
 ([conn index type id req]
  (request conn :post (format-uri index type id "_termvectors") req)))

(defmethod
 termvectors
 false
 ([conn index type] (termvectors conn index type nil))
 ([conn index type id]
  (request conn :post (format-uri index type id "_termvectors") {})))

(defn update
  ([conn index type id] (update conn index type id nil))
  ([conn index type id req]
    (request conn :post (format-uri index type id "_update") req)))

(defn explain
  ([conn index type id] (explain conn index type id nil))
  ([conn index type id req]
    (request conn :post (format-uri index type id "_explain") req)))

(defmulti count-percolate (fn [_ & args] (map? (last args))))

(defmethod
 count-percolate
 true
 ([conn index type req] (count-percolate conn index type nil req))
 ([conn index type id req]
  (request
   conn
   :post
   (format-uri index type id "_percolate" "count")
   req)))

(defmethod
 count-percolate
 false
 ([conn index type] (count-percolate conn index type nil))
 ([conn index type id]
  (request
   conn
   :post
   (format-uri index type id "_percolate" "count")
   {})))

(defn delete-template
  ([conn id] (delete-template conn id nil))
  ([conn id req]
    (request conn :delete (format-uri "_search" "template" id) req)))

(defmulti field-stats (fn [_ & args] (map? (last args))))

(defmethod
 field-stats
 true
 ([conn req] (field-stats conn nil req))
 ([conn index req]
  (request conn :post (format-uri index "_field_stats") req)))

(defmethod
 field-stats
 false
 ([conn] (field-stats conn nil))
 ([conn index]
  (request conn :post (format-uri index "_field_stats") {})))

(defn get-template
  ([conn id] (get-template conn id nil))
  ([conn id req]
    (request conn :get (format-uri "_search" "template" id) req)))

(defmulti percolate (fn [_ & args] (map? (last args))))

(defmethod
 percolate
 true
 ([conn index type req] (percolate conn index type nil req))
 ([conn index type id req]
  (request conn :post (format-uri index type id "_percolate") req)))

(defmethod
 percolate
 false
 ([conn index type] (percolate conn index type nil))
 ([conn index type id]
  (request conn :post (format-uri index type id "_percolate") {})))

(defn put-template
  ([conn id req]
    (request conn :put (format-uri "_search" "template" id) req)))

(defmulti update-by-query (fn [_ & args] (map? (last args))))

(defmethod
 update-by-query
 true
 ([conn index req] (update-by-query conn index nil req))
 ([conn index type req]
  (request conn :post (format-uri index type "_update_by_query") req)))

(defmethod
 update-by-query
 false
 ([conn index] (update-by-query conn index nil))
 ([conn index type]
  (request conn :post (format-uri index type "_update_by_query") {})))
