# Elasticsearch on Clojure

There will be seqs.


## Usage

Leiningen setup:

For 7.x:
```clojure
[org.elasticsearch/elasticsearch-clojure "7.0.0"]
```

For pre-7.x:

```clojure
[org.elasticsearch/elasticsearch-clojure "0.99.0-SNAPSHOT"]
```

Common namespaces:

```clojure
(:require [elasticsearch.document :as doc]
          [elasticsearch.connection.http :as conn]
          [elasticsearch.indices :as indices]
          [elasticsearch.cluster :as cluster])
```

## 6.x -> 7.x differences

The big difference is that, with document types removed in 7.x, any functions
which had a type argument have had that argument removed to match.

For example, `(index conn index type id req)` becomes `(index conn index id req)`.

## Releasing

Make sure you have your lein credentials set up.  If you haven't yet done that, this should get you started:

```
% cat <<EOF | gpg -e -r YOURKEY > ~/.lein/credentials.clj.gpg
{#"https://clojars.org/repo"
 {:username "USER" :password "PASS"}}
EOF
```

Now you should be able to deploy to Clojars.  This will deploy the current snap, feel free to do at will.

```
% lein with-profile package deploy clojars
```

When you're ready to bump versions and make a release (again, make sure your GPG setup is correct or you'll have to `git reset` and `git tag -d` some things):

```
% lein release
```

## Development

### Connection API

#### Single node

The simplest connection. A single node, no auth or encryption.

```clojure
(require '[elasticsearch.connection.http :as conn])
(def conn
  (conn/make {:url "http://localhost:9200"}))
```

#### Encryption and authentication

Configuration is passed straight through to
`#'clj-http.client/request`, so the semantics of `wrap-request` apply
here, like `:url` will overwrite `:scheme` and so forth.

The usual ceremony with Java KeyStores is supported as well.

```clojure
(def conn
  (let [opts {:keystore "path/to/ks"
              :keystore-pass "keystore!"
              :trust-store "path/to/ts"
              :trust-store-pass "truststore!"
              :user-info "foo:pass"}]
    (conn/make
     (merge opts {:url (format "https://node01:9200")}))))
```

#### Multi-node

TODO

```clojure
(def conn
  (let [opts {:user-info "foo:pass"}]
    (conn/make-connection-pool
     (for [i (range 3)]
       (conn/make-connection
        (merge opts {:url (format "https://node%d:9200" i)}))))))
```

### Create Index API

```clojure
(indices/create conn "foo" {:body {:settings {:index.number_of_shards 1}}})
```

### Index API

```clojure
(doc/index conn {:index "foo" :type "t" :id "1" :body {:f "some field"}})
```


### GET API

```clojure
(doc/get conn {:index "foo" :type "t" :id "1"})
```

### Search API

```clojure
(doc/search conn {:body
                  {:query
                   {:match
                    {:field "some"}}}})
```
