(ns elasticsearch.ilm
  (:refer-clojure :exclude [get delete ensure remove])
  (:require [elasticsearch.common :refer [format-uri request]]
            [elasticsearch.connection :as conn]
            [elasticsearch.connection.http :as hconn]))

;; policy APIs
(defn policies
  "Returns all definied policies."
  [conn]
  (request conn :get (format-uri "_ilm" "policy")))

(defn create
  "Create an ILM policy.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-put-lifecycle.html"
  [conn policy req]
  (request conn :put (format-uri "_ilm" "policy" (name policy)) {:body req}))

(defn ensure
  "Ensure a policy exists, updating it if present, creating it if not."
  [& args]
  ;; ILM create is effectively idempotent; this function is here to
  ;; provide standard APIs, for example see elasticsearch.indices/ensure
  (apply create args))

(defn get
  "Retrieve a policy.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-get-lifecycle.html"
  ([conn policy]
   (get conn policy {}))
  ([conn policy req]
   (request conn :get (format-uri "_ilm" "policy" (name policy)) req)))

(defn exists
  "Test if a policy exists. Returns the policy if it does, boolean false if not."
  ([conn policy]
   (exists conn policy {}))
  ([conn policy req]
   (try
     (get conn policy req)
     (catch clojure.lang.ExceptionInfo e
       (if (= 404 (:status (ex-data e)))
         false
         (throw e))))))

(defn delete
  "Delete a policy.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-delete-lifecycle.html"
  ([conn policy]
   (delete conn policy {}))
  ([conn policy req]
   (request conn :delete (format-uri "_ilm" "policy" (name policy)))))

;; index APIs
(defn move
  "Move an index to a particular step in the policy.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-move-to-step.html"
  [conn idx req]
  (request conn :post (format-uri "_ilm" "move" idx) req))

(defn retry
  "Retry executing the policy for an index that is in ERROR step.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-retry-policy.html"
  ([conn idx]
   (retry conn idx {}))
  ([conn idx req]
   (request conn :post (format-uri idx "_ilm" "retry") req)))

(defn remove
  "Remove policies from an index.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-remove-policy.html"
  ([conn idx]
   (remove conn idx {}))
  ([conn idx req]
   (request conn :post (format-uri idx "_ilm" "remove") req)))

(defn explain
  "Retrieve the current lifecycle status for an index.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-explain-lifecycle.html"
  ([conn idx]
   (explain conn idx {}))
  ([conn idx req]
   (request conn :get (format-uri idx "_ilm" "explain") req)))

;; management APIs
(defn status
  "Return the status of ILM's current operation mode.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-get-status.html"
  ([conn]
   (status conn {}))
  ([conn req]
   (request conn :get (format-uri "_ilm" "status") req)))

(defn start
  "Start the ILM system for the cluster.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-start.html"
  ([conn]
   (start conn {}))
  ([conn req]
   (request conn :post (format-uri "_ilm" "start") req)))

(defn stop
  "Stop the ILM system for the cluster.
  https://www.elastic.co/guide/en/elasticsearch/reference/current/ilm-stop.html"
  ([conn]
   (stop conn {}))
  ([conn req]
   (request conn :post (format-uri "_ilm" "stop") req)))
