var Prime = (function (exports) {
	'use strict';

	/*
	 * Copyright (c) 2012-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	/**
	 * The Browser namespace. This namespace does not contain any classes, just functions.
	 *
	 * @namespace Browser
	 */
	const Browser = {
	  /**
	   * Detects the browser name and version.
	   */
	  detect: function() {
	    this.name = this._searchString(this.dataBrowser) || "An unknown browser";
	    this.version = this._searchVersion(navigator.userAgent) || this._searchVersion(navigator.appVersion) || "an unknown version";
	    this.os = this._searchString(this.dataOS) || "an unknown OS";
	  },


	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  /**
	   *
	   * @param {Object} data The data array.
	   * @returns {?string} The browser identity String.
	   * @private
	   */
	  _searchString: function(data) {
	    for (let i = 0; i < data.length; i++) {
	      const dataString = data[i].string;
	      const dataProp = data[i].prop;
	      this.versionSearchString = data[i].versionSearch || data[i].identity;
	      if (dataString && dataString.indexOf(data[i].subString) !== -1) {
	        return data[i].identity;
	      } else if (dataProp) {
	        return data[i].identity;
	      }
	    }

	    return null;
	  },

	  /**
	   *
	   * @param {string} dataString The browser data string.
	   * @returns {?number} The version or null.
	   * @private
	   */
	  _searchVersion: function(dataString) {
	    const index = dataString.indexOf(this.versionSearchString);
	    if (index === -1) {
	      return null;
	    }

	    return parseFloat(dataString.substring(index + this.versionSearchString.length + 1));
	  },

	  dataBrowser: [
	    {
	      string: navigator.userAgent,
	      subString: "Chrome",
	      identity: "Chrome"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "OmniWeb",
	      versionSearch: "OmniWeb/",
	      identity: "OmniWeb"
	    },
	    {
	      string: navigator.vendor,
	      subString: "Apple",
	      identity: "Safari",
	      versionSearch: "Version"
	    },
	    {
	      prop: window.opera,
	      identity: "Opera",
	      versionSearch: "Version"
	    },
	    {
	      string: navigator.vendor,
	      subString: "iCab",
	      identity: "iCab"
	    },
	    {
	      string: navigator.vendor,
	      subString: "KDE",
	      identity: "Konqueror"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "Firefox",
	      identity: "Firefox"
	    },
	    {
	      string: navigator.vendor,
	      subString: "Camino",
	      identity: "Camino"
	    },
	    {    // for newer Netscapes (6+)
	      string: navigator.userAgent,
	      subString: "Netscape",
	      identity: "Netscape"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "MSIE",
	      identity: "Explorer",
	      versionSearch: "MSIE"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "Gecko",
	      identity: "Mozilla",
	      versionSearch: "rv"
	    },
	    {     // for older Netscapes (4-)
	      string: navigator.userAgent,
	      subString: "Mozilla",
	      identity: "Netscape",
	      versionSearch: "Mozilla"
	    }
	  ],
	  dataOS: [
	    {
	      string: navigator.platform,
	      subString: "Win",
	      identity: "Windows"
	    },
	    {
	      string: navigator.platform,
	      subString: "Mac",
	      identity: "Mac"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "iPhone",
	      identity: "iPhone/iPod"
	    },
	    {
	      string: navigator.userAgent,
	      subString: "iPad",
	      identity: "iPad"
	    },
	    {
	      string: navigator.platform,
	      subString: "Linux",
	      identity: "Linux"
	    }
	  ]
	};
	Browser.detect();

	/*
	 * Copyright (c) 2017-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const blockElementRegexp = /^(?:ARTICLE|ASIDE|BLOCKQUOTE|BODY|BR|BUTTON|CANVAS|CAPTION|COL|COLGROUP|DD|DIV|DL|DT|EMBED|FIELDSET|FIGCAPTION|FIGURE|FOOTER|FORM|H1|H2|H3|H4|H5|H6|HEADER|HGROUP|HR|LI|MAP|OBJECT|OL|OUTPUT|P|PRE|PROGRESS|SECTION|TABLE|TBODY|TEXTAREA|TFOOT|TH|THEAD|TR|UL|VIDEO)$/;
	const mouseEventsRegexp = /^(?:click|dblclick|mousedown|mouseup|mouseover|mousemove|mouseout|mouseenter|mouseleave)$/;
	const htmlEventsRegexp = /^(?:abort|blur|change|error|focus|load|reset|resize|scroll|select|submit|unload)$/;
	let anonymousId = 1;
	const ieAlphaRegexp = /alpha\(opacity=(.+)\)/;

	class PrimeElement {
	  /**
	   * Creates an Element class for the given DOM element.
	   *
	   * @constructor
	   * @param {Element|EventTarget} element The element
	   */
	  constructor(element) {
	    if (!Utils.isDefined(element.nodeType) || element.nodeType !== 1) {
	      throw new TypeError('You can only pass in DOM element Node objects to the PrimeDocument.Element constructor');
	    }

	    Utils.bindAll(this);
	    this.domElement = element;
	    this.domElement.customEventListeners = [];
	    this.domElement.eventListeners = {};
	  }

	  /**
	   * Regular expression that captures the tagnames of all the block elements in HTML5.
	   *
	   * @type {RegExp}
	   */
	  static get blockElementRegexp() {
	    return blockElementRegexp;
	  }

	  static get mouseEventsRegexp() {
	    return mouseEventsRegexp;
	  }

	  static get htmlEventsRegexp() {
	    return htmlEventsRegexp;
	  }

	  static get anonymousId() {
	    return anonymousId;
	  }

	  static set anonymousId(value) {
	    anonymousId = value;
	  }

	  static get ieAlphaRegexp() {
	    return ieAlphaRegexp;
	  }

	  /**
	   * Static method that wraps an element in a PrimeElement unless it is already wrapped. In that case, it simply
	   * returns the element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The element to wrap.
	   * @static
	   */
	  static wrap(element) {
	    return (element instanceof PrimeElement) ? element : new PrimeElement(element);
	  }

	  /**
	   * Static method that unwraps an element to a DOM element. This is the inverse of Element.wrap. If this is passed an
	   * Element, it returns domElement. Otherwise, this just returns the element.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  static unwrap(element) {
	    return (element instanceof PrimeElement) ? element.domElement : element;
	  }

	  /**
	   * Adds the given class (or list of space separated classes) to this Element.
	   *
	   * @param {string} classNames The class name(s) separated by a space.
	   * @returns {PrimeElement} This Element.
	   */
	  addClass(classNames) {
	    let currentClassName = this.domElement.className;
	    if (currentClassName === '') {
	      currentClassName = classNames;
	    } else {
	      const currentClassNameList = this.domElement.className.split(Utils.spaceRegex);
	      const newClassNameList = classNames.split(Utils.spaceRegex);
	      for (let i = 0; i < newClassNameList.length; i++) {
	        if (currentClassNameList.indexOf(newClassNameList[i]) === -1) {
	          currentClassNameList.push(newClassNameList[i]);
	        }
	      }

	      currentClassName = currentClassNameList.join(' ');
	    }

	    this.domElement.className = currentClassName;
	    return this;
	  }

	  /**
	   * @callback PrimeDelegationEventListener
	   *
	   * @param {Event} event the original event
	   * @param {PrimeElement} target the target domElement that matched the registered selector
	   */

	  /**
	   * Attaches an event listener to the element and will only invoke the listener when the event target matches
	   * the provided selector.
	   *
	   * The intent of this function is to provide a delegated listener and handle events from nested elements.
	   *
	   * @param {string} event The name of the event
	   * @param  {string} selector The selector to match against the Element
	   * @param {PrimeDelegationEventListener} listener The event listener function
	   */
	  addDelegatedEventListener(event, selector, listener) {
	    this.domElement.delegatedListeners =  this.domElement.delegatedListeners || [];
	    let allDelegatedListeners = this.domElement.delegatedListeners;

	    // Store the event listeners in the following format:
	    //   [ event_type -> [ selector -> [ listeners ] ] ]
	    //
	    // Example:
	    //   [ 'click' -> [ 'a.foo' -> [ funcA, funcB ] ] ]

	    if (!(event in allDelegatedListeners)) {
	      allDelegatedListeners[event] = [];
	    }

	    if (!(selector in allDelegatedListeners[event])) {
	      allDelegatedListeners[event][selector] = [];
	    }

	    let delegatedListeners = allDelegatedListeners[event][selector] || [];
	    if (!(listener in delegatedListeners)) {
	      delegatedListeners.push(listener);
	    }

	    this.addEventListener(event, this._handleDelegatedEvent);
	  }

	  /**
	   * Attaches an event listener to this Element.
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The event listener function.
	   * @returns {PrimeElement} This Element.
	   */
	  addEventListener(event, listener) {
	    if (event.indexOf(':') === -1) {
	      // Traditional event
	      this.domElement.eventListeners = this.domElement.eventListeners || {};
	      this.domElement.eventListeners[event] = this.domElement.eventListeners[event] || [];
	      if (this.domElement.eventListeners[event].indexOf(listener) === -1) {
	        this.domElement.eventListeners[event].push(listener);
	      }
	      this.domElement.addEventListener(event, listener, false);
	    } else {
	      // Custom event
	      this.domElement.customEventListeners = this.domElement.customEventListeners || {};
	      this.domElement.customEventListeners[event] = this.domElement.customEventListeners[event] || [];
	      if (this.domElement.customEventListeners[event].indexOf(listener) === -1) {
	        this.domElement.customEventListeners[event].push(listener);
	      }
	    }

	    return this;
	  }

	  /**
	   * Appends the given element to this element. If the given element already exists in the DOM, it is removed from its
	   * current location and placed at the end of this element.
	   *
	   * @param {PrimeElement|Node} element The element to append.
	   * @returns {PrimeElement} This Element.
	   */
	  appendElement(element) {
	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    if (domElement.parentNode) {
	      domElement.parentNode.removeChild(domElement);
	    }

	    this.domElement.appendChild(domElement);
	    return this;
	  }

	  /**
	   * Appends the given HTML string to this element.
	   *
	   * @param {string} html The HTML to append.
	   * @returns {PrimeElement} This Element.
	   */
	  appendHTML(html) {
	    this.domElement.insertAdjacentHTML('beforeend', html);
	    return this;
	  }

	  /**
	   * Inserts this Element (which must be a newly created Element) into the DOM inside at the very end of the given
	   * element.
	   *
	   * @param {PrimeElement|Node} element The element to insert this Element into.
	   * @returns {PrimeElement} This Element.
	   */
	  appendTo(element) {
	    // Error out for now if this element is in the document so we can punt on cloning for now
	    if (this.domElement.parentNode) {
	      throw new TypeError('You can only insert new PrimeElements for now');
	    }

	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    if (domElement.parentNode) {
	      domElement.appendChild(this.domElement);
	    } else {
	      throw new TypeError('The element you passed into appendTo is not in the DOM. You can\'t insert a PrimeElement inside an element that isn\'t in the DOM yet.');
	    }

	    return this;
	  }

	  /**
	   * Fires an event on the Element.
	   *
	   * @param {string} event The name of the event.
	   * @param {Object} [memo] Assigned to the memo field of the event.
	   * @param {Object} [target] The target.
	   * @param {boolean} [bubbling] If the event is bubbling, defaults to true.
	   * @param {boolean} [cancelable] If the event is cancellable, defaults to true.
	   * @returns {PrimeElement} This Element.
	   */
	  fireEvent(event, memo, target, bubbling, cancelable) {
	    memo = Utils.isDefined(memo) ? memo : {};
	    target = Utils.isDefined(target) ? target : this;
	    bubbling = Utils.isDefined(bubbling) ? bubbling : true;
	    cancelable = Utils.isDefined(cancelable) ? cancelable : true;

	    let evt;
	    if (event.indexOf(':') === -1) {
	      // Traditional event
	      if (document.createEventObject) {
	        // Dispatch for IE
	        evt = document.createEventObject();
	        evt.memo = memo || {};
	        evt.cancelBubble = !bubbling;
	        this.domElement.fireEvent('on' + event, evt);
	      } else if (document.createEvent) {
	        // Dispatch for others
	        if (PrimeElement.mouseEventsRegexp.test(event)) {
	          evt = document.createEvent("MouseEvents");
	          evt.initMouseEvent(event, bubbling, cancelable, window, 0, 0, 0, 0, 0, false, false, false, false, 0, null);
	        } else if (PrimeElement.htmlEventsRegexp.test(event)) {
	          evt = document.createEvent("HTMLEvents");
	          evt.initEvent(event, bubbling, cancelable);
	        } else {
	          throw new TypeError('Invalid event [' + event + ']');
	        }

	        evt.memo = memo || {};
	        this.domElement.dispatchEvent(evt);
	      } else {
	        throw new TypeError('Unable to fire event. Neither createEventObject nor createEvent methods are available');
	      }
	    } else {
	      // Custom event
	      this.domElement.customEventListeners[event] = this.domElement.customEventListeners[event] || [];
	      evt = {event: event, memo: memo, target: target};
	      for (let index in this.domElement.customEventListeners[event]) {
	        if (this.domElement.customEventListeners[event].hasOwnProperty(index)) {
	          this.domElement.customEventListeners[event][index](evt);
	        }
	      }
	    }

	    return this;
	  }

	  /**
	   * Fires a custom event on the Element using the given event object rather than creating a new event object. This is
	   * useful for pass-through event handling.
	   *
	   * @param {string} event The name of the event.
	   * @param {Object} eventObj The event object to pass to the handlers.
	   * @returns {PrimeElement} This Element.
	   */
	  fireCustomEvent(event, eventObj) {
	    eventObj = Utils.isDefined(eventObj) ? eventObj : {};
	    if (event.indexOf(':') === -1) {
	      throw new TypeError('This method can only be used for custom events');
	    }

	    // Custom event
	    this.domElement.customEventListeners[event] = this.domElement.customEventListeners[event] || [];
	    for (let index in this.domElement.customEventListeners[event]) {
	      if (this.domElement.customEventListeners[event].hasOwnProperty(index)) {
	        this.domElement.customEventListeners[event][index](eventObj);
	      }
	    }

	    return this;
	  }

	  /**
	   * Puts the focus on this element.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  focus() {
	    this.domElement.focus();
	    return this;
	  }

	  /**
	   * Returns the absolute top of this element relative to the document.
	   *
	   * @returns {number} The number of pixels that this element is from the top of the document.
	   */
	  getAbsoluteTop() {
	    let top = 0;
	    let e = this.domElement;
	    while (e) {
	      top += e.offsetTop;
	      e = e.offsetParent;
	    }

	    return top;
	  }

	  /**
	   * Returns the value of the given attribute.
	   *
	   * @param {string} name The attribute name.
	   * @returns {?string} This attribute value or null.
	   */
	  getAttribute(name) {
	    const attr = this.domElement.attributes.getNamedItem(name);
	    if (attr) {
	      return attr.value;
	    }

	    return null;
	  }

	  /**
	   * Returns all of the attributes on the element as an object.
	   *
	   * @returns {object} This attributes or an empty object if there are no attributes on this element.
	   */
	  getAttributes() {
	    const attrs = {};
	    if (this.domElement.hasAttributes()) {
	      for (let i = 0; i < this.domElement.attributes.length; i++) {
	        attrs[this.domElement.attributes[i].name] = this.domElement.attributes[i].value;
	      }
	    }

	    return attrs;
	  }

	  /**
	   * Gets the viewable height of the Element as an integer value in pixels. This height includes border, padding and scroll bar but excludes the margins.
	   *
	   * @returns {number} The height as pixels (number) or a string.
	   */
	  getBorderedHeight() {
	    return this.domElement.offsetHeight;
	  }

	  /**
	   * Gets the width of the Element as an integer value. This width includes border, padding and scroll bar but excludes the margins.
	   *
	   * @returns {number} The height in pixels.
	   */
	  getBorderedWidth() {
	    return this.domElement.offsetWidth;
	  }

	  /**
	   * @returns {number} The bottom position (in pixels) of the current element.
	   */
	  getBottom() {
	    return this.domElement.getBoundingClientRect().bottom;
	  }

	  /**
	   * Gets the children elements of this Element, optionally reduced to those matching the optional selector.
	   *
	   * @param {string} [selector] The selector. Optional, if not provided all children will be returned.
	   * @returns {PrimeElementList} The children.
	   */
	  getChildren(selector) {
	    if (!Utils.isDefined(selector)) {
	      return new PrimeElementList(this.domElement.children);
	    }

	    const matched = [];
	    for (let i = 0; i < this.domElement.children.length; i++) {
	      const child = this.domElement.children[i];
	      if (child.matches(selector)) {
	        matched.push(child);
	      }
	    }

	    return new PrimeElementList(matched);
	  }

	  /**
	   * Gets the class value of the current element. This might be a single class or multiple class names.
	   *
	   * @returns {string} The class.
	   */
	  getClass() {
	    return this.domElement.className;
	  }

	  /**
	   * Gets the computed style information for this Element.
	   *
	   * @returns {IEElementStyle|CSSStyleDeclaration} The style information.
	   */
	  getComputedStyle() {
	    return (this.domElement.currentStyle) ? this.domElement.currentStyle : document.defaultView.getComputedStyle(this.domElement, null);
	  }

	  /**
	   * Calculates the location of this element with respect to the document rather than the elements parent, offset parent
	   * or scroll position.
	   *
	   * @returns {{top: number, left: number}}
	   */
	  getCoordinates() {
	    const box = this.domElement.getBoundingClientRect();

	    const body = document.body;
	    const documentElement = document.documentElement;

	    const scrollTop = window.pageYOffset || documentElement.scrollTop || body.scrollTop;
	    const scrollLeft = window.pageXOffset || documentElement.scrollLeft || body.scrollLeft;

	    const clientTop = documentElement.clientTop || body.clientTop || 0;
	    const clientLeft = documentElement.clientLeft || body.clientLeft || 0;

	    const top = box.top + scrollTop - clientTop;
	    const left = box.left + scrollLeft - clientLeft;

	    return {top: Math.round(top), left: Math.round(left)};
	  }

	  /**
	   * Returns the dataset if it exists, otherwise, this creates a new dataset object and returns it.
	   *
	   * @returns {object} This dataset object.
	   */
	  getDataSet() {
	    if (this.domElement.dataset) {
	      return this.domElement.dataset;
	    }

	    this.domElement.dataset = {};
	    const attrs = this.getAttributes();
	    for (let prop in attrs) {
	      if (attrs.hasOwnProperty(prop) && prop.indexOf('data-') === 0) {
	        const dataName = prop.substring(5).replace(/-([a-z])/g, function(g) {
	          return g[1].toUpperCase();
	        });
	        this.domElement.dataset[dataName] = attrs[prop];
	      }
	    }
	    return this.domElement.dataset;
	  }

	  /**
	   * Returns the data value if it exists, otherwise returns null.
	   *
	   * @param {string} name The attribute name.
	   * @returns {string} This attribute value or null.
	   */
	  getDataAttribute(name) {
	    return this.getDataSet()[name] || null;
	  }

	  /**
	   * Get the first child element of this Element, optionally filtered using the optional selector.
	   *
	   * @param {string} [selector] The selector. Optional.
	   * @returns {PrimeElement} The first child element or null if the element has no children or a selector was provided and nothing matched the selector..
	   */
	  getFirstChild(selector) {
	    const lastChild = this.getChildren(selector)[0];
	    if (!Utils.isDefined(lastChild)) {
	      return null;
	    }
	    return lastChild;
	  }

	  /**
	   * Gets the viewable height of the Element as an integer value in pixels. This height includes padding and scroll bar
	   * but excludes the margin and borders. This is often called the innerHeight of the element.
	   *
	   * @returns {number} The height as pixels (number) or a string.
	   */
	  getHeight() {
	    const computedStyle = this.getComputedStyle();
	    const offsetHeight = this.domElement.offsetHeight;
	    const borderTop = computedStyle['borderTopWidth'];
	    const borderBottom = computedStyle['borderBottomWidth'];
	    return offsetHeight - Utils.parseCSSMeasure(borderTop) - Utils.parseCSSMeasure(borderBottom);
	  }

	  /**
	   * Gets the inner HTML content of the Element.
	   *
	   * @returns {string} The HTML content.
	   */
	  getHTML() {
	    return this.domElement.innerHTML;
	  }

	  /**
	   * Gets the ID of this element from the domElement.
	   *
	   * @returns {string} ID The id of the domElement if it exists.
	   */
	  getId() {
	    return this.domElement.id;
	  }

	  /**
	   * Get the last child element of this Element, optionally filtered using the optional selector.
	   *
	   * @param {string} [selector] The selector. Optional.
	   * @returns {PrimeElement} The last child element or null if the element has no children or a selector was provided and nothing matched the selector..
	   */
	  getLastChild(selector) {
	    const elementList = this.getChildren(selector);
	    if (elementList.length > 0) {
	      return elementList[elementList.length - 1];
	    }
	    return null;
	  }

	  /**
	   * @returns {number} The left position (in pixels) of the current element.
	   */
	  getLeft() {
	    return this.domElement.getBoundingClientRect().left;
	  }

	  /**
	   * @returns {PrimeElement} This elements next sibling or null.
	   */
	  getNextSibling() {
	    let sibling = this.domElement.nextSibling;
	    while (sibling !== null && sibling.nodeType !== 1) {
	      sibling = sibling.nextSibling;
	    }

	    if (sibling === null) {
	      return null;
	    }

	    return new PrimeElement(sibling);
	  }

	  /**
	   * The elements offset left in pixels.
	   *
	   * @returns {number} The offset left.
	   */
	  getOffsetLeft() {
	    return this.domElement.offsetLeft;
	  }

	  /**
	   * The elements offset top in pixels.
	   *
	   * @returns {number} The offset top.
	   */
	  getOffsetTop() {
	    return this.domElement.offsetTop;
	  }

	  /**
	   * Retrieves the opacity value for the Element. This handles the IE alpha filter.
	   *
	   * @returns {number} The opacity value.
	   */
	  getOpacity() {
	    const computedStyle = this.getComputedStyle();
	    let opacity = 1.0;
	    if (Browser.name === 'Explorer' && Browser.version < 9) {
	      const filter = computedStyle['filter'];
	      if (filter !== undefined && filter !== '') {
	        const matches = PrimeElement.ieAlphaRegexp.match(filter);
	        if (matches.length > 0) {
	          opacity = parseFloat(matches[0]);
	        }
	      }
	    } else {
	      opacity = parseFloat(computedStyle['opacity']);
	    }

	    return opacity;
	  }

	  /**
	   * @returns {PrimeElementList} If this element is a select box, this returns the options of the select box in
	   *          an ElementList.
	   */
	  getOptions() {
	    if (this.getTagName() !== 'SELECT') {
	      throw new TypeError('You can only get the options for select elements');
	    }

	    return new PrimeElementList(this.domElement.options);
	  }

	  /**
	   * Gets the outer height of the element, including the margins. This does not include the padding or borders.
	   *
	   * @returns {number} The outer height of the element.
	   */
	  getOuterHeight() {
	    const computedStyle = this.getComputedStyle();
	    const offsetHeight = this.domElement.offsetHeight;
	    const marginTop = computedStyle['marginTop'];
	    const marginBottom = computedStyle['marginBottom'];
	    return offsetHeight + Utils.parseCSSMeasure(marginTop) + Utils.parseCSSMeasure(marginBottom);
	  }

	  /**
	   * Gets the outer HTML content of the Element.
	   *
	   * @returns {string} The outer HTML content.
	   */
	  getOuterHTML() {
	    return this.domElement.outerHTML;
	  }

	  /**
	   * Gets the outer width of the element, including the margins. This does not include the padding or borders.
	   *
	   * @returns {number} The outer width of the element.
	   */
	  getOuterWidth() {
	    const computedStyle = this.getComputedStyle();
	    const offsetWidth = this.domElement.offsetWidth;
	    const marginLeft = computedStyle['marginLeft'];
	    const marginRight = computedStyle['marginRight'];
	    return offsetWidth + Utils.parseCSSMeasure(marginLeft) + Utils.parseCSSMeasure(marginRight);
	  }

	  /**
	   * Returns this element's parent as a PrimeElement.
	   *
	   * @returns {PrimeElement} This element's parent or null if there is no parent
	   */
	  getParent() {
	    if (Utils.isDefined(this.domElement.parentElement)) {
	      return new PrimeElement(this.domElement.parentElement);
	    } else {
	      return null;
	    }
	  }

	  /**
	   * @returns {PrimeElement} This elements previous sibling or null.
	   */
	  getPreviousSibling() {
	    let sibling = this.domElement.previousSibling;
	    while (sibling !== null && sibling.nodeType !== 1) {
	      sibling = sibling.previousSibling;
	    }

	    if (sibling === null) {
	      return null;
	    }

	    return new PrimeElement(sibling);
	  }

	  /**
	   * @returns {number} The zIndex style of this element based on the element or the first positioned parent.
	   */
	  getRelativeZIndex() {
	    let e = this;
	    while (e !== null && e.getComputedStyle()['zIndex'] === 'auto') {
	      e = e.getParent();
	    }
	    return e === null ? 0 : parseInt(e.getComputedStyle()['zIndex']);
	  }

	  /**
	   * @returns {number} The right position (in pixels) of the current element.
	   */
	  getRight() {
	    return this.domElement.getBoundingClientRect().right;
	  }

	  /**
	   * @returns {number} The scroll height of this element.
	   */
	  getScrollHeight() {
	    return this.domElement.scrollHeight;
	  }

	  /**
	   * @returns {number} The scroll left position of this element.
	   */
	  getScrollLeft() {
	    return this.domElement.scrollLeft;
	  }

	  /**
	   * @returns {number} The scroll top position of this element.
	   */
	  getScrollTop() {
	    return this.domElement.scrollTop;
	  }

	  /**
	   * @returns {number} The scroll width of this element.
	   */
	  getScrollWidth() {
	    return this.domElement.scrollWidth;
	  }

	  /**
	   * Retrieves the selected texts of this Element, if the element is a select. If it is anything else this returns
	   * null.
	   *
	   * @returns {Array} The texts of this Element.
	   */
	  getSelectedTexts() {
	    let texts;
	    if (this.domElement.tagName === 'SELECT') {
	      texts = [];
	      for (let i = 0; i < this.domElement.options.length; i++) {
	        if (this.domElement.options[i].selected) {
	          texts.push(this.domElement.options[i].text);
	        }
	      }
	    } else {
	      texts = null;
	    }

	    return texts;
	  }

	  /**
	   * Retrieves the values of this Element, if the element is a checkbox or select. If it is anything else this returns
	   * null.
	   *
	   * @returns {Array} The values of this Element.
	   */
	  getSelectedValues() {
	    let values;
	    if (this.domElement.tagName === 'INPUT' && (this.domElement.type === 'checkbox' || this.domElement.type === 'radio')) {
	      values = [];
	      const name = this.domElement.name;
	      const form = PrimeDocument.queryUp('form', this.domElement);
	      PrimeDocument.query('input[name="' + name + '"]', form).each(function(element) {
	        if (element.isChecked()) {
	          values.push(element.getValue());
	        }
	      });
	    } else if (this.domElement.tagName === 'SELECT') {
	      values = [];
	      for (let i = 0; i < this.domElement.length; i++) {
	        if (this.domElement.options[i].selected) {
	          values.push(this.domElement.options[i].value);
	        }
	      }
	    } else {
	      values = null;
	    }

	    return values;
	  }

	  /**
	   * Gets value of a style attribute.
	   *
	   * @returns {string} The style value.
	   */
	  getStyle(name) {
	    name = Utils.convertStyleName(name);
	    return this.domElement.style[name];
	  }

	  /**
	   * @returns {string} The tag name of this element as a string. This is always uppercase.
	   */
	  getTagName() {
	    return this.domElement.tagName;
	  }

	  /**
	   * Retrieves the text content of this Element.
	   *
	   * @returns {string} The text contents of this Element.
	   */
	  getTextContent() {
	    return this.domElement.innerText ? this.domElement.innerText : this.domElement.textContent;
	  }

	  /**
	   * @returns {number} The top position (in pixels) of the current element.
	   */
	  getTop() {
	    return this.domElement.getBoundingClientRect().top;
	  }

	  /**
	   * Gets the width of the Element as an integer value. This width includes padding and scroll bar but excludes the margin and borders.
	   * This is often called the innerWidth of the element.
	   *
	   * @returns {number} The height in pixels.
	   */
	  getWidth() {
	    const computedStyle = this.getComputedStyle();
	    const offsetWidth = this.domElement.offsetWidth;
	    const borderLeft = computedStyle['borderLeftWidth'];
	    const borderRight = computedStyle['borderRightWidth'];
	    return offsetWidth - Utils.parseCSSMeasure(borderLeft) - Utils.parseCSSMeasure(borderRight);
	  }

	  /**
	   * Retrieves the value attribute of this Element. This works on all checkboxes, radio buttons, text, text areas, and
	   * options. However, this does not retrieve the selected options in a select box, checked checkboxes or checked radio
	   * buttons. Use the getSelectedValues function for that.
	   *
	   * @returns {string} The value of this Element.
	   */
	  getValue() {
	    return this.domElement.value;
	  }

	  /**
	   * Returns true if the element has one or all class names
	   *
	   * @param {string} classNames The class name(s) in a string.
	   * @returns {boolean} True if all class names are present.
	   */
	  hasClass(classNames) {
	    const currentClassNames = this.domElement.className;
	    if (currentClassNames === '') {
	      return classNames === '';
	    }

	    const currentClassNameList = currentClassNames.split(Utils.spaceRegex);
	    const findClassNameList = classNames.split(Utils.spaceRegex);
	    for (let i = 0; i < findClassNameList.length; i++) {
	      if (currentClassNameList.indexOf(findClassNameList[i]) === -1) {
	        return false;
	      }
	    }

	    return true;
	  }

	  /**
	   * Hides the Element by setting the display style to none.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  hide() {
	    this.domElement.style.display = 'none';
	    return this;
	  }

	  /**
	   * Inserts this Element into the DOM after the given element, removing it from it's parent if it's an existing element.
	   *
	   * @param {PrimeElement|Element} element The element to insert this Element after.
	   * @returns {PrimeElement} This Element.
	   */
	  insertAfter(element) {
	    if (this.domElement.parentNode) {
	      this.domElement.parentNode.removeChild(this.domElement);
	    }

	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    const parentElement = domElement.parentNode;
	    if (parentElement) {
	      parentElement.insertBefore(this.domElement, domElement.nextSibling);
	    } else {
	      throw new TypeError('The element you passed into insertAfter is not in the DOM. You can\'t insert a PrimeElement after an element that isn\'t in the DOM yet.');
	    }

	    return this;
	  }

	  /**
	   * Inserts this Element into the DOM before the given element, removing it from it's parent if it's an existing element.
	   *
	   * @param {PrimeElement|Element} element The element to insert this Element before.
	   * @returns {PrimeElement} This Element.
	   */
	  insertBefore(element) {
	    if (this.domElement.parentNode) {
	      this.domElement.parentNode.removeChild(this.domElement);
	    }

	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    const parentElement = domElement.parentNode;
	    if (parentElement) {
	      parentElement.insertBefore(this.domElement, domElement);
	    } else {
	      throw new TypeError('The element you passed into insertBefore is not in the DOM. You can\'t insert a PrimeElement before an element that isn\'t in the DOM yet.');
	    }

	    return this;
	  }

	  /**
	   * Inserts the given HTML snippet directly after this element.
	   *
	   * @param {string} html The HTML string.
	   * @returns {PrimeElement} This Element.
	   */
	  insertHTMLAfter(html) {
	    this.domElement.insertAdjacentHTML('afterend', html);
	    return this;
	  }

	  /**
	   * Inserts the given HTML snippet inside this element, before its first child.
	   *
	   * @param {string} html The HTML string.
	   * @returns {PrimeElement} This Element.
	   */
	  insertHTMLAfterBegin(html) {
	    this.domElement.insertAdjacentHTML('afterbegin', html);
	    return this;
	  }

	  /**
	   * Inserts the given text after this Element.
	   *
	   * @param {string} text The text to insert.
	   * @returns {PrimeElement} This Element.
	   */
	  insertTextAfter(text) {
	    if (!this.domElement.parentNode) {
	      throw new TypeError('This Element is not currently in the DOM');
	    }

	    const textNode = document.createTextNode(text);
	    this.domElement.parentNode.insertBefore(textNode, this.domElement.nextSibling);

	    return this;
	  }

	  /**
	   * Inserts the given text before this Element.
	   *
	   * @param {string} text The text to insert.
	   * @returns {PrimeElement} This Element.
	   */
	  insertTextBefore(text) {
	    if (!this.domElement.parentNode) {
	      throw new TypeError('This Element is not currently in the DOM');
	    }

	    const textNode = document.createTextNode(text);
	    this.domElement.parentNode.insertBefore(textNode, this.domElement);

	    return this;
	  }

	  /**
	   * Returns true if the element matches the provided selector.
	   *
	   * @param {string} selector to match against the Element
	   * @returns {boolean} True if the element matches the selector, false if it does not match the selector.
	   */
	  is(selector) {
	    return this.domElement.matches(selector);
	  }

	  /**
	   * Returns whether or not the element is checked. If the element is not a checkbox or a radio this returns false.
	   *
	   * @returns {boolean} True if the element is selected, false if it isn't or is not a checkbox or a radio.
	   */
	  isChecked() {
	    return this.domElement.tagName === 'INPUT' && (this.domElement.type === 'checkbox' || this.domElement.type === 'radio') && this.domElement.checked;
	  }

	  /**
	   * Determines if this element is a child of the given element.
	   *
	   * @param {PrimeElement|Node} element The element to check to see if this element is a child of.
	   * @returns {boolean} True if this element is a child of the given element, false otherwise.
	   */
	  isChildOf(element) {
	    const domElement = element instanceof PrimeElement ? element.domElement : element;
	    let parent = this.domElement.parentNode;
	    while (domElement !== parent && parent !== null) {
	      parent = parent.parentNode;
	    }

	    return domElement === parent;
	  }

	  /**
	   * @returns {boolean} Whether or not this element is disabled according to the disabled property.
	   */
	  isDisabled() {
	    return this.domElement.disabled;
	  }

	  /**
	   * @returns {boolean} True if this element has focus.
	   */
	  isFocused() {
	    return document.activeElement === this.domElement;
	  }

	  /**
	   * @return {boolean} True if this element is an INPUT, SELECT or TEXTAREA.
	   */
	  isInput() {
	    const tagName = this.getTagName();
	    return tagName === 'SELECT' || tagName === 'INPUT' || tagName === 'TEXTAREA';
	  }

	  /**
	   * Determines if the this element is inside the given element
	   *
	   * @param target {PrimeElement} The target element.
	   * @returns {boolean} True if this element is inside the given element.
	   */
	  isInside(target) {
	    if (this.domElement === document.body || this.domElement === document.documentElement || this.domElement === document) {
	      return false;
	    }

	    let parent = this.getParent();
	    while (parent.domElement !== document.body) {
	      if (parent.domElement === target.domElement) {
	        return true;
	      }
	      parent = parent.getParent();
	    }

	    return false;
	  }

	  /**
	   * Returns whether or not the element is selected. If the element is not an option this returns false.
	   *
	   * @returns {boolean} True if the element is selected, false if it isn't or is not an option.
	   */
	  isSelected() {
	    return this.domElement.tagName === 'OPTION' && this.domElement.selected;
	  }

	  /**
	   * Determines if the element is visible using its display and visibility styles.
	   *
	   * @returns {boolean} True if the element is visible, false otherwise. This might return an invalid value if the element
	   * is absolutely positioned and off the screen, but is still technically visible.
	   */
	  isVisible() {
	    const computedStyle = this.getComputedStyle();
	    return computedStyle['display'] !== 'none' && computedStyle['visibility'] !== 'hidden';
	  }

	  /**
	   * Inserts this Element (which must be a newly created Element) into the DOM inside at the very beginning of the given
	   * element.
	   *
	   * @param {PrimeElement|Element} element The element to insert this Element into.
	   * @returns {PrimeElement} This Element.
	   */
	  prependTo(element) {
	    // Error out for now if this element is in the document so we can punt on cloning for now
	    if (this.domElement.parentNode) {
	      throw new TypeError('You can only insert new PrimeElements for now');
	    }

	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    if (domElement.parentNode) {
	      domElement.insertBefore(this.domElement, domElement.firstChild);
	    } else {
	      throw new TypeError('The element you passed into prependTo is not in the DOM. You can\'t insert a PrimeElement inside an element that isn\'t in the DOM yet.');
	    }

	    return this;
	  }

	  /**
	   * Queries the DOM using the given selector starting at this element and returns all the matched elements.
	   *
	   * @param {string} selector The selector.
	   * @returns {PrimeElementList} An element list.
	   */
	  query(selector) {
	    return PrimeDocument.query(selector, this);
	  }

	  /**
	   * Queries the DOM using the given selector starting at this element and returns the first matched element
	   * or null if there aren't any matches.
	   *
	   * @param {string} selector The selector.
	   * @returns {PrimeElement} An element or null.
	   */
	  queryFirst(selector) {
	    return PrimeDocument.queryFirst(selector, this);
	  }

	  /**
	   * Queries the DOM using the given selector starting at this element and returns the last matched element
	   * or null if there aren't any matches.
	   *
	   * @param {string} selector The selector.
	   * @returns {PrimeElement} An element or null.
	   */
	  queryLast(selector) {
	    return PrimeDocument.queryLast(selector, this);
	  }

	  /**
	   * Traverses up the DOM from this element and looks for a match to the selector.
	   *
	   * @param {string} selector The selector.
	   * @returns {PrimeElement} An element or null.
	   */
	  queryUp(selector) {
	    return PrimeDocument.queryUp(selector, this);
	  }

	  /**
	   * Removes all of the event listeners for the given element.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  removeAllEventListeners() {
	    for (let event in this.domElement.eventListeners) {
	      if (this.domElement.eventListeners.hasOwnProperty(event)) {
	        for (let i = 0; i < this.domElement.eventListeners[event].length; i++) {
	          this._internalRemoveEventListener(event, this.domElement.eventListeners[event][i]);
	        }
	      }
	    }

	    this.domElement.eventListeners = {};
	    this.domElement.customEventListeners = {};

	    return this;
	  }

	  /**
	   * Removes an attribute from the Element
	   *
	   * @param {string} name The name of the attribute.
	   * @returns {PrimeElement} This Element.
	   */
	  removeAttribute(name) {
	    this.domElement.removeAttribute(name);
	    return this;
	  }

	  /**
	   * Removes the given class (or list of space separated classes) from this Element.
	   *
	   * @param {string} classNames The class name(s).
	   * @returns {PrimeElement} This Element.
	   */
	  removeClass(classNames) {
	    const currentClassName = this.domElement.className;
	    if (currentClassName === '') {
	      return this;
	    }

	    const currentClassNameList = currentClassName.split(Utils.spaceRegex);
	    const removeClassNameList = classNames.split(Utils.spaceRegex);
	    for (let i = 0; i < removeClassNameList.length; i++) {
	      Utils.removeFromArray(currentClassNameList, removeClassNameList[i]);
	    }

	    this.domElement.className = currentClassNameList.join(' ');
	    return this;
	  }

	  /**
	   * Removes an event listener for a specific event from this Element, you must have attached using addEventListener
	   *
	   * @param {string} event The name of the event.
	   * @param {*} listener The event listener that was bound.
	   * @returns {PrimeElement} This Element.
	   */
	  removeEventListener(event, listener) {
	    let listeners;
	    if (event.indexOf(':') === -1) {
	      this._internalRemoveEventListener(event, listener);
	      listeners = this.domElement.eventListeners[event];
	    } else {
	      listeners = this.domElement.customEventListeners[event];
	    }

	    if (listeners) {
	      Utils.removeFromArray(listeners, listener);
	    }

	    return this;
	  }

	  /**
	   * Removes all of the event listeners for the given event from this element.
	   *
	   * @param {string} event The name of the event to remove the listeners for.
	   * @returns {PrimeElement} This Element.
	   */
	  removeEventListeners(event) {
	    if (event.indexOf(':') === -1) {
	      if (this.domElement.eventListeners[event]) {
	        for (let i = 0; i < this.domElement.eventListeners[event].length; i++) {
	          this._internalRemoveEventListener(event, this.domElement.eventListeners[event][i]);
	        }

	        delete this.domElement.eventListeners[event];
	      }
	    } else {
	      if (this.domElement.customEventListeners[event]) {
	        delete this.domElement.customEventListeners[event];
	      }
	    }

	    return this;
	  }

	  /**
	   * Removes all of the event listeners for the given pattern from this element.
	   *
	   * @param {RegExp} pattern The regular expression that matches the names of the events to remove the listeners for.
	   * @returns {PrimeElement} This Element.
	   */
	  removeEventListenersByPattern(pattern) {
	    for (let event in this.domElement.eventListeners) {
	      if (this.domElement.eventListeners.hasOwnProperty(event) && pattern.test(event)) {
	        for (let i = 0; i < this.domElement.eventListeners[event].length; i++) {
	          this._internalRemoveEventListener(event, this.domElement.eventListeners[event][i]);
	        }

	        delete this.domElement.eventListeners[event];
	      }
	    }

	    for (let event in this.domElement.customEventListeners) {
	      if (this.domElement.customEventListeners.hasOwnProperty(event) && pattern.test(event)) {
	        delete this.domElement.customEventListeners[event];
	      }
	    }

	    return this;
	  }

	  /**
	   * Removes this Element from the DOM. If the Element isn't in the DOM this does nothing.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  removeFromDOM() {
	    if (this.domElement.parentNode) {
	      this.domElement.parentNode.removeChild(this.domElement);
	    }

	    return this;
	  }

	  /**
	   * Create a selected range for this element.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  selectElementContents() {
	    let range;
	    let selection;

	    if (document.body.createTextRange) {
	      /* IE */
	      range = document.body.createTextRange();
	      range.moveToElementText(this.domElement);
	      range.select();
	    } else if (window.getSelection) {
	      /* Rest of the world */
	      selection = window.getSelection();
	      range = document.createRange();
	      range.selectNodeContents(this.domElement);
	      selection.removeAllRanges();
	      selection.addRange(range);
	    }

	    return this;
	  }

	  /**
	   * Scrolls this Element into the visible area of the browser window.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  scrollIntoView() {
	    this.domElement.scrollIntoView();
	    return this;
	  }

	  /**
	   * Scrolls this element to the given horizontal position.
	   *
	   * @param {number} position The position to scroll the element to.
	   * @returns {PrimeElement} This Element.
	   */
	  scrollLeftTo(position) {
	    this.domElement.scrollLeft = position;
	    return this;
	  }

	  /**
	   * Scrolls this element to the given vertical position.
	   *
	   * @param {number} position The position to scroll the element to.
	   * @returns {PrimeElement} This Element.
	   */
	  scrollTo(position) {
	    this.domElement.scrollTop = position;
	    return this;
	  }

	  /**
	   * Scrolls this element to the bottom.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  scrollToBottom() {
	    this.domElement.scrollTop = this.domElement.scrollHeight;
	    return this;
	  }

	  /**
	   * Scrolls this element to the top.
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  scrollToTop() {
	    this.domElement.scrollTop = 0;
	    return this;
	  }

	  /**
	   * Sets an attribute of the Element.
	   *
	   * @param {string} name The attribute name
	   * @param {number|string} value The attribute value
	   * @returns {PrimeElement} This Element.
	   */
	  setAttribute(name, value) {
	    if (typeof value === 'number') {
	      value = value.toString();
	    }
	    if (this.domElement.setAttribute) {
	      this.domElement.setAttribute(name, value);
	    } else {
	      const attribute = document.createAttribute(name);
	      attribute.nodeValue = value;
	      this.domElement.setAttributeNode(attribute);
	    }

	    return this;
	  }

	  /**
	   * Sets a data- attribute of the Element.
	   *
	   * Example: setDataAttribute('fooBar', 'baz');
	   *  is equivalent to calling setAttribute('data-foo-bar', 'baz');
	   *
	   * @param {string} name The attribute name
	   * @param {number|string} value The attribute value
	   * @returns {PrimeElement} This Element.
	   */
	  setDataAttribute(name, value) {
	    const dataName = 'data-' + name.replace(/([a-z])([A-Z])/g, "$1-$2").toLowerCase();
	    return this.setAttribute(dataName, value);
	  }

	  /**
	   * Sets multiple attributes of the Element from the hash
	   *
	   * @param {Object} attributes An object of key value style pairs.
	   * @returns {PrimeElement} This Element.
	   */
	  setAttributes(attributes) {
	    for (let key in attributes) {
	      if (attributes.hasOwnProperty(key)) {
	        this.setAttribute(key, attributes[key]);
	      }
	    }
	    return this;
	  }

	  /**
	   * If this element is a checkbox or radio button, this sets the checked field on the DOM object equal to the given
	   * value.
	   *
	   * @param {boolean} value The value to set the checked state of this element to.
	   * @returns {PrimeElement} This Element.
	   */
	  setChecked(value) {
	    this.domElement.checked = value;
	    return this;
	  }

	  /**
	   * Sets if this element is disabled or not. This works with any element that responds to the disabled property.
	   *
	   * @param {boolean} value The value to set the disabled state of this element to.
	   * @returns {PrimeElement} This Element.
	   */
	  setDisabled(value) {
	    this.domElement.disabled = value;
	    return this;
	  }

	  /**
	   * Sets the height of this element using the height style.
	   *
	   * @param {number|string} height The new height as a number (for pixels) or string.
	   * @returns {PrimeElement} This Element.
	   */
	  setHeight(height) {
	    if (typeof(height) === 'number') {
	      height = height + 'px';
	    }

	    this.setStyle('height', height);
	    return this;
	  }

	  /**
	   * Sets the inner HTML content of the Element.
	   *
	   * @param {string|PrimeElement} newHTML The new HTML content for the Element.
	   * @returns {PrimeElement} This Element.
	   */
	  setHTML(newHTML) {
	    if (newHTML !== null) {
	      if (newHTML instanceof PrimeElement) {
	        this.domElement.innerHTML = newHTML.getHTML();
	      } else {
	        this.domElement.innerHTML = newHTML;
	      }
	    }
	    return this;
	  }

	  /**
	   * Sets the ID of the Element.
	   *
	   * @param {string} id The ID.
	   * @returns {PrimeElement} This Element.
	   */
	  setId(id) {
	    this.domElement.id = id;
	    return this;
	  }

	  /**
	   * Sets left position of the element.
	   *
	   * @param {number|string} left The left position of the element in pixels or as a string.
	   * @returns {PrimeElement} This Element.
	   */
	  setLeft(left) {
	    let leftString = left;
	    if (typeof(left) === 'number') {
	      leftString = left + 'px';
	    }

	    this.setStyle('left', leftString);
	    return this;
	  }

	  /**
	   * Sets the opacity of the element. This also sets the IE alpha filter for IE version 9 or younger.
	   *
	   * @param {number} opacity The opacity.
	   * @returns {PrimeElement} This Element.
	   */
	  setOpacity(opacity) {
	    if (Browser.name === 'Explorer' && Browser.version < 9) {
	      this.domElement.style.filter = 'alpha(opacity=' + opacity + ')';
	    } else {
	      this.domElement.style.opacity = opacity;
	    }

	    return this;
	  }

	  /**
	   * Sets the selected value on the element. If the element is not an option or radio, this does nothing.
	   *
	   * @param {boolean} selected Selected value.
	   */
	  setSelected(selected) {
	    this.domElement.selected = selected;
	  }

	  /**
	   * Sets the selected value(s) of this element. This works on selects, checkboxes, and radio buttons.
	   *
	   * @param {string} [arguments] The value(s) to select (var args).
	   * @returns {PrimeElement} This Element.
	   */
	  setSelectedValues() {
	    // Handle the case where they passed in an array
	    let values = null;
	    if (arguments.length === 1 && Utils.isArray(arguments[0])) {
	      values = arguments[0];
	    } else {
	      values = Array.prototype.slice.call(arguments, 0);
	    }

	    if (this.domElement.tagName === 'INPUT' && (this.domElement.type === 'checkbox' || this.domElement.type === 'radio')) {
	      const name = this.domElement.name;
	      const form = PrimeDocument.queryUp('form', this.domElement);
	      PrimeDocument.query('input[name="' + name + '"]', form).each(function(element) {
	        element.setChecked(values.indexOf(element.getValue()) !== -1);
	      });
	    } else if (this.domElement.tagName === 'SELECT') {
	      for (let i = 0; i < this.domElement.length; i++) {
	        this.domElement.options[i].selected = values.indexOf(this.domElement.options[i].value) !== -1;
	      }
	    }

	    return this;
	  }

	  /**
	   * Sets the style for the name of this Element.
	   *
	   * @param {string} name The style name.
	   * @param {number|string} value The style value.
	   * @returns {PrimeElement} This Element.
	   */
	  setStyle(name, value) {
	    if (typeof value === 'number') {
	      value = value.toString();
	    }
	    this.domElement.style[name] = value;
	    return this;
	  }

	  /**
	   * Sets multiple styles of this Element.
	   *
	   * @param {Object} styles An object with key value pairs for the new style names and values.
	   * @returns {PrimeElement} This Element.
	   */
	  setStyles(styles) {
	    for (let key in styles) {
	      if (styles.hasOwnProperty(key)) {
	        this.setStyle(key, styles[key]);
	      }
	    }
	    return this;
	  }

	  /**
	   * Sets the textContent of the Element.
	   *
	   * @param {number|string|PrimeElement} newText The new text content for the Element.
	   * @returns {PrimeElement} This Element.
	   */
	  setTextContent(newText) {
	    if (newText !== null) {
	      if (newText instanceof PrimeElement) {
	        this.domElement.textContent = newText.getTextContent();
	      } else {
	        if (typeof newText === 'number') {
	          newText = newText.toString();
	        }
	        this.domElement.textContent = newText;
	      }
	    }
	    return this;
	  }

	  /**
	   * Sets top position of the element.
	   *
	   * @param {number|string} top The top position of the element in pixels or as a string.
	   * @returns {PrimeElement} This Element.
	   */
	  setTop(top) {
	    let topString = top;
	    if (typeof(top) === 'number') {
	      topString = top + 'px';
	    }

	    this.setStyle('top', topString);
	    return this;
	  }

	  /**
	   * Sets the value of this Element. This handles checkboxes, radio buttons, options, text inputs and text areas. This
	   * works on checkboxes and radio buttons, but it change the value attribute on them rather than checking and unchecking
	   * the buttons themselves. To check and uncheck the buttons, use the select method.
	   *
	   * @param {number|string} value The new value.
	   * @returns {PrimeElement} This Element.
	   */
	  setValue(value) {
	    if (typeof value === 'number') {
	      value = value.toString();
	    }
	    this.domElement.value = value;
	    return this;
	  }

	  /**
	   * Sets the width of this element using the height style.
	   *
	   * @param {number|string} width The new width as a number (for pixels) or string.
	   * @returns {PrimeElement} This Element.
	   */
	  setWidth(width) {
	    if (typeof(width) === 'number') {
	      width = width + 'px';
	    }

	    this.setStyle('width', width);
	    return this;
	  }

	  /**
	   * Shows the Element by setting the display style first to empty string. After this, the elements computed style is
	   * checked to see if the element is still not visible. If that is true, the element must have a CSS style defined in
	   * a stylesheet that is setting it to display: none. In this case, we determine if the element is a block level element
	   * and either set the display to 'block' or 'inline'.
	   *
	   * @param {string} [displayValue] The display value to use for the show. This defaults to the W3C standard display
	   * setting depending on the type of element you are showing. For example, INPUT is inline and DIV is block.
	   * @returns {PrimeElement} This Element.
	   */
	  show(displayValue) {
	    if (Utils.isDefined(displayValue)) {
	      this.domElement.style.display = displayValue;
	      return this;
	    }

	    this.domElement.style.display = '';

	    const computedDisplay = this.getComputedStyle()['display'];
	    if (computedDisplay === 'none') {
	      if (!Utils.isDefined(displayValue)) {
	        displayValue = (PrimeElement.blockElementRegexp.test(this.domElement.tagName)) ? 'block' : 'inline';
	      }

	      this.domElement.style.display = displayValue;
	    }

	    return this;
	  }

	  /**
	   * Toggle the class on this element.
	   *
	   * @param {string} className The class name.
	   * @returns {PrimeElement} This Element.
	   */
	  toggleClass(className) {
	    if (this.hasClass(className)) {
	      this.removeClass(className);
	    } else {
	      this.addClass(className);
	    }

	    return this;
	  }

	  /**
	   * Removes this element from the DOM while preserving the inner HTML.
	   *
	   * Example, call unwrap on the italic element:
	   *   <strong>Hello</strong><italic> World </italic> --> <strong>Hello</strong> World
	   *
	   * @returns {PrimeElement} This Element.
	   */
	  unwrap() {
	    const parent = this.getParent().domElement;
	    while (this.domElement.firstChild) {
	      parent.insertBefore(this.domElement.firstChild, this.domElement);
	    }

	    this.removeFromDOM();
	  }

	  /**
	   * Builds a new element using the given HTML snippet (currently this only supports the tag).
	   *
	   * @param {string} elementString The element string.
	   * @param {Object} [properties={}] The properties for the new element.
	   * @returns {PrimeElement} A new PrimeElement.
	   */
	  wrapInnerHTML(elementString, properties) {
	    const element = PrimeDocument.newElement(elementString, properties);
	    element.setHTML(this.getOuterHTML());
	    this.domElement.outerHTML = element.domElement.outerHTML;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  /**
	   * Handle delegated events.
	   *
	   * @param {Event} event
	   * @private
	   */
	  _handleDelegatedEvent(event) {
	    this._callMatchedListeners(event, event.target);

	    // If the event cannot bubble, we are done.
	    if (!event.bubbles) {
	      return;
	    }

	    // While propagation has not been stopped, walk up the tree to simulate the bubble
	    let domElement = event.target;
	    while (true) {
	      if (domElement === this.domElement || domElement === document || event.isPropagationStopped) {
	        break;
	      }

	      domElement = domElement.parentNode;
	      this._callMatchedListeners(event, domElement);
	    }
	  }

	  _callMatchedListeners(event, target) {
	    // Test each selector we have against this target
	    let delegatedListeners = this.domElement.delegatedListeners[event.type] || [];
	    for (let selector in delegatedListeners) {
	      if (delegatedListeners.hasOwnProperty(selector)) {
	        if (target.matches(selector)) {

	          let scopedListeners = delegatedListeners[selector];
	          // Call each listener unless immediate propagation has been stopped
	          for (let i = 0; i < scopedListeners.length; i++) {

	            if (event.isImmediatePropagationStopped) {
	              return;
	            }

	            scopedListeners[i](event, target);
	          }
	        }
	      }
	    }
	  }

	  /**
	   * Removes the event listener proxy from this element.
	   *
	   * @param {string} event The event name.
	   * @param {Function} listener The listener function.
	   * @private
	   */
	  _internalRemoveEventListener(event, listener) {
	    if (event.indexOf(':') === -1) {
	      // Traditional event
	      if (this.domElement.removeEventListener) {
	        this.domElement.removeEventListener(event, listener, false);
	      } else if (this.domElement.detachEvent) {
	        this.domElement.detachEvent('on' + event, listener);
	      } else {
	        throw new TypeError('Unable to remove event from the element. Neither removeEventListener nor detachEvent methods are available');
	      }
	    } else if (this.domElement.customEventListeners && this.domElement.customEventListeners[event]) {
	      // Custom event
	      const customListeners = this.domElement.customEventListeners[event];
	      Utils.removeFromArray(customListeners, listener);
	    }
	  }
	}


	/* ===================================================================================================================
	 * Polyfill
	 * ===================================================================================================================*/

	(function() {
	  if (!Element.prototype.matches) {
	    Element.prototype.matches = function(selector) {
	      const domElement = this;
	      const matches = (domElement.parentNode || domElement.document).querySelectorAll(selector);
	      let i = 0;

	      while (matches[i] && matches[i] !== domElement) {
	        i++;
	      }

	      return !!matches[i];
	    };
	  }

	  // Add isPropagationStopped, this is part of the DOM Level 3 spec for CustomEvents
	  // - adding it here for all types so it can be used by the delegation event listener
	  // https://www.w3.org/TR/2003/NOTE-DOM-Level-3-Events-20031107/events.html#Events-Event-isPropagationStopped
	  if (!Event.prototype.isPropagationStopped) {
	    let _stopPropagation = Event.prototype.stopPropagation;
	    Event.prototype.stopPropagation = function() {
	      this.isPropagationStopped = true;
	      _stopPropagation.apply(this, arguments);
	    };
	  }

	  // Add isImmediatePropagationStopped, this is part of the DOM Level 3 spec for CustomEvents
	  // - adding it here for all types so it can be used by the delegation event listener
	  // https://www.w3.org/TR/2003/NOTE-DOM-Level-3-Events-20031107/events.html#Events-Event-isImmediatePropagationStopped
	  if (!Event.prototype.isImmediatePropagationStopped) {
	    let _stopImmediatePropagation = Event.prototype.stopImmediatePropagation;
	    Event.prototype.stopImmediatePropagation = function() {
	      this.isImmediatePropagationStopped = true;
	      _stopImmediatePropagation.apply(this, arguments);
	    };
	  }
	})();

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class PrimeElementList {
	  /**
	   * Constructs an PrimeElementList object using the given array containing DOMElements or PrimeElement objects, or the NodeList containing Node objects.
	   *
	   * @constructor
	   * @param {Array|NodeList} elements An array containing DOMElement or PrimeElement objects, or a NodeList containing Node objects.
	   */
	  constructor(elements) {
	    Utils.bindAll(this);

	    // NodeList does not inherit from Array so do not assume object type.
	    this.length = elements.length;
	    for (let i = 0; i < elements.length; i++) {
	      if (elements[i] instanceof PrimeElement) {
	        this[i] = elements[i];
	      } else {
	        this[i] = new PrimeElement(elements[i]);
	      }
	    }
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.addClass} on each Element in the PrimeElementList.
	   *
	   * Adds the given class (or list of space separated classes) to all Elements in this PrimeElementList.
	   *
	   * @param {string} classNames The class name(s) separated by a space.
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  addClass(classNames) {
	    return this._proxyToElement('addClass', classNames);
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.addEventListener} on each Element in the PrimeElementList.
	   *
	   * Attaches an event listener to all Elements in this PrimeElementList.
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The event listener function.
	   * @returns {PrimeElement|PrimeElementList} This Element.
	   */
	  addEventListener(event, listener) {
	    return this._proxyToElement('addEventListener', event, listener);
	  }

	  /**
	   * Iterates over each of the PrimeElement objects in this PrimeElementList and calls the given function for each one.
	   * The <code>this</code> variable inside the function will be managed by the caller of this method. You should use the
	   * <code>bind</code> method on the Function object if you want to manage the <code>this</code> reference.
	   *
	   * The function can optionally take two parameters. The first parameter is the current element. The second parameter
	   * is the current index.
	   *
	   * @param {Function} iterationFunction The function to call.
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  each(iterationFunction) {
	    for (let i = 0; i < this.length; i++) {
	      iterationFunction(this[i], i);
	    }

	    return this;
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.hide} on each Element in the PrimeElementList.
	   *
	   * Hides the Element by setting the display style to none.
	   *
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  hide() {
	    return this._proxyToElement('hide');
	  }

	  /**
	   * Returns the indexOf the element that matches the parameter, either Prime Element or DOMElement.
	   *
	   * @param {PrimeElement|Element} element The element to look for
	   * @returns {number} The position of the element in the list, or -1 if not present.
	   */
	  indexOf(element) {
	    const domElement = (element instanceof PrimeElement) ? element.domElement : element;

	    for (let i = 0; i < this.length; i++) {
	      if (this[i].domElement === domElement) {
	        return i;
	      }
	    }

	    return -1;
	  }

	  /**
	   * Removes all the matched elements in the PrimeElementList from the DOM.
	   *
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  removeAllFromDOM() {
	    for (let i = 0; i < this.length; i++) {
	      this[i].removeFromDOM();
	    }

	    return this;
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.removeClass} on each Element in the PrimeElementList.
	   *
	   * Removes the given class (or list of space separated classes) from all Elements in this PrimeElementList.
	   *
	   * @param {string} classNames The class name(s) separated by a space.
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  removeClass(classNames) {
	    return this._proxyToElement('removeClass', classNames);
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.setChecked} on each Element in the PrimeElementList.
	   *
	   * If this element is a checkbox or radio button, this sets the checked field on the DOM object equal to the given
	   * value.
	   *
	   * @param {boolean} value The value to set the checked state of this element to.
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  setChecked(value) {
	    return this._proxyToElement('setChecked', value);
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.setDisabled} on each Element in the PrimeElementList.
	   *
	   * Sets if this element is disabled or not. This works with any element that responds to the disabled property.
	   *
	   * @param {boolean} value The value to set the disabled state of this element to.
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  setDisabled(value) {
	    return this._proxyToElement('setDisabled', value);
	  }

	  /**
	   * Shorthand for calling {@link PrimeElement.show} on each Element in the PrimeElementList.
	   *
	   * Shows the element.
	   *
	   * @returns {PrimeElementList} This PrimeElementList.
	   */
	  show() {
	    return this._proxyToElement('show');
	  }

	  /**
	   * @callback PrimeElementListPredicate
	   *
	   * A function that defines a condition on a PrimeElement
	   *
	   * @param {PrimeElement} element
	   * @returns {boolean} True if the element matches a condition
	   */

	  /**
	   * A function that tests for any element that matches a condition.
	   * @param {PrimeElementListPredicate} predicate A function that defines the condition to check
	   * @returns {boolean} True if any element matches the predicate
	   */
	  some(predicate) {
	    for (let i = 0; i < this.length; ++i) {
	      if (predicate(this[i])) {
	        return true;
	      }
	    }
	    return false;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Proxy function calls to each Element in the PrimeElementList.
	   * The first parameter is function name, followed by a variable length of arguments.
	   *
	   * Example usage: this._proxyToElement('addClass', classNames);
	   *
	   * @returns {PrimeElementList}
	   * @private
	   */
	  _proxyToElement() {
	    const args = Array.prototype.slice.apply(arguments);
	    for (let i = 0; i < this.length; i++) {
	      this[i][args[0]].apply(this[i], args.slice(1));
	    }
	    return this;
	  }
	}

	/*
	 * Copyright (c) 2012-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const readyFunctions = [];
	const tagRegexp = /^<(\w+)\s*\/?>.*(?:<\/\1>)?$/;

	/**
	 * The Body element as a PrimeElement object.
	 *
	 * @type {?PrimeElement}
	 */
	let bodyElement = null;

	class PrimeDocument {

	  /**
	   * @returns {PrimeElement}
	   */
	  static get Element() {
	    return PrimeElement;
	  }

	  /**
	   * @returns {PrimeElementList}
	   */
	  static get ElementList() {
	    return PrimeElementList;
	  }

	  /**
	   * @returns {Array<Function>}
	   */
	  static get readyFunctions() {
	    return readyFunctions;
	  }

	  /**
	   * @returns {RegExp}
	   */
	  static get tagRegexp() {
	    return tagRegexp;
	  }

	  /**
	   * @returns {?PrimeElement} the Prime body element
	   */
	  static get bodyElement() {
	    if (bodyElement === null) {
	      bodyElement = new PrimeElement(document.body);
	    }
	    return bodyElement;
	  }

	  /**
	   * Set the body element
	   * @param {?PrimeElement} body the Prime body element
	   */
	  static set bodyElement(body) {
	    bodyElement = body;
	  }

	  /**
	   * Attaches an event listener to the document body and will only invoke the listener when the event target matches
	   * the provided selector.
	   *
	   * The intent of this function is to provide a delegated listener and handle events from nested elements.
	   *
	   * @param {string} event The name of the event
	   * @param  {string} selector The selector to match against the Element
	   * @param {PrimeDelegationEventListener} listener The event listener function
	   */
	  static addDelegatedEventListener(event, selector, listener) {
	    PrimeDocument.bodyElement.addDelegatedEventListener(event, selector, listener);
	  }

	  /**
	   * Attaches an event listener to the document, returning the handler proxy.
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The event listener function.
	   * @returns {PrimeDocument} The PrimeDocument object so you can chain method calls together.
	   */
	  static addEventListener(event, listener) {
	    if (event.indexOf(':') === -1) {
	      // Traditional event
	      document.eventListeners = document.eventListeners || {};
	      document.eventListeners[event] = document.eventListeners[event] || [];
	      document.eventListeners[event].push(listener);
	      document.addEventListener(event, listener, false);
	    } else {
	      // Custom event
	      document.customEventListeners = document.customEventListeners || {};
	      document.customEventListeners[event] = document.customEventListeners[event] || [];
	      document.customEventListeners[event].push(listener);
	    }

	    return PrimeDocument;
	  }

	  /**
	   * Returns the height of the document.
	   *
	   * @returns {number} The height of the document in pixels.
	   */
	  static getHeight() {
	    const body = document.body;
	    const html = document.documentElement;

	    return Math.max(body.scrollHeight, body.offsetHeight, html.clientHeight, html.scrollHeight, html.offsetHeight);
	  }

	  /**
	   * Returns the width of the document.
	   *
	   * @returns {number} The width of the document in pixels.
	   */
	  static getWidth() {
	    const body = document.body;
	    const html = document.documentElement;

	    return Math.max(body.scrollWidth, body.offsetWidth, html.clientWidth, html.scrollWidth, html.offsetWidth);
	  }

	  /**
	   * Builds a new document using the given HTML document.
	   *
	   * @param {string} documentString The HTML string to build the document.
	   * @returns {Document} A new document.
	   */
	  static newDocument(documentString) {
	    return new DOMParser().parseFromString(documentString, "text/html");
	  }

	  /**
	   * Builds a new element using the given HTML snippet (currently this only supports the tag).
	   *
	   * @param {string} elementString The element string.
	   * @param {Object} [properties={}] The properties for the new element.
	   * @returns {PrimeElement} A new PrimeElement.
	   */
	  static newElement(elementString, properties) {
	    properties = Utils.isDefined(properties) ? properties : {};
	    const result = PrimeDocument.tagRegexp.exec(elementString);
	    if (result === null) {
	      throw new TypeError('Invalid string to create a new element [' + elementString + ']. It should look like <a/>');
	    }

	    const element = new PrimeElement(document.createElement(result[1]));
	    for (let key in properties) {
	      if (properties.hasOwnProperty(key)) {
	        if (key === 'id') {
	          element.setId(properties[key]);
	        } else {
	          element.setAttribute(key, properties[key]);
	        }
	      }
	    }

	    return element;
	  }

	  /**
	   * Adds the given callback function to the list of functions to invoke when the document is ready. If the document is
	   * already fully loaded, this simply invokes the callback directly.
	   *
	   * @param {Function} callback The callback function.
	   */
	  static onReady(callback) {
	    if (document.readyState === 'complete') {
	      callback();
	    } else {
	      // If this is the first call, register the event listener on the document
	      if (this.readyFunctions.length === 0) {
	        if (document.addEventListener) {
	          document.addEventListener('DOMContentLoaded', PrimeDocument._callReadyListeners, false);
	        } else if (document.attachEvent) {
	          document.attachEvent('onreadystatechange', PrimeDocument._callReadyListeners);
	        } else {
	          throw new TypeError('No way to attach an event to the document. What browser are you running?');
	        }
	      }

	      // Add the callback
	      PrimeDocument.readyFunctions.push(callback);
	    }
	  }

	  /**
	   * Take the HTML string and append it to the body.
	   *
	   * @param {string} html The HTML to append
	   */
	  static appendHTML(html) {
	    document.body.insertAdjacentHTML('beforeend', html);
	  }

	  /**
	   * Moves the given element by appending it to the element provided by the second argument.
	   *
	   * @param {Element|PrimeElement} element The element to move.
	   * @param {Element|PrimeElement} appendToElement [appendToElement=body] The element to append to, defaults to the body if not provided.
	   * @returns {PrimeElement} The element that has been moved.
	   */
	  static move(element, appendToElement) {
	    element = (element instanceof PrimeElement) ? element : new PrimeElement(element);

	    if (!Utils.isDefined(appendToElement)) {
	      appendToElement = new PrimeElement(document.body);
	    } else {
	      appendToElement = (appendToElement instanceof PrimeElement) ? appendToElement : new PrimeElement(appendToElement);
	    }

	    appendToElement.appendHTML(element.getOuterHTML());
	    element.removeFromDOM();
	    return appendToElement.getLastChild();
	  }

	  /**
	   * Queries the DOM using the given selector starting at the given element and returns all the matched elements.
	   *
	   * @param {string} selector The selector.
	   * @param {Element|Document|PrimeElement} [element=document] The starting point for the search (defaults to document if not provided).
	   * @returns {PrimeElementList} An element list.
	   */
	  static query(selector, element) {
	    let domElement = null;
	    if (!Utils.isDefined(element)) {
	      domElement = document;
	    } else {
	      domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    }

	    const elements = domElement.querySelectorAll(selector);
	    return new PrimeElementList(elements);
	  }

	  /**
	   * Queries the DOM for an element that has the given ID.
	   *
	   * @param {string} id The ID.
	   * @returns {PrimeElement} The element or null.
	   */
	  static queryById(id) {
	    let element = document.getElementById(id);
	    if (!element) {
	      return null;
	    }

	    return new PrimeElement(element);
	  }

	  /**
	   * Queries the DOM using the given selector starting at the given element and returns the first matched element
	   * or null if there aren't any matches.
	   *
	   * @param {string} selector The selector.
	   * @param {Element|Document|PrimeElement} [element=document] The starting point for the search (defaults to document if not provided).
	   * @returns {PrimeElement} An element or null.
	   */
	  static queryFirst(selector, element) {
	    let domElement = null;
	    if (!Utils.isDefined(element)) {
	      domElement = document;
	    } else {
	      domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    }

	    domElement = domElement.querySelector(selector);
	    if (domElement === null) {
	      return null;
	    }

	    return new PrimeElement(domElement);
	  }

	  /**
	   * Queries the DOM using the given selector starting at the given element and returns the last matched element
	   * or null if there aren't any matches.
	   *
	   * @param {string} selector The selector.
	   * @param {Element|Document|PrimeElement} [element=document] The starting point for the search (defaults to document if not provided).
	   * @returns {PrimeElement} An element or null.
	   */
	  static queryLast(selector, element) {
	    let domElement = null;
	    if (!Utils.isDefined(element)) {
	      domElement = document;
	    } else {
	      domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    }

	    const domElements = domElement.querySelectorAll(selector);
	    if (domElements.length === 0) {
	      return null;
	    }

	    return new PrimeElement(domElements[domElements.length - 1]);
	  }

	  /**
	   * Traverses up the DOM from the starting element and looks for a match to the selector.
	   *
	   * @param {string} selector The selector.
	   * @param {PrimeElement|Element} element The starting point for the upward traversal.
	   * @returns {PrimeElement} An element or null.
	   */
	  static queryUp(selector, element) {
	    let domElement = null;
	    if (!Utils.isDefined(element)) {
	      throw new SyntaxError('Missing required parameter. The element is required.');
	    } else {
	      domElement = (element instanceof PrimeElement) ? element.domElement : element;
	    }

	    domElement = domElement.parentNode;
	    while (domElement !== null && !domElement.matches(selector)) {
	      domElement = domElement.parentNode;
	      if (domElement === document) {
	        domElement = null;
	      }
	    }

	    if (domElement !== null) {
	      return new PrimeElement(domElement);
	    }

	    return null;
	  }

	  /**
	   * Removes an event handler for a specific event from the document that you attached using addEventListener
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The listener function.
	   */
	  static removeEventListener(event, listener) {
	    if (document.removeEventListener) {
	      document.removeEventListener(event, listener, false);
	    } else if (document.detachEvent) {
	      document.detachEvent('on' + event, listener);
	    } else {
	      throw new TypeError('Unable to remove event from the element. Neither removeEventListener nor detachEvent methods are available');
	    }
	  }

	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  /**
	   * Calls all the registered document ready listeners.
	   *
	   * @private
	   */
	  static _callReadyListeners() {
	    if (document.addEventListener || document.readyState === 'complete') {
	      let readyFunction;
	      while (readyFunction = PrimeDocument.readyFunctions.shift()) {
	        readyFunction();
	      }
	    }

	    if (document.removeEventListener) {
	      document.removeEventListener('DOMContentLoaded', PrimeDocument._callReadyListeners, false);
	    } else {
	      document.detachEvent('onreadystatechange', PrimeDocument._callReadyListeners);
	    }
	  }
	}

	/* ===================================================================================================================
	 * Polyfill
	 * ===================================================================================================================*/

	/* https://developer.mozilla.org/en-US/docs/Web/API/DOMParser */
	(function(DOMParser) {
	  const proto = DOMParser.prototype;
	  const nativeParse = proto.parseFromString;

	  // Firefox/Opera/IE throw errors on unsupported types
	  try {
	    // WebKit returns null on unsupported types
	    if ((new DOMParser()).parseFromString('', 'text/html')) {
	      // text/html parsing is natively supported
	      return;
	    }
	  } catch (ex) {
	  }

	  proto.parseFromString = function(markup, type) {
	    if (/^\s*text\/html\s*(?:;|$)/i.test(type)) {
	      const doc = document.implementation.createHTMLDocument('');
	      if (markup.toLowerCase().indexOf('<!doctype') > -1) {
	        doc.documentElement.innerHTML = markup;
	      } else {
	        doc.body.innerHTML = markup;
	      }
	      return doc;
	    } else {
	      return nativeParse.apply(this, arguments);
	    }
	  };
	}(DOMParser));

	/*
	 * Copyright (c) 2012-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	/**
	 * The Utils namespace. This contains utility functions.
	 *
	 * @namespace Utils
	 */
	const Utils = {
	  spaceRegex: /\s+/,
	  typeRegex: /^\[object\s(.*)\]$/,

	  /**
	   * Binds all of the functions inside the object so that <code>this</code> is the object. This is extremely useful for
	   * objects that have functions that will be used as event listeners. Rather than having to manage binding and function
	   * references manually you can instead bind all of the functions in the object and then use them directly for event
	   * listeners.
	   *
	   * Here's an example:
	   *
	   * <pre>
	   *   function Foo() {
	   *     Utils.bindAll(this);
	   *
	   *     // This function is bound to this (i.e. this.handleClick = this.handleClick.bind(this)).
	   *     PrimeDocument.queryFirst('a').addEventListener('click', this.handleClick);
	   *   }
	   *
	   *   Foo.prototype = {
	   *     handleClick: function(event) {
	   *       ...
	   *     }
	   *   };
	   * </pre>
	   *
	   * @param {*} object The object to bind all the functions for.
	   */
	  bindAll: function(object) {
	    Utils.getAllPropertyNames(object).forEach((property) => {
	      if (property !== 'constructor' && typeof object[property] === 'function' &&
	          !(object[property].name && object[property].name.startsWith('bound '))) { // name isn't defined in ie
	        Object.defineProperty(object, property, {value: object[property].bind(object)});
	      }
	    });
	  },

	  /**
	   * HTML escape a string.
	   *
	   * @param string The string to escape
	   * @returns {string} the escaped string
	   */
	  escapeHTML: function(string) {
	    let div = document.createElement('div');
	    div.appendChild(document.createTextNode(string));
	    return div.innerHTML;
	  },

	  /**
	   * Returns all of the properties for this object and all of its
	   * inherited properties from parent objects.
	   *
	   * @param object
	   * @returns {Array<string>}
	   */
	  getAllPropertyNames: function(object) {
	    let props = {};

	    do {
	      Object.getOwnPropertyNames(object).forEach((prop) => {
	        if (!props[prop]) {
	          props[prop]=prop;
	        }
	      });
	    } while (object = Object.getPrototypeOf(object));

	    return Object.keys(props);
	  },

	  /**
	   * Binds all of the functions inside the object so that <code>this</code> is the object. This is extremely useful for
	   * objects that have functions that will be used as event listeners. Rather than having to manage binding and function
	   * references manually you can instead bind all of the functions in the object and then use them directly for event
	   * listeners.
	   *
	   * Here's an example:
	   *
	   * <pre>
	   *   function Foo() {
	   *     Utils.bindAll(this);
	   *
	   *     // This function is bound to this (i.e. this.handleClick = this.handleClick.bind(this)).
	   *     PrimeDocument.queryFirst('a').addEventListener('click', this.handleClick);
	   *   }
	   *
	   *   Foo.prototype = {
	   *     handleClick: function(event) {
	   *       ...
	   *     }
	   *   };
	   * </pre>
	   *
	   * @param {*} object The object to bind all the functions for.
	   * @param {String} arguments A varargs list of function names to bind.
	   */
	  bindSome: function(object) {
	    if (arguments.length > 1) {
	      for (let i = 1; i < arguments.length; i++) {
	        const func = object[arguments[i]];
	        if (!Utils.isDefined(func) || !(func instanceof Function)) {
	          throw new TypeError('The object does not contain a function named [' + arguments[i] + ']');
	        }

	        object[arguments[i]] = func.bind(object);
	      }
	    }
	    for (let property in object) {
	      if (object[property] instanceof Function) {
	        object[property] = object[property].bind(object);
	      }
	    }
	  },

	  /**
	   * Safely binds a function to a context.
	   *
	   * @param {Function} func The function to bind.
	   * @param {Object} [context] An optional context to bind the function to.
	   * @returns {Function} Either <code>func</code> or the newly bound function.
	   */
	  bindSafe: function(func, context) {
	    if (!Utils.isDefined(func)) {
	      throw new Error('Invalid arguments');
	    }

	    if (!Utils.isDefined(context)) {
	      return func;
	    }

	    return func.bind(context);
	  },

	  /**
	   * Calculates the length of the given text using the style of the given element.
	   *
	   * @param {PrimeElement} element The element to use the style of.
	   * @param {string} text The text to calculate the length of.
	   * @returns {number} The length of the text.
	   */
	  calculateTextLength: function(element, text) {
	    const computedStyle = element.getComputedStyle();
	    let textCalculator = PrimeDocument.queryById('prime-text-calculator');
	    if (textCalculator === null) {
	      textCalculator = PrimeDocument.newElement('<span/>')
	          .setStyles({
	            position: 'absolute',
	            width: 'auto',
	            fontSize: computedStyle['fontSize'],
	            fontFamily: computedStyle['fontFamily'],
	            fontWeight: computedStyle['fontWeight'],
	            letterSpacing: computedStyle['letterSpacing'],
	            whiteSpace: 'nowrap'
	          })
	          .setId('prime-text-calculator')
	          .setTop(-9999)
	          .setLeft(-9999)
	          .appendTo(document.body);
	    }

	    textCalculator.setHTML(text);
	    return textCalculator.getWidth();
	  },

	  /**
	   * Attempts to invoke a function iteratively in the background a specific number of times within a specific duration.
	   * This might not complete in the specified duration. The functions passed in should be short functions that return as
	   * quickly as possible. If you are using long functions, use the recursive setTimeout trick by-hand instance.
	   *
	   * @param {number} totalDuration The duration in milliseconds.
	   * @param {number} timesToCall The number of times to call the function.
	   * @param {Function} stepFunction The step function to call each iteration.
	   * @param {Function} [endFunction] The function to invoke at the end.
	   */
	  callIteratively: function(totalDuration, timesToCall, stepFunction, endFunction) {
	    const step = totalDuration / timesToCall;
	    let count = 0;
	    const id = setInterval(function() {
	      count++;
	      const last = (count >= timesToCall);
	      stepFunction(last);
	      if (last) {
	        clearInterval(id);

	        if (Utils.isDefined(endFunction)) {
	          endFunction();
	        }
	      }
	    }, step - 1);
	  },

	  /**
	   * Capitalizes the given String.
	   *
	   * @param {string} str The String to capitalize.
	   * @returns {string} The capitalized String.
	   */
	  capitalize: function(str) {
	    return str.charAt(0).toUpperCase() + str.substring(1);
	  },

	  /**
	   * Converts CSS style names to style JavaScript names.
	   *
	   * @param {string} name The CSS style name to convert
	   * @returns {string} The converted style name.
	   */
	  convertStyleName: function(name) {
	    if (name === 'float') {
	      return 'cssFloat';
	    }

	    let dash = name.indexOf('-');
	    if (dash === -1) {
	      return name;
	    }

	    let start = 0;
	    let result = '';
	    while (dash !== -1) {
	      const piece = name.substring(start, dash);
	      if (start === 0) {
	        result = result.concat(piece);
	      } else {
	        result = result.concat(Utils.capitalize(piece));
	      }

	      start = dash + 1;
	      dash = name.indexOf('-', start);
	    }

	    return result + Utils.capitalize(name.substring(start));
	  },

	  /**
	   * Return an options map {Object} of the data set values coerced to a typed value of boolean, string or number.
	   *
	   * @param {PrimeElement} element The element.
	   * @returns {Object} The options object.
	   */
	  dataSetToOptions: function(element) {
	    const options = {};
	    const data = element.getDataSet();
	    for (let prop in data) {
	      if (!data.hasOwnProperty(prop)) {
	        continue;
	      }
	      const value = data[prop];
	      if (isNaN(value)) {
	        if (value === 'true') {
	          options[prop] = true;
	        } else if (value === 'false') {
	          options[prop] = false;
	        } else {
	          options[prop] = value;
	        }
	      } else {
	        options[prop] = parseInt(value);
	      }
	    }

	    return options;
	  },

	  /**
	   * Determines if an object is an array or not.
	   *
	   * @param {*} o The object to check.
	   * @returns {boolean} True if the object is an array, false otherwise.
	   */
	  isArray: function(o) {
	    return Object.prototype.toString.call(o) === '[object Array]';
	  },

	  /**
	   * Tests whether or not the value is not null and not 'undefined'.
	   *
	   * @param {*} value The value to test.
	   * @returns {boolean} True if the value is defined (not null or undefined).
	   */
	  isDefined: function(value) {
	    return value !== null && typeof(value) !== 'undefined';
	  },

	  /**
	   * Left pad a number.
	   * @param {number} number the number to pad
	   * @param {number} width the width of the final result
	   * @returns {string}
	   */
	  leftPadNumber: function(number, width) {
	    const sign = Math.sign(number) === -1 ? '-' : '';
	    return sign + new Array(width).concat([Math.abs(number)]).join('0').slice(-width);
	  },

	  /**
	   * Parses a CSS measure value (12px) as an integer.
	   *
	   * @param {string} measure The CSS measure
	   * @returns {number} The value as an integer.
	   */
	  parseCSSMeasure: function(measure) {
	    const index = measure.indexOf('px');
	    if (index > 0) {
	      return parseInt(measure.substring(0, measure.length - 2));
	    }

	    return parseInt(measure) || 0;
	  },

	  /**
	   * Parses JSON.
	   *
	   * @param {string} json The JSON.
	   * @returns {Object} The JSON data as an object.
	   */
	  parseJSON: function(json) {
	    return JSON.parse(json);
	  },

	  /**
	   * Removes the objects in the toRemove array from the fromArray.
	   *
	   * @param {Array} fromArray The array to remove from.
	   * @param {Array} toRemove The values to remove.
	   */
	  removeAllFromArray: function(fromArray, toRemove) {
	    for (let i = 0; i < toRemove.length; i++) {
	      Utils.removeFromArray(fromArray, toRemove[i]);
	    }
	  },

	  /**
	   * Removes the given object from the given array.
	   *
	   * @param {Array} array The array to remove from.
	   * @param {*} obj The object to remove.
	   */
	  removeFromArray: function(array, obj) {
	    const index = array.indexOf(obj);
	    if (index !== -1) {
	      const shift = array.splice(index + 1, array.length);
	      array.length = index;
	      array.push.apply(array, shift);
	    }
	  },

	  /**
	   * Helper function to provide a one liner to behave as if you returned 'false' from a legacy version of Prime.js.
	   *
	   * Calling this method is equivalent to calling event.preventDefault and event.stopPropagation.
	   * @param event
	   */
	  stopEvent: function(event) {
	    // Compatibility with older versions of IE
	    event.cancelBubble = true;
	    if (event.stopPropagation) {
	      event.stopPropagation();
	    }
	    if (event.preventDefault) {
	      event.preventDefault();
	    }
	  },

	  type: function(object) {
	    return Object.prototype.toString.call(object).match(Utils.typeRegex)[1];
	  }
	};

	/*
	 * Copyright (c) 2017-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */

	/**
	 * The singleton instance.
	 *
	 * @type {Overlay}
	 */
	let instance;

	class Overlay {
	  /**
	   * Constructs a new Overlay instance once per DOM.
	   *
	   * @constructor
	   */
	  constructor() {
	    Utils.bindAll(this);

	    // Check if the overlay doesn't exist and add it
	    this.overlay = PrimeDocument.queryById('prime-overlay');
	    if (this.overlay === null) {
	      this.overlay = PrimeDocument.newElement('<div/>').setId('prime-overlay').appendTo(document.body).hide();
	    }
	    this.bodyOverflow = null;
	  }

	  /**
	   * Return the instance of the Overlay widget
	   * @returns {Overlay}
	   */
	  static get instance() {
	    return instance;
	  }

	  /**
	   * Set the instance value of the Overlay instance
	   * @param  value {Overlay}
	   */
	  static set instance(value) {
	    instance = value;
	  }

	  /**
	   * Closes the overlay and the target element.
	   */
	  close() {
	    // using null ensures that if this style is not defined, we'll remove it when we're done
	    let overflowStyle = this.bodyOverflow || '';
	    PrimeDocument.bodyElement.setStyle('overflow', overflowStyle);
	    this.overlay.setStyle('zIndex', '10');
	    this.overlay.hide();
	    return this;
	  }

	  /**
	   * Opens the overlay and positions the element over it.
	   * @param zIndex {Number|string}
	   */
	  open(zIndex) {
	    if (this.bodyOverflow === null) {
	      this.bodyOverflow = PrimeDocument.bodyElement.getStyle('overflow');
	    }
	    PrimeDocument.bodyElement.setStyle('overflow', 'hidden');
	    this.overlay.show();

	    // Set the z-index of this dialog and the overlay
	    this.overlay.setStyle('zIndex', zIndex.toString());
	    return this;
	  }

	  /**
	   * Changes the id of the Overlay element.
	   *
	   * @param id {string} The new id.
	   * @returns {Overlay}
	   */
	  setId(id) {
	    this.overlay.setId(id);
	    return this;
	  }

	  /**
	   * Updates the zindex of the overlay.
	   *
	   * @param zIndex {string|number} The new zIndex.
	   */
	  setZIndex(zIndex) {
	    this.overlay.setStyle('zIndex', zIndex.toString());
	    return this;
	  }
	}

	PrimeDocument.onReady(function() {
	  Overlay.instance = new Overlay();
	});

	/*
	 * Copyright (c) 2014-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class PrimeRequest {
	  /**
	   * Makes a new AJAX request.
	   *
	   * @constructor
	   * @param {string} [url] The URL to call. This can be left out for sub-classing but should otherwise be provided.
	   * @param {string} [method=GET] The HTTP method to use. You can specify GET, POST, PUT, DELETE, HEAD, SEARCH, etc.
	   */
	  constructor(url, method) {
	    Utils.bindAll(this);
	    this.xhr = new XMLHttpRequest();
	    this.async = true;
	    this.body = null;
	    this.queryParams = null;
	    this.contentType = null;
	    this.inProgress = null;
	    this.errorHandler = this.onError;
	    this.headers = {};
	    this.loadingHandler = this.onLoading;
	    this.method = method || 'GET';
	    this.openHandler = this.onOpen;
	    this.password = null;
	    this.sendHandler = this.onSend;
	    this.successHandler = this.onSuccess;
	    this.unsetHandler = this.onUnset;
	    this.url = url;
	    this.username = null;
	  }

	  /**
	   * Changes the URL to call.
	   *
	   * @param {string} url The new URL to call.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  forURL(url) {
	    this.url = url;
	    return this;
	  }

	  /**
	   * Invokes the AJAX request. If the URL is not set, this throws an exception.
	   *
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  go() {
	    if (!this.url) {
	      throw new TypeError('No URL set for AJAX request');
	    }

	    let requestUrl = this.url;
	    if ((this.method === 'GET' || this.method === 'DELETE') && this.queryParams !== null) {
	      if (requestUrl.indexOf('?') === -1) {
	        requestUrl += '?' + this.queryParams;
	      } else {
	        requestUrl += '&' + this.queryParams;
	      }
	    }

	    if (this.async) {
	      if (this.inProgress !== null) {
	        this.inProgress.open();
	      }

	      this.xhr.onreadystatechange = this._handler.bind(this);
	    }

	    this.xhr.open(this.method, requestUrl, this.async, this.username, this.password);

	    if (Object.keys(this.headers).length > 0) {
	      for (let key in this.headers) {
	        if (this.headers.hasOwnProperty(key)) {
	          this.xhr.setRequestHeader(key, this.headers[key]);
	        }
	      }
	    }

	    if (this.contentType) {
	      this.xhr.setRequestHeader('Content-Type', this.contentType);
	    }

	    this.xhr.send(this.body);

	    return this;
	  }

	  /**
	   * Default handler for the "completed" state and an HTTP response status of anything but 2xx. Sub-classes can override
	   * this handler or you can pass in a handler function to the {@link #withUnsetHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onError(xhr) {
	  }

	  /**
	   * Default handler for the "loading" state. Sub-classes can override this handler or you can pass in a handler function
	   * to the {@link #withLoadingHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onLoading(xhr) {
	  }

	  /**
	   * Default handler for the "open" state. Sub-classes can override this handler or you can pass in a handler function
	   * to the {@link #withOpenHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onOpen(xhr) {
	  }

	  /**
	   * Default handler for the "send" state. Sub-classes can override this handler or you can pass in a handler function
	   * to the {@link #withSendHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onSend(xhr) {
	  }

	  /**
	   * Default handler for the "complete" state and an HTTP response status of 2xx. Sub-classes can override this handler
	   * or you can pass in a handler function to the {@link #withUnsetHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onSuccess(xhr) {
	  }

	  /**
	   * Default handler for the "unset" state. Sub-classes can override this handler or you can pass in a handler function
	   * to the {@link #withUnsetHandler}.
	   *
	   * @param {XMLHttpRequest} xhr The XMLHttpRequest object.
	   */
	  onUnset(xhr) {
	  }

	  /**
	   * Sets the async flag to false.
	   *
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  synchronously() {
	    this.async = false;
	    return this;
	  }

	  /**
	   * Sets the method used to make the AJAX request.
	   *
	   * @param {string} method The HTTP method.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  usingMethod(method) {
	    this.method = method;
	    return this;
	  }

	  /**
	   * Sets the request body for the request.
	   *
	   * @param {string} body The request body.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withBody(body) {
	    this.body = body;
	    return this;
	  }

	  /**
	   * Sets the content type for the request.
	   *
	   * @param {string} contentType The contentType.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withContentType(contentType) {
	    this.contentType = contentType;
	    return this;
	  }

	  /**
	   * Sets the data object for the request. Will store the values for query parameters or post data depending on the
	   * method that is set.  If the method is a post or put, will also set content-type to x-www-form-urlencoded.
	   *
	   * @param {Object} data The data object.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withData(data) {
	    for (let prop in data) {
	      if (data.hasOwnProperty(prop)) {
	        if (this.method === 'PUT' || this.method === 'POST') {
	          this.body = this._addDataValue(this.body, prop, data[prop]);
	        } else {
	          this.queryParams = this._addDataValue(this.queryParams, prop, data[prop]);
	        }
	      }
	    }

	    if (this.method === "PUT" || this.method === "POST") {
	      this.contentType = 'application/x-www-form-urlencoded';
	    }
	    return this;
	  }

	  /**
	   * Sets the data for the request using the form fields in the given form element. Will store the values for query
	   * parameters or post data depending on the method that is set.  If the method is a post or put, will also set
	   * content-type to x-www-form-urlencoded.
	   *
	   * @param {PrimeElement|HTMLFormElement} form The form object.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withDataFromForm(form) {
	    let domElement = form;
	    if (form instanceof PrimeElement) {
	      domElement = form.domElement;
	    }

	    for (let i = 0; i < domElement.elements.length; i++) {
	      const primeElement = new PrimeElement(domElement.elements[i]);
	      if (primeElement.isDisabled() || !primeElement.isInput()) {
	        continue;
	      }

	      let type = primeElement.getAttribute('type');
	      if (type !== null) {
	        type = type.toLowerCase();
	      }

	      let values;
	      if (primeElement.getTagName() === 'SELECT') {
	        values = primeElement.getSelectedValues();
	      } else if ((type === 'radio' || type === 'checkbox') && !primeElement.isChecked()) {
	        continue;
	      } else {
	        values = primeElement.getValue();
	      }

	      const name = primeElement.domElement.name;
	      if (this.method === 'PUT' || this.method === 'POST') {
	        this.body = this._addDataValue(this.body, name, values);
	      } else {
	        this.queryParams = this._addDataValue(this.queryParams, name, values);
	      }
	    }

	    if (this.method === "PUT" || this.method === "POST") {
	      this.contentType = 'application/x-www-form-urlencoded';
	    }

	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "complete" and the HTTP status in the response is
	   * not 2xx.
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withErrorHandler(func) {
	    this.errorHandler = func;
	    return this;
	  }

	  /**
	   * Sets an InProgress object that will be called by this AJAX request.
	   *
	   * @param {InProgress} inProgress The InProgress object.
	   * @return {PrimeRequest} This.
	   */
	  withInProgress(inProgress) {
	    this.inProgress = inProgress;
	    return this;
	  }

	  /**
	   * Sets the body of the AJAX request to the string value of the provided JSON object. The content-type of the request
	   * will also be set to 'application/json'. The provided JSON object may be passed as a string or an object.
	   *
	   * @param {Object} json The JSON object.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withJSON(json) {
	    this.body = typeof(json) === String ? json : JSON.stringify(json);
	    this.contentType = 'application/json';
	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "loading".
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withLoadingHandler(func) {
	    this.loadingHandler = func;
	    return this;
	  }

	  /**
	   * Set the request headers using the key and value.
	   *
	   * @param {String} key The key name.
	   * @param {String} value The value.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withHeader(key, value) {
	    this.headers[key] = value;
	    return this;
	  }

	  /**
	   * Set the key value pairs provided as request headers.
	   *
	   * @param {Object} headers A map of key value pairs.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withHeaders(headers) {
	    for (let key in headers) {
	      if (headers.hasOwnProperty(key)) {
	        this.headers[key] = headers[key];
	      }
	    }
	    return this;
	  }

	  /**
	   * Sets the XMLHTTPRequest's response type field, which will control how the response is parsed.
	   *
	   * @param {string} responseType The response type.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withResponseType(responseType) {
	    this.xhr.responseType = responseType;
	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "open".
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withOpenHandler(func) {
	    this.openHandler = func;
	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "send".
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withSendHandler(func) {
	    this.sendHandler = func;
	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "complete" and the HTTP status in the response is
	   * 2xx.
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withSuccessHandler(func) {
	    this.successHandler = func;
	    return this;
	  }

	  /**
	   * Sets the handler to invoke when the state of the AJAX request is "unset".
	   *
	   * @param {Function} func The handler function.
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  withUnsetHandler(func) {
	    this.unsetHandler = func;
	    return this;
	  }

	  /**
	   * Resets the Request back to a base state (basically just the URL + method).  This can be
	   * useful if a component is going to make many requests to the same endpoint with different parameters.
	   *
	   * @returns {PrimeRequest} This PrimeRequest.
	   */
	  reset() {
	    this.queryParams = null;
	    this.data = null;
	    this.body = null;
	    this.contentType = null;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  // noinspection JSMethodCanBeStatic
	  /**
	   * Adds the given name-value pair to the given data String. If the value is an array, it adds multiple values for each
	   * piece. Otherwise, it assumes value is a String or can be converted to a String.
	   *
	   * @param {string} dataString The data String used to determine if an ampersand is necessary.
	   * @param {string} name The name of the name-value pair.
	   * @param {string|Array} value The value of the name-value pair.
	   * @returns {string} The new data string.
	   * @private
	   */
	  _addDataValue(dataString, name, value) {
	    let result = '';
	    if (value instanceof Array) {
	      for (let i = 0; i < value.length; i++) {
	        result += encodeURIComponent(name) + '=' + encodeURIComponent(value[i]);
	        if (i + 1 < value.length) {
	          result += '&';
	        }
	      }
	    } else {
	      result = encodeURIComponent(name) + '=' + encodeURIComponent(value);
	    }

	    if (dataString !== null && result !== '') {
	      result = dataString + '&' + result;
	    } else if (dataString !== null && result === '') {
	      result = dataString;
	    }

	    return result;
	  }

	  /**
	   * @private
	   */
	  _handler() {
	    if (this.xhr.readyState === 0) {
	      this.unsetHandler(this.xhr);
	    } else if (this.xhr.readyState === 1) {
	      this.openHandler(this.xhr);
	    } else if (this.xhr.readyState === 2) {
	      this.sendHandler(this.xhr);
	    } else if (this.xhr.readyState === 3) {
	      this.loadingHandler(this.xhr);
	    } else if (this.xhr.readyState === 4) {

	      // Call the InProgress before hand because the success handler might call another AJAX method that might open another InProgress
	      if (this.inProgress !== null) {
	        this.inProgress.close(function() {
	          if (this.xhr.status >= 200 && this.xhr.status <= 299) {
	            this.successHandler(this.xhr);
	          } else {
	            this.errorHandler(this.xhr);
	          }
	        }.bind(this));
	      } else {
	        if (this.xhr.status >= 200 && this.xhr.status <= 299) {
	          this.successHandler(this.xhr);
	        } else {
	          this.errorHandler(this.xhr);
	        }
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2012-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class PrimeWindow {
	  /**
	   * Attaches an event listener to the window, returning the handler proxy.
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The event handler.
	   * @returns {Window} The window object.
	   */
	  static addEventListener(event, listener) {
	    if (event.indexOf(':') === -1) {
	      window.eventListeners = window.eventListeners || {};
	      window.eventListeners[event] = window.eventListeners[event] || [];
	      window.eventListeners[event].push(listener);
	      if (window.addEventListener) {
	        window.addEventListener(event, listener, false);
	      } else if (window.attachEvent) {
	        window.attachEvent('on' + event, listener);
	      } else {
	        throw new TypeError('Unable to set event onto the window. Neither addEventListener nor attachEvent methods are available');
	      }
	    } else {
	      // Custom event
	      window.customEventListeners = window.customEventListeners || {};
	      window.customEventListeners[event] = window.customEventListeners[event] || [];
	      window.customEventListeners[event].push(listener);
	    }

	    return window;
	  }

	  /**
	   * Returns the inner height of the window. This includes only the rendering area and not the window chrome (toolbars,
	   * status bars, etc). If this method can't figure out the inner height, it throws an exception.
	   *
	   * @returns {number} The inner height of the window.
	   */
	  static getInnerHeight() {
	    if (typeof(window.innerHeight) === 'number') {
	      // Most browsers
	      return window.innerHeight;
	    } else if (document.documentElement && document.documentElement.clientHeight) {
	      // IE 6+ in 'standards compliant mode'
	      return document.documentElement.clientHeight;
	    } else if (document.body && document.body.clientHeight) {
	      // IE 4 compatible
	      return document.body.clientHeight;
	    }

	    throw new Error('Unable to determine inner height of the window');
	  }

	  /**
	   * Returns the inner width of the window. This includes only the rendering area and not the window chrome (toolbars,
	   * status bars, etc). If this method can't figure out the inner width, it throws an exception.
	   *
	   * @returns {number} The inner width of the window.
	   */
	  static getInnerWidth() {
	    if (typeof(window.innerWidth) === 'number') {
	      // Most browsers
	      return window.innerWidth;
	    } else if (document.documentElement && document.documentElement.clientWidth) {
	      // IE 6+ in 'standards compliant mode'
	      return document.documentElement.clientWidth;
	    } else if (document.body && document.body.clientWidth) {
	      // IE 4 compatible
	      return document.body.clientWidth;
	    }

	    throw new Error('Unable to determine inner width of the window');
	  }

	  /**
	   * Returns the number of pixels the Window is scrolled by.
	   *
	   * @returns {number} The number of pixels.
	   */
	  static getScrollTop() {
	    if (typeof(window.pageYOffset) === 'number') {
	      return window.pageYOffset;
	    } else if (document.body && document.body.scrollTop) {
	      return document.body.scrollTop;
	    } else if (document.documentElement && document.documentElement.scrollTop) {
	      return document.documentElement.scrollTop;
	    }

	    throw new Error('Unable to determine scrollTop of the window');
	  }
	}

	/*
	 * Copyright (c) 2015-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class Draggable {
	  /**
	   * Constructs a new Draggable object for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the Draggable widget.
	   * @param {string} [gripSelector=] gripSelector The optional selector to identify the 'grippy' part.
	   * @constructor
	   */
	  constructor(element, gripSelector) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this.offset = {};

	    if (!Utils.isDefined(gripSelector)) {
	      this.grip = this.element;
	    } else {
	      this.grip = this.element.queryFirst(gripSelector);
	      if (this.grip === null) {
	        throw Error('Unable to find an element using the provided selector [' + gripSelector + ']');
	      }
	    }
	  }

	  /**
	   * Destroys the Draggable Widget
	   */
	  destroy() {
	    this.element.removeClass('active');
	    this.element.setStyles(this.originalStyle);

	    this.grip.removeEventListener('mousedown', this._handleMouseDown);
	    PrimeDocument.removeEventListener('mousemove', this._handleMouseMove);
	    PrimeDocument.removeEventListener('mouseup', this._handleMouseUp);
	  }

	  /**
	   * Initializes the Draggable by attaching the event listeners.
	   *
	   * @returns {Draggable} This.
	   */
	  initialize() {
	    this.originalStyle = {
	      cursor: this.element.getStyle('cursor'),
	      zIndex: this.element.getStyle('zIndex')
	    };

	    this.grip.addEventListener('mousedown', this._handleMouseDown).setStyle('cursor', 'move');
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handle Mouse Down Event
	   * @param {Event} event The mouse event.
	   * @private
	   */
	  _handleMouseDown(event) {
	    event.preventDefault();
	    this.element.addClass('active');

	    this.offset = {
	      zIndex: this.element.getStyle('zIndex'),
	      height: this.element.getOuterHeight(),
	      width: this.element.getOuterWidth(),
	      x: event.screenX,
	      y: event.screenY
	    };

	    this.element.setStyle('zIndex', this.offset.zIndex + 10);

	    // Remove old listeners
	    PrimeDocument.removeEventListener('mousemove', this._handleMouseMove);
	    PrimeDocument.removeEventListener('mouseup', this._handleMouseUp);

	    // Attach all the events
	    PrimeDocument.addEventListener('mousemove', this._handleMouseMove);
	    PrimeDocument.addEventListener('mouseup', this._handleMouseUp);
	  }

	  /**
	   * Handle the Mouse Move event for the body element.
	   *
	   * @param {Event} event The mouse event.
	   * @private
	   */
	  _handleMouseMove(event) {
	    const xDiff = event.screenX - this.offset.x;
	    const yDiff = event.screenY - this.offset.y;
	    this.offset.x = event.screenX;
	    this.offset.y = event.screenY;
	    this.element.setLeft(this.element.getLeft() + xDiff);
	    this.element.setTop(this.element.getTop() + yDiff);
	  }

	  /**
	   * Handle the Mouse Up event for this draggable widget.
	   * @private
	   */
	  _handleMouseUp() {
	    PrimeDocument.removeEventListener('mousemove', this._handleMouseMove);
	    PrimeDocument.removeEventListener('mouseup', this._handleMouseUp);
	    this.element.setStyle('zIndex', this.offset.zIndex);
	    this.element.removeClass('active');
	  }
	}

	/*
	 * Copyright (c) 2017-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */

	class AJAXDialog {
	  /**
	   * Constructs a new dialog box, which is dynamically built and then populated with the HTML returned from an AJAX call.
	   *
	   * @constructor
	   */
	  constructor() {
	    Utils.bindAll(this);

	    this.draggable = null;
	    this.element = null;
	    this.initialized = false;
	    this._setInitialOptions();
	  }

	  /**
	   * Closes the dialog, destroys the HTML and updates or hides the overlay.
	   * @returns {AJAXDialog} This.
	   */
	  close() {
	    this.element.removeClass('open');
	    if (this.draggable !== null) {
	      this.draggable.destroy();
	      this.draggable = null;
	    }

	    setTimeout(function() {
	      this.element.removeFromDOM();
	      this.element = null;

	      const highestZIndex = this._determineZIndex();
	      if (highestZIndex !== 0) {
	        Overlay.instance.setZIndex(highestZIndex);
	      } else {
	        Overlay.instance.close();
	      }
	    }.bind(this), this.options.closeTimeout);

	    return this;
	  }

	  /**
	   * Destroys the dialog by calling the close function.
	   * @returns {AJAXDialog} This.
	   */
	  destroy() {
	    this.close();
	    return this;
	  }

	  /**
	   * Initializes the dialog.
	   * @returns {AJAXDialog} This.
	   */
	  initialize() {
	    return this;
	  }

	  /**
	   * Opens the dialog by making the AJAX GET request to the given URI and the opening then dialog.
	   *
	   * @param uri {string} The URI to make the AJAX GET request to.
	   * @returns {AJAXDialog} This.
	   */
	  open(uri) {
	    const request = this.options.ajaxRequest || new PrimeRequest(uri, 'GET');
	    request.withSuccessHandler(this._handleAJAXDialogResponse)
	        .withErrorHandler(this._handleAJAXDialogResponse)
	        .go();
	    return this;
	  }

	  /**
	   * Opens the dialog by making the AJAX POST request to the given URI with the given form and extra data (optional)
	   * and then opening the dialog.
	   *
	   * @param uri {string} The URI to make the AJAX POST request to.
	   * @param form {HTMLFormElement|PrimeElement} The Form element to retrieve the data from.
	   * @param extraData [extraData=] {object} (Optional) Extra data to send with the POST.
	   * @returns {AJAXDialog} This.
	   */
	  openPost(uri, form, extraData) {
	    new PrimeRequest(uri, 'POST')
	        .withDataFromForm(form)
	        .withData(extraData)
	        .withSuccessHandler(this._handleAJAXDialogResponse)
	        .go();
	    return this;
	  }

	  /**
	   * Updates the HTML contents of the dialog.
	   *
	   * @param html {String} The HTML.
	   * @returns {AJAXDialog} This.
	   */
	  setHTML(html) {
	    this.element.setHTML(html);
	    this._initializeDialog();
	    return this;
	  }

	  /**
	   * Sets any additional classes that should be on the dialog.
	   *
	   * @param classes {string} The list of additional classes.
	   * @returns {AJAXDialog} This.
	   */
	  withAdditionalClasses(classes) {
	    this.options.additionalClasses = classes;
	    return this;
	  }

	  /**
	   * Override the default Ajax Request used to open the dialog. This does not override the
	   * success and error handlers.
	   *
	   * @param request {PrimeRequest} The Ajax Request to use to open the dialog.
	   * @returns {AJAXDialog} This.
	   */
	  withAjaxRequest(request) {
	    this.options.ajaxRequest = request;
	    return this;
	  }

	  /**
	   * Sets the callback that is called after the dialog has been fetched and rendered.
	   *
	   * @param callback {function} The callback function.
	   * @returns {AJAXDialog} This.
	   */
	  withCallback(callback) {
	    this.options.callback = callback;
	    return this;
	  }

	  /**
	   * Sets the class name for the dialog element.
	   *
	   * @param className {string} The class name.
	   * @returns {AJAXDialog} This.
	   */
	  withClassName(className) {
	    if (className.indexOf(' ') !== -1) {
	      throw 'Invalid class name [' + className + ']. You can use the additionalClasses options to add more classes.';
	    }

	    this.options.className = className;
	    return this;
	  }

	  /**
	   * Sets the close button element selector that is used to setup the close button in the HTML that was returned from
	   * the server.
	   *
	   * @param selector {string} The element selector.
	   * @returns {AJAXDialog} This.
	   */
	  withCloseButtonElementSelector(selector) {
	    this.options.closeButtonElementSelector = selector;
	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {AJAXDialog} This.
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Sets the draggable element selector that is used for the DraggableWidget.
	   *
	   * @param selector {string} The element selector.
	   * @returns {AJAXDialog} This.
	   */
	  withDraggableButtonElementSelector(selector) {
	    this.options.draggableElementSelector = selector;
	    return this;
	  }

	  /**
	   * Sets an error callback for AJAX form handling. This is called after a failed form submission.
	   *
	   * @param callback {Function} The callback function. The callback function will called with two parameters,
	   *        the first is a reference this object, the second is the XMLHttpRequest object.
	   * @returns {AJAXDialog} This.
	   */
	  withFormErrorCallback(callback) {
	    this.options.formErrorCallback = callback;
	    return this;
	  }

	  /**
	   * Sets whether or not forms inside the dialog are handled via AJAX or not.
	   *
	   * @param enabled {boolean} The choice.
	   * @returns {AJAXDialog} This.
	   */
	  withFormHandling(enabled) {
	    this.options.formHandling = enabled;
	    return this;
	  }

	  /**
	   * Sets a pre-submit callback for AJAX form handling. This is called before the form is submitted.
	   *
	   * @param callback {Function} The callback function.
	   * @returns {AJAXDialog} This.
	   */
	  withFormPreSubmitCallback(callback) {
	    this.options.formPreSubmitCallback = callback;
	    return this;
	  }

	  /**
	   * Sets a success callback for AJAX form handling. This is called after a successful form submission.
	   *
	   * @param callback {Function} The callback function. The callback function will called with two parameters,
	   *        the first is a reference this object, the second is the XMLHttpRequest object.
	   * @returns {AJAXDialog} This.
	   */
	  withFormSuccessCallback(callback) {
	    this.options.formSuccessCallback = callback;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {AJAXDialog} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  _determineZIndex() {
	    let highestZIndex = 0;
	    PrimeDocument.query('.' + this.options.className).each(function(dialog) {
	      const zIndex = parseInt(dialog.getComputedStyle()['zIndex']);
	      if (dialog.isVisible() && zIndex > highestZIndex) {
	        highestZIndex = zIndex;
	      }
	    });
	    return highestZIndex;
	  }

	  _handleCloseClickEvent(event) {
	    Utils.stopEvent(event);
	    this.close();
	  }

	  _handleAJAXDialogResponse(xhr) {
	    this.element = PrimeDocument.newElement('<div/>', {class: this.options.className + ' ' + this.options.additionalClasses}).appendTo(document.body);
	    this.setHTML(xhr.responseText);
	  }

	  _handleAJAXFormError(xhr) {
	    this.setHTML(xhr.responseText);
	    this.form = this.element.queryFirst('form').addEventListener('submit', this._handleAJAXFormSubmit);

	    if (this.options.formErrorCallback !== null) {
	      this.options.formErrorCallback(this, xhr);
	    }

	    if (this.draggable !== null) {
	      this.draggable.destroy();
	    }

	    if (this.options.draggableElementSelector !== null && this.element.queryFirst(this.options.draggableElementSelector) !== null) {
	      this.draggable = new Draggable(this.element, this.options.draggableElementSelector).initialize();
	    }
	  }

	  _handleAJAXFormSuccess(xhr) {
	    if (this.options.formSuccessCallback !== null) {
	      this.options.formSuccessCallback(this, xhr);
	    } else {
	      const successURI = this.form.getDataSet()['ajaxSuccessUri'];
	      if (successURI !== undefined) {
	        window.location = successURI;
	      } else {
	        window.location.reload();
	      }
	    }
	  }

	  _handleAJAXFormSubmit(event) {
	    Utils.stopEvent(event);

	    if (this.options.formPreSubmitCallback !== null) {
	      this.options.formPreSubmitCallback(this);
	    }

	    new PrimeRequest(this.form.getAttribute('action'), this.form.getAttribute('method'))
	        .withDataFromForm(this.form)
	        .withSuccessHandler(this._handleAJAXFormSuccess)
	        .withErrorHandler(this._handleAJAXFormError)
	        .go();
	  }

	  _initializeDialog() {
	    this.element.query(this.options.closeButtonElementSelector).each(function(e) {
	      e.addEventListener('click', this._handleCloseClickEvent);
	    }.bind(this));

	    // Only set the z-index upon first open
	    if (!this.initialized) {
	      const highestZIndex = this._determineZIndex();
	      Overlay.instance.open(highestZIndex + this.options.zIndexOffset);
	      this.element.setStyle('zIndex', (highestZIndex + this.options.zIndexOffset + 10).toString());
	      this.element.addClass('open');
	    }

	    // Call the callback before positioning to ensure all changes to the dialog have been made
	    if (this.options.callback !== null) {
	      this.options.callback(this);
	    }

	    // Setup forms if enabled
	    if (this.options.formHandling) {
	      this.form = this.element.queryFirst('form').addEventListener('submit', this._handleAJAXFormSubmit);
	    }

	    // Only set the position of the dialog when we first open it, if someone calls setHTML on the dialog we are not resizing it.
	    if (!this.initialized) {
	      // Position the fixed dialog in the center of the screen
	      const windowHeight = PrimeWindow.getInnerHeight();
	      const dialogHeight = this.element.getHeight();
	      this.element.setTop(((windowHeight - dialogHeight) / 2) - 20);
	    }

	    if (this.options.draggableElementSelector !== null && this.element.queryFirst(this.options.draggableElementSelector) !== null) {
	      this.draggable = new Draggable(this.element, this.options.draggableElementSelector).initialize();
	    }

	    this.initialized = true;
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      additionalClasses: '',
	      ajaxRequest: null,
	      callback: null,
	      className: 'prime-dialog',
	      closeButtonElementSelector: '[data-dialog-role="close-button"]',
	      closeTimeout: 200,
	      draggableElementSelector: '[data-dialog-role="draggable"]',
	      formErrorCallback: null,
	      formHandling: false,
	      formPreSubmitCallback: null,
	      formSuccessCallback: null,
	      zIndexOffset: 1000
	    };
	  }
	}

	/*
	 * Copyright (c) 2013-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	/**
	 * @const {{Keys: {BACKSPACE: number, ENTER: number, ESCAPE: number, SPACE: number, TAB: number, LEFT_ARROW: number, UP_ARROW: number, RIGHT_ARROW: number, DOWN_ARROW: number, DELETE: number}} Events}
	 */
	const Events = {
	  Keys: {
	    BACKSPACE: 8,
	    ENTER: 13,
	    ESCAPE: 27,
	    SPACE: 32,
	    TAB: 9,
	    LEFT_ARROW: 37,
	    UP_ARROW: 38,
	    RIGHT_ARROW: 39,
	    DOWN_ARROW: 40,
	    DELETE: 46
	  }
	};

	/*
	 * Copyright (c) 2015-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const PrimeDate = {
	  DAYS_IN_MONTH: [31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31],

	  /**
	   * Return the hour in a 12-hour format. AM and PM are not communicated by the returned hour.
	   *
	   * @param date {Date} The date object to retrieve the hour from.
	   * @returns {Number} The hour of the day between 1 and 12.
	   */
	  getHourOfDay: function(date) {
	    return (date.getHours() + 24) % 12 || 12;
	  },

	  /**
	   * @param year The year.
	   * @returns {boolean} True if this is a leap year, otherwise false.
	   */
	  isLeapYear: function(year) {
	    return !((year % 4) || (!(year % 100) && (year % 400)));
	  },

	  /**
	   * Return the number of days in the month.
	   * @param year The year, the days in the month may change during a leap year.
	   * @param month The month.
	   * @returns {Number} The number of days in the month.
	   */
	  numberOfDaysInMonth: function(year, month) {
	    if (month === 1 && this.isLeapYear(year)) {
	      return 29;
	    } else {
	      return PrimeDate.DAYS_IN_MONTH[month];
	    }
	  },

	  /**
	   * Adds the given number of days to the given Date.
	   *
	   * @param date {Date} The date to add the days to.
	   * @param number {Number} The number of days to add.
	   */
	  plusDays: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    let newDate = date.getDate() + number;
	    let numberOfDaysInMonth = PrimeDate.numberOfDaysInMonth(date.getFullYear(), date.getMonth());

	    if (newDate > 0) {
	      while (newDate > numberOfDaysInMonth) {
	        PrimeDate.plusMonths(date, 1);
	        newDate = newDate - numberOfDaysInMonth;
	        numberOfDaysInMonth = PrimeDate.numberOfDaysInMonth(date.getFullYear(), date.getMonth());
	      }

	      date.setDate(newDate);
	    } else {
	      while (newDate <= 0) {
	        PrimeDate.plusMonths(date, -1);
	        numberOfDaysInMonth = PrimeDate.numberOfDaysInMonth(date.getFullYear(), date.getMonth());
	        newDate = newDate + numberOfDaysInMonth;
	      }

	      date.setDate(newDate);
	    }
	  },

	  /**
	   * Adds the given number of hours to the given Date. The number can be negative.
	   *
	   * @param date {Date} The date.
	   * @param number {Number} The number of hours to add.
	   */
	  plusHours: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    const deltaDays = parseInt(number / 24);
	    PrimeDate.plusDays(date, deltaDays);

	    const deltaHours = number % 24;
	    const newHour = date.getHours() + deltaHours;
	    if (newHour > 23) {
	      PrimeDate.plusDays(date, 1);
	      date.setHours(newHour - 24);
	    } else if (newHour < 0) {
	      PrimeDate.plusDays(date, -1);
	      date.setHours(24 + newHour);
	    } else {
	      date.setHours(newHour);
	    }
	  },

	  /**
	   * Adds the given number of minutes to the given Date. The number can be negative.
	   *
	   * @param date {Date} The date.
	   * @param number {Number} The number of minutes to add.
	   */
	  plusMinutes: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    const deltaHours = parseInt(number / 60);
	    PrimeDate.plusHours(date, deltaHours);

	    const deltaMinutes = number % 60;
	    const newMinute = date.getMinutes() + deltaMinutes;
	    if (newMinute > 60) {
	      PrimeDate.plusHours(date, 1);
	      date.setMinutes(newMinute - 60);
	    } else if (newMinute < 0) {
	      PrimeDate.plusHours(date, -1);
	      date.setMinutes(60 + newMinute);
	    } else {
	      date.setMinutes(newMinute);
	    }
	  },

	  /**
	   * Adds the given number of months to the given Date. The number can be negative.
	   *
	   * @param date {Date} The date.
	   * @param number {Number} The number of months to add.
	   */
	  plusMonths: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    let deltaYears = parseInt(number / 12);
	    let deltaMonths = number % 12;
	    let currentMonth = date.getMonth();
	    const newMonth = currentMonth + deltaMonths;
	    if (newMonth < 0) {
	      deltaYears--;
	      deltaMonths = newMonth;
	      currentMonth = 12;
	    } else if (newMonth >= 12) {
	      deltaYears++;
	      deltaMonths = newMonth - 12;
	      currentMonth = 0;
	    }

	    date.setYear(date.getFullYear() + deltaYears);
	    // If the day is 31 and you set month to 1 (February) it will adjust to March 3 (Feb 28 + 3)
	    const adjustedMonth = currentMonth + deltaMonths;
	    if (date.getDate() > this.DAYS_IN_MONTH[adjustedMonth]) {
	      date.setDate(this.DAYS_IN_MONTH[adjustedMonth]);
	    }
	    date.setMonth(adjustedMonth);
	  },

	  /**
	   * Adds the given number of seconds to the given Date. The number can be negative.
	   *
	   * @param date {Date} The date.
	   * @param number {Number} The number of seconds to add.
	   */
	  plusSeconds: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    const deltaMinutes = parseInt(number / 60);
	    PrimeDate.plusMinutes(date, deltaMinutes);

	    const deltaSeconds = number % 60;
	    const newSecond = date.getSeconds() + deltaSeconds;
	    if (newSecond > 60) {
	      PrimeDate.plusMinutes(date, 1);
	      date.setSeconds(newSecond - 60);
	    } else if (newSecond < 0) {
	      PrimeDate.plusMinutes(date, -1);
	      date.setSeconds(60 + newSecond);
	    } else {
	      date.setSeconds(newSecond);
	    }
	  },

	  /**
	   * Adds the given number of years to the given Date. The number can be negative.
	   *
	   * @param date {Date} The date.
	   * @param number {Number} The number of years to add.
	   */
	  plusYears: function(date, number) {
	    if (number === 0) {
	      return;
	    }

	    date.setFullYear(date.getFullYear() + number);
	  },

	  /**
	   * Return a string in simplified extended ISO format (ISO 8601) truncated to only return YYYY-MM-DD.
	   *
	   * For example: new Date(2015, 6, 4) --> 2015-07-04
	   *
	   * @param date {Date} The date.
	   * @returns {String} A date string in the format YYYY-MM-DD.
	   */
	  toDateOnlyISOString: function(date) {
	    if (date instanceof Date) {
	      return date.getFullYear()
	          + '-' + Utils.leftPadNumber(date.getMonth() + 1, 2)
	          + '-' + Utils.leftPadNumber(date.getDate(), 2);
	    }
	    throw TypeError('date parameter must be a Date object.');
	  }
	};

	/*
	 * Copyright (c) 2015-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const SHORT_DAY_NAMES = ['Su', 'Mo', 'Tu', 'We', 'Th', 'Fr', 'Sa'];
	const MONTHS = ['January', 'February', 'March', 'April', 'May', 'June', 'July', 'August', 'September', 'October', 'November', 'December'];
	const DATE_SEPARATOR = '/';
	const TIME_SEPARATOR = ':';
	const AM_PM = ['AM', 'PM'];

	class DateTimePicker {
	  /**
	   * Constructs a new DateTimePicker object for the given input element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the DateTimePicker widget.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    if (!this.element.is('input')) {
	      throw new TypeError('You can only use DateTimePicker with an input element');
	    }

	    this.callback = null;
	    this.customFormatHandler = null;
	    this._setInitialOptions();
	  };

	  static get SHORT_DAY_NAMES() {
	    return SHORT_DAY_NAMES;
	  }

	  static get MONTHS() {
	    return MONTHS;
	  }

	  static get DATE_SEPARATOR() {
	    return DATE_SEPARATOR;
	  }

	  static get TIME_SEPARATOR() {
	    return TIME_SEPARATOR;
	  }

	  static get AM_PM() {
	    return AM_PM;
	  }

	  /**
	   * Closes the Date Picker widget.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  close() {
	    this.datepicker.removeClass('open');

	    // Pause a bit to cancel focus event and allow transition to play
	    setTimeout(function() {
	      this.datepicker.hide();
	    }.bind(this), this.options.closeTimeout);
	    return this;
	  }

	  /**
	   * Closes the months select box.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  closeMonthsSelect() {
	    this.months.removeClass('open');
	    setTimeout(function() {
	      this.months.hide();
	    }.bind(this), this.options.closeTimeout);
	    return this;
	  }

	  /**
	   * Closes the years select box.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  closeYearsSelect() {
	    this.years.removeClass('open');
	    setTimeout(function() {
	      this.years.hide();
	    }.bind(this), this.options.closeTimeout);
	    return this;
	  }

	  /**
	   * Destroys the DateTimePicker Widget
	   */
	  destroy() {
	    this.datepicker.removeFromDOM();
	    this.element.removeEventListener('click', this._handleInputClick)
	        .removeEventListener('focus', this._handleInputClick)
	        .removeEventListener('keydown', this._handleInputKey);
	    PrimeDocument.removeEventListener('click', this._handleGlobalClick);
	    PrimeDocument.removeEventListener('keydown', this._handleGlobalKey);
	  }

	  /**
	   * Draws the calendar using the month and year from the given Date object.
	   *
	   * @param date {Date} The date to draw the calendar for.
	   * @return {DateTimePicker} This DateTimePicker.
	   */
	  drawCalendar(date) {
	    const month = date.getMonth();
	    const year = date.getFullYear();
	    const firstDay = new Date(year, month, 1);
	    const firstDayOfMonth = firstDay.getDay();
	    const daysInMonth = PrimeDate.numberOfDaysInMonth(year, month);
	    const used = firstDayOfMonth + daysInMonth;
	    const weeksInMonth = Math.ceil(used / 7);

	    let rows = '';
	    let startDay = 1;
	    for (let i = 0; i < weeksInMonth; i++) {
	      const startDayOfWeek = i === 0 ? firstDayOfMonth : 0;
	      rows += this._buildCalendarWeek(date, startDayOfWeek, startDay, daysInMonth);
	      startDay += 7 - startDayOfWeek; // increment by 7 adjusted by a week day of week offset
	    }

	    this.calendarBody.setHTML(rows);

	    // update data- attributes
	    this.monthDisplay.setDataAttribute('month', month);
	    this.yearDisplay.setDataAttribute('year', year);

	    // update text
	    this.monthDisplay.setTextContent(DateTimePicker.MONTHS[month]);
	    this.yearDisplay.setTextContent(year);

	    return this;
	  }

	  /**
	   * Rebuilds the entire widget using the date value. Even if the user has moved to a different month display, this will
	   * rebuild the table completely.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  initialize() {
	    const value = this.element.getValue();
	    if (value === '' || value === null) {
	      this.date = new Date();
	    } else {
	      this.date = new Date(value);
	    }

	    const year = this.date.getUTCFullYear();
	    const timeSeparator = `<span>${DateTimePicker.TIME_SEPARATOR}</span>`;
	    const dateSeparator = `<span>${DateTimePicker.DATE_SEPARATOR}</span>`;
	    let html =
	        `<div class="${this.options.className }">
  <header>
    <span class="prev">&#9664;</span>
    <span class="month"></span>
    <span class="year"></span>
    <span class="next">&#9654;</span>
  </header>
  <table>
    <thead>
      <tr>
        <th>${DateTimePicker.SHORT_DAY_NAMES[0]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[1]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[2]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[3]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[4]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[5]}</th>
        <th>${DateTimePicker.SHORT_DAY_NAMES[6]}</th>
      </tr>
    </thead>
    <tbody>
    </tbody>
  </table>
  <div class="inputs">
    <div class="date">
      <input size="2" maxlength="2" type="text" name="month" autocomplete="off"/>${dateSeparator}
      <input size="2" maxlength="2" type="text" name="day" autocomplete="off"/>${dateSeparator}
      <input size="4" maxlength="4" type="text" name="year" autocomplete="off"/>
    </div>
    <div class="time">
      <input size="2" maxlength="2" type="text" name="hour" autocomplete="off"/>${timeSeparator}
      <input size="2" maxlength="2" type="text" name="minute" autocomplete="off"/>${timeSeparator}
      <input size="2" maxlength="2" type="text" name="second" autocomplete="off"/>
      <input size="2" maxlength="2" type="text" name="am_pm" autocomplete="off"/>
    </div>
  </div>
</div>`;
	    PrimeDocument.appendHTML(html);
	    this.datepicker = PrimeDocument.queryLast('.' + this.options.className).hide();
	    this.element.addEventListener('click', this._handleInputClick);
	    this.element.addEventListener('focus', this._handleInputClick);
	    this.element.addEventListener('keydown', this._handleInputKey);

	    this.calendarBody = this.datepicker.queryFirst('table tbody').addEventListener('click', this._handleDayClick);
	    this.monthDisplay = this.datepicker.queryFirst('header .month').addEventListener('click', this._handleMonthExpand);
	    this.yearDisplay = this.datepicker.queryFirst('header .year').addEventListener('click', this._handleYearExpand);

	    this.time = this.datepicker.queryFirst('.time');
	    this.inputs = this.datepicker.queryFirst('div.inputs');
	    this.hourInput = this.inputs.queryFirst('input[name=hour]').addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleHourKey);
	    this.minuteInput = this.inputs.queryFirst('input[name=minute]').addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleMinuteKey);
	    this.secondInput = this.inputs.queryFirst('input[name=second]').addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleSecondKey);
	    this.ampmInput = this.inputs.queryFirst('input[name=am_pm]').addEventListener('keydown', this._handleAmPmKey);
	    this.monthInput = this.inputs.queryFirst('input[name=month]').setValue(this.date.getMonth() + 1).addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleMonthKey);
	    this.dayInput = this.inputs.queryFirst('input[name=day]').setValue(this.date.getDate()).addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleDayKey);
	    this.yearInput = this.inputs.queryFirst('input[name=year]').setValue(this.date.getFullYear()).addEventListener('change', this._handleDateTimeChange).addEventListener('keydown', this._handleYearKey);

	    this.datepicker.queryFirst('header .next').addEventListener('click', this._handleNextMonth);
	    this.datepicker.queryFirst('header .prev').addEventListener('click', this._handlePreviousMonth);

	    PrimeDocument.addEventListener('click', this._handleGlobalClick);
	    PrimeDocument.addEventListener('keydown', this._handleGlobalKey);

	    // Setup months dropdown
	    html = '<div class="months">';
	    for (let i = 0; i < DateTimePicker.MONTHS.length; i++) {
	      html += `<div data-month="${i}">${DateTimePicker.MONTHS[i]}</div>`;
	    }
	    html += '</div>';
	    this.datepicker.appendHTML(html);
	    this.months = this.datepicker.queryFirst('.months');
	    this.months.hide();
	    this.months.getChildren().each(function(month) {
	      month.addEventListener('click', function() {
	        this.setMonth(parseInt(month.getDataAttribute('month')));
	        this.closeMonthsSelect();
	      }.bind(this));
	    }.bind(this));

	    // Setup year dropdown
	    html = '<div class="years">';
	    const startYear = this.date.getFullYear() - 10;
	    const endYear = this.date.getFullYear() + 10;
	    for (let i = startYear; i < endYear; i++) {
	      html += `<div data-year="${i}">${i}</div>`;
	    }
	    html += '</div>';
	    this.datepicker.appendHTML(html);
	    this.years = this.datepicker.queryFirst('.years');
	    this.years.hide();
	    this.years.getChildren().each(function(year) {
	      year.addEventListener('click', function() {
	        this.setYear(parseInt(year.getDataAttribute('year')));
	        this.closeYearsSelect();
	      }.bind(this));
	    }.bind(this));

	    this._rebuild();

	    if (this.customFormatHandler !== null) {
	      this.element.setValue(this.customFormatHandler.call(null, this.date));
	    }

	    return this;
	  }

	  /**
	   * @returns {Date} Return the current value of the time picker.
	   */
	  getDate() {
	    return new Date(this.date.getTime());
	  }

	  /**
	   * Moves the DateTimePicker to the next month and redraws the calendar.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  nextMonth() {
	    const newDate = new Date(this.date);
	    newDate.setDate(1); // Set the day to 1 to keep us from wrapping months on the 30 and 31st.
	    newDate.setMonth(parseInt(this.monthDisplay.getDataAttribute('month')));
	    newDate.setFullYear(parseInt(this.yearDisplay.getDataAttribute('year')));
	    PrimeDate.plusMonths(newDate, 1);
	    this.drawCalendar(newDate);
	    return this;
	  }

	  /**
	   * Opens the Date Picker widget.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  open() {
	    this.datepicker.setLeft(this.element.getLeft());
	    this.datepicker.setTop(this.element.getAbsoluteTop() + this.element.getHeight() + 8);
	    this.datepicker.show();
	    this.datepicker.addClass('open');

	    const zIndex = this.element.getRelativeZIndex();
	    this.datepicker.setStyle('zIndex', zIndex + 10);
	    return this;
	  }

	  /**
	   * Opens the month select box.
	   */
	  openMonthSelect() {
	    this.closeYearsSelect();

	    this.months.setLeft(this.monthDisplay.getOffsetLeft() - 5);
	    this.months.setTop(this.monthDisplay.getOffsetTop() - 5);
	    this.months.setStyle('zIndex', this.monthDisplay.getRelativeZIndex() + 10);
	    this.months.show();
	    this.months.addClass('open');

	    const currentMonth = this.months.queryFirst('[data-month="' + this.date.getMonth() + '"]');
	    this.months.getChildren().each(function(month) {
	      month.removeClass('selected');
	    });
	    currentMonth.addClass('selected');
	  }

	  /**
	   * Opens the year select box.
	   */
	  openYearSelect() {
	    this.closeMonthsSelect();

	    this.years.setLeft(this.yearDisplay.getOffsetLeft() - 5);
	    this.years.setTop(this.yearDisplay.getOffsetTop() - 5);
	    this.years.setStyle('zIndex', this.yearDisplay.getRelativeZIndex() + 10);
	    this.years.show();
	    this.years.addClass('open');

	    const currentYear = this.years.queryFirst('[data-year="' + this.date.getFullYear() + '"]');
	    this.years.getChildren().each(function(year) {
	      year.removeClass('selected');
	    });
	    currentYear.addClass('selected');
	  }

	  /**
	   * Moves the DateTimePicker to the previous month and redraws the calendar.
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  previousMonth() {
	    const newDate = new Date(this.date);
	    newDate.setDate(1); // Set to 1 until month has been set
	    newDate.setMonth(parseInt(this.monthDisplay.getDataAttribute('month')));
	    newDate.setFullYear(parseInt(this.yearDisplay.getDataAttribute('year')));
	    PrimeDate.plusMonths(newDate, -1);
	    this.drawCalendar(newDate);
	    return this;
	  }

	  /**
	   * Sets the date of the DateTimePicker and redraws the calendar to the month for the date.
	   *
	   * @param {Date} newDate The new date.
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  setDate(newDate) {
	    this.date = newDate;

	    if (this.customFormatHandler !== null) {
	      this.element.setValue(this.customFormatHandler.call(null, this.date));
	    } else {
	      if (this.options.dateOnly) {
	        this.element.setValue(PrimeDate.toDateOnlyISOString(newDate));
	      } else {
	        this.element.setValue(newDate.toISOString());
	      }
	    }

	    this._rebuild();

	    if (this.callback !== null) {
	      this.callback(this);
	    }

	    return this;
	  }

	  /**
	   * @param {number} month The month. A 0 based number between 0 (January) and 11 (December).
	   * @returns {DateTimePicker}
	   */
	  setMonth(month) {
	    let currentYear = parseInt(this.yearDisplay.getDataAttribute('year'));
	    if (month < 0) {
	      month = 11;
	      currentYear--;
	    } else if (month > 11) {
	      currentYear++;
	      month = 0;
	    }

	    this.date.setMonth(month);
	    this.date.setYear(currentYear);
	    this.setDate(this.date);

	    return this;
	  }

	  /**
	   *
	   * @param {number} year The year.
	   * @returns {DateTimePicker}
	   */
	  setYear(year) {
	    this.yearDisplay.setDataAttribute('year', year);
	    this.yearDisplay.setTextContent(year);
	    this.date.setYear(year);
	    this.setDate(this.date);
	    return this;
	  }

	  /**
	   * Sets the callback handler that is called with the DateTimePicker's value is changed.
	   *
	   * @param callback {Function} The callback function.
	   * @return {DateTimePicker} This.
	   */
	  withCallback(callback) {
	    this.callback = callback;
	    return this;
	  }

	  /**
	   * Sets the class name for the main div of the date time picker.
	   *
	   * @param className {string} The class name.
	   * @returns {DateTimePicker} This.
	   */
	  withClassName(className) {
	    this.options.className = className;
	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {DateTimePicker}
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Sets a custom format handler responsible for formatting the date string that will be set into the input field.
	   * When not defined the default behavior will be used.
	   *
	   * @param formatHandler {Function} The handler function.
	   * @return {DateTimePicker} This.
	   */
	  withCustomFormatHandler(formatHandler) {
	    this.customFormatHandler = formatHandler;
	    return this;
	  }

	  /**
	   * Render the DateTimePicker w/out the time picker. Only the calendar will be displayed and the input field will be updated with date only.
	   *
	   *
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  withDateOnly() {
	    this.options.dateOnly = true;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {DateTimePicker} This DateTimePicker.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Build the HTML for a single calendar week.
	   *
	   * @param date {Date} The date to build the calendar week based on.
	   * @param startDayOfWeek {Number} The day of the week of this week begins. A 0 based number between 0 and 6.
	   * @param startDayOfMonth {Number} The day of the month this week begins. A number between 1 and 31.
	   * @param daysInMonth {Number} The number of days in this calendar month.
	   * @returns {string} The HTML for this week.
	   * @private
	   */
	  _buildCalendarWeek(date, startDayOfWeek, startDayOfMonth, daysInMonth) {
	    const daysInPreviousMonth = new Date(date.getFullYear(), date.getMonth(), 0).getDate();
	    let startDayOfPreviousMonth = daysInPreviousMonth - startDayOfWeek + 1;
	    let startDayOfNextMonth = 1;

	    let row = '<tr>';
	    let emptyColumns = 0;
	    const year = date.getFullYear();
	    const month = date.getMonth();

	    for (let i = 0; i < 7; i++) {
	      const dayOfWeek = startDayOfMonth + i;
	      // Days of the previous month
	      if (dayOfWeek <= startDayOfWeek) {
	        row += '<td><a class="inactive" href="#" data-year="' + year + '" data-month="' + (month - 1) + '" data-day="' + startDayOfPreviousMonth + '">' + startDayOfPreviousMonth + '</a></td>';
	        startDayOfPreviousMonth++;
	        emptyColumns++;
	      } else if (dayOfWeek > daysInMonth) {
	        // Days of the next month
	        row += '<td><a class="inactive" href="#" data-year="' + year + '" data-month="' + month + '" data-day="' + dayOfWeek + '">' + startDayOfNextMonth + '</a></td>';
	        startDayOfNextMonth++;
	      } else {
	        // Days in the current month
	        const day = dayOfWeek - emptyColumns;
	        const selected = this.date.getDate() === day && this.date.getMonth() === month;
	        row += '<td><a ' + (selected ? 'class="selected"' : '') + 'href="#" data-year="' + year + '" data-month="' + month + '" data-day="' + day + '">' + day + '</a></td>';
	      }
	    }

	    row += '</tr>';
	    return row;
	  }

	  // noinspection JSMethodCanBeStatic
	  /**
	   * Clamp the value between the minimum and maximum values.
	   *
	   * @param {Number} min the minimum number value.
	   * @param {Number} max The maximum number value.
	   * @param {Number} value The value to clamp.
	   * @returns {Number} The resulting value, either the min, max or actual value if not out of bounds.
	   * @private
	   */
	  _clamp(min, max, value) {
	    return Math.max(min, Math.min(value, max));
	  }

	  /**
	   * Handles when the AM/PM element is selected and the user hits a key. If the user hits A, this changes to AM. If the
	   * user hits P, this changes to PM. If the use hits the up or down arrows, this toggles between AM and PM.
	   *
	   * @param {KeyboardEvent} event The keyboard event.
	   * @private
	   */
	  _handleAmPmKey(event) {
	    if (event.keyCode === Events.Keys.TAB) {
	      Utils.stopEvent(event);
	      if (event.shiftKey) {
	        this.secondInput.domElement.setSelectionRange(0, this.secondInput.getValue().length);
	        this.secondInput.focus();
	      } else {
	        this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	        this.monthInput.focus();
	      }
	      return;
	    }

	    // Decode the key event
	    const current = this.ampmInput.getValue();
	    if (event.keyCode === 65) {
	      // User hit A
	      if (current === DateTimePicker.AM_PM[1]) {
	        PrimeDate.plusHours(this.date, -12);
	      }
	    } else if (event.keyCode === 80) {
	      // User hit P
	      if (current === DateTimePicker.AM_PM[0]) {
	        PrimeDate.plusHours(this.date, 12);
	      }
	    } else if (event.keyCode === Events.Keys.UP_ARROW || event.keyCode === Events.Keys.DOWN_ARROW) {
	      // User hit up or down arrow
	      if (current === DateTimePicker.AM_PM[0]) {
	        PrimeDate.plusHours(this.date, 12);
	      } else if (current === DateTimePicker.AM_PM[1]) {
	        PrimeDate.plusHours(this.date, -12);
	      }
	    } else if (event.keyCode === Events.Keys.ENTER || event.keyCode === Events.Keys.ESCAPE) {
	      return;
	    }

	    this.setDate(this.date);
	    this.ampmInput.domElement.setSelectionRange(0, this.ampmInput.getValue().length);
	    Utils.stopEvent(event);
	  }

	  /**
	   * Handle date/time change events. This pulls the values from the 3 date fields and makes a new Date. Then it calls
	   * {@link #setDate(Date)}.
	   *
	   * @param {KeyboardEvent} event The keyboard event.
	   * @private
	   */
	  _handleDateTimeChange(event) {
	    Utils.stopEvent(event);
	    const newDate = new Date();
	    const hours = this._clamp(1, 12, parseInt(this.hourInput.getValue()));
	    if (this.ampmInput.getValue() === DateTimePicker.AM_PM[0]) {
	      if (hours === 12) {
	        newDate.setHours(0);
	      } else {
	        newDate.setHours(hours);
	      }
	    } else {
	      if (hours === 12) {
	        newDate.setHours(12);
	      } else {
	        newDate.setHours(hours + 12);
	      }
	    }

	    const seconds = this._clamp(0, 59, parseInt(this.secondInput.getValue()));
	    const minutes = this._clamp(0, 59, parseInt(this.minuteInput.getValue()));

	    newDate.setSeconds(seconds);
	    newDate.setMinutes(minutes);
	    newDate.setDate(1); // Set to 1 until month has been set
	    newDate.setMonth(parseInt(this.monthInput.getValue()) - 1);
	    newDate.setDate(parseInt(this.dayInput.getValue()));
	    newDate.setYear(parseInt(this.yearInput.getValue()));

	    this.setDate(newDate);
	  }

	  /**
	   * Handle the click on a day.
	   *
	   * @parameter {MouseEvent} event The click event.
	   * @private
	   */
	  _handleDayClick(event) {
	    Utils.stopEvent(event);
	    let dayElement = new PrimeElement(event.target);
	    if (!dayElement.is('a')) {
	      dayElement = dayElement.queryFirst('a');
	    }

	    const newDate = new Date(this.date);
	    newDate.setDate(1); // Set to 1 until month has been set
	    newDate.setFullYear(parseInt(dayElement.getDataAttribute('year')));
	    newDate.setMonth(parseInt(dayElement.getDataAttribute('month')));
	    newDate.setDate(parseInt(dayElement.getDataAttribute('day')));
	    this.setDate(newDate);
	  }

	  /**
	   * Handles when a key is click in the day input field so that tab and shift tab work properly.
	   * <p>
	   * Also handles up and down arrow to increment and decrement the day.
	   *
	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleDayKey(event) {
	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      PrimeDate.plusDays(this.date, 1);
	      this.setDate(this.date);
	      this.dayInput.domElement.setSelectionRange(0, this.dayInput.getValue().length);
	      Utils.stopEvent(event);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusDays(this.date, -1);
	      this.setDate(this.date);
	      this.dayInput.domElement.setSelectionRange(0, this.dayInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setDate(parseInt(this.dayInput.getValue()));
	    }
	  }

	  /**
	   * Handles a global click event. This determines if the click was outside of the DateTimePicker and closes it.
	   *
	   * @param {MouseEvent} event The click event.
	   * @private
	   */
	  _handleGlobalClick(event) {
	    // Skip this function completely if they clicked the input field
	    if (event.target === this.element.domElement) {
	      return;
	    }

	    const top = this.datepicker.getTop();
	    const bottom = this.datepicker.getBottom();
	    const left = this.datepicker.getLeft();
	    const right = this.datepicker.getRight();
	    if (this.datepicker.isVisible() && (event.x < left || event.x > right || event.y < top || event.y > bottom)) {
	      this.close();
	      this.closeYearsSelect();
	      this.closeMonthsSelect();
	    } else {
	      if (this.years.isVisible()) {
	        this.closeYearsSelect();
	      }
	      if (this.months.isVisible()) {
	        this.closeMonthsSelect();
	      }
	    }
	  }

	  /**
	   * Handles a global key event. This determines if the DateTimePicker is open and if enter or escape was hit.
	   *
	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleGlobalKey(event) {
	    // Skip this function completely if the DateTimePicker isn't open
	    if (!this.datepicker.isVisible()) {
	      return;
	    }

	    if (event.keyCode === Events.Keys.ENTER) {
	      Utils.stopEvent(event);
	      this.setDate(this.date);
	      this.close();
	      this.element.focus();
	    } else if (event.keyCode === Events.Keys.ESCAPE) {
	      this.close();
	      this.element.focus();
	    }
	  }

	  /**
	   * Handles when a key is click in the hours input field so that tab and shift tab work properly.
	   * <p>
	   * Also handles up and down arrow to increment and decrement the hour.
	   *
	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleHourKey(event) {
	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusHours(this.date, 1);
	      this.setDate(this.date);
	      this.hourInput.domElement.setSelectionRange(0, this.hourInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusHours(this.date, -1);
	      this.setDate(this.date);
	      this.hourInput.domElement.setSelectionRange(0, this.hourInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setHours(parseInt(this.hourInput.getValue()));
	    }
	  }

	  /**
	   * Handle the click event for the input date field. If the DateTimePicker is hidden this will call the {@link #show()}
	   * function.
	   *
	   * @returns {boolean} Always true.
	   * @private
	   */
	  _handleInputClick() {
	    if (!this.datepicker.isVisible()) {
	      this.open();
	      this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	      this.monthInput.focus();
	    }
	  }

	  /**
	   * Handle the key event for the input date field. If the user hits tab or shift-tab, this moves the focus to the
	   * nested inputs.
	   *
	   * @param {KeyboardEvent} event The keyboard event.
	   * @private
	   */
	  _handleInputKey(event) {
	    if (this.datepicker.isVisible() && event.keyCode === Events.Keys.TAB) {
	      Utils.stopEvent(event);
	      if (event.shiftKey) {
	        this.ampmInput.domElement.setSelectionRange(0, this.ampmInput.getValue().length);
	        this.ampmInput.focus();
	      } else {
	        this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	        this.monthInput.focus();
	      }
	    }
	  }

	  /**
	   * Handle the key down event and capture the up and down arrow key to increment and decrement the minute.

	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleMinuteKey(event) {
	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusMinutes(this.date, 1);
	      this.setDate(this.date);
	      this.minuteInput.domElement.setSelectionRange(0, this.minuteInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusMinutes(this.date, -1);
	      this.setDate(this.date);
	      this.minuteInput.domElement.setSelectionRange(0, this.minuteInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setMinutes(parseInt(this.minuteInput.getValue()));
	    }
	  }

	  /**
	   * Handles the click on the month to open the month select.
	   *
	   * @private
	   */
	  _handleMonthExpand(event) {
	    Utils.stopEvent(event);
	    this.openMonthSelect();
	  }

	  /**
	   * Handles when a key is click in the month input field so that tab and shift tab work properly.
	   * <p>
	   * Also handles up and down arrow to increment and decrement the month.
	   *
	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleMonthKey(event) {
	    if (event.keyCode === Events.Keys.TAB && event.shiftKey) {
	      Utils.stopEvent(event);
	      if (this.options.dateOnly) {
	        this.yearInput.domElement.setSelectionRange(0, this.yearInput.getValue().length);
	        this.yearInput.focus();
	      } else {
	        this.ampmInput.domElement.setSelectionRange(0, this.ampmInput.getValue().length);
	        this.ampmInput.focus();
	      }
	      return;
	    }

	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusMonths(this.date, 1);
	      this.setDate(this.date);
	      this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusMonths(this.date, -1);
	      this.setDate(this.date);
	      this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setMonth(parseInt(this.monthInput.getValue()) - 1);
	    }
	  }

	  /**
	   * Handle the next month button click.
	   *
	   * @param {MouseEvent} event The mouse event.
	   * @private
	   */
	  _handleNextMonth(event) {
	    Utils.stopEvent(event);
	    this.nextMonth();
	  }

	  /**
	   * Handle the previous month button click.
	   *
	   * @param {MouseEvent} event The mouse event.
	   * @private
	   */
	  _handlePreviousMonth(event) {
	    Utils.stopEvent(event);
	    this.previousMonth();
	  }

	  /**
	   * Handle the key down event and capture the up and down arrow key to increment and decrement the second.

	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleSecondKey(event) {
	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusSeconds(this.date, 1);
	      this.setDate(this.date);
	      this.secondInput.domElement.setSelectionRange(0, this.secondInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusSeconds(this.date, -1);
	      this.setDate(this.date);
	      this.secondInput.domElement.setSelectionRange(0, this.secondInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setSeconds(parseInt(this.secondInput.getValue()));
	    }
	  }

	  /**
	   * Handles the click on the year to open the year select.
	   *
	   * @private
	   */
	  _handleYearExpand(event) {
	    Utils.stopEvent(event);
	    this.openYearSelect();
	  }

	  /**
	   * Handles when a key is click in the year input field so that tab and shift tab work properly.
	   * <p>
	   * Also handles up and down arrow to increment and decrement the year.
	   *
	   * @param {KeyboardEvent} event The key event.
	   * @private
	   */
	  _handleYearKey(event) {
	    if (event.keyCode === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusYears(this.date, 1);
	      this.setDate(this.date);
	      this.yearInput.domElement.setSelectionRange(0, this.yearInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      PrimeDate.plusYears(this.date, -1);
	      this.setDate(this.date);
	      this.yearInput.domElement.setSelectionRange(0, this.yearInput.getValue().length);
	    } else if (event.keyCode === Events.Keys.TAB && this.options.dateOnly) {
	      Utils.stopEvent(event);
	      if (event.shiftKey) {
	        this.dayInput.domElement.setSelectionRange(0, this.dayInput.getValue().length);
	        this.dayInput.focus();
	      } else {
	        this.monthInput.domElement.setSelectionRange(0, this.monthInput.getValue().length);
	        this.monthInput.focus();
	      }
	    } else if (event.keyCode === Events.Keys.ENTER) {
	      this.date.setFullYear(parseInt(this.yearInput.getValue()));
	    }
	  }

	  /**
	   * Rebuilds the HTML of the DateTimePicker.
	   * @private
	   */
	  _rebuild() {
	    this.drawCalendar(this.date);
	    this._refreshInputs();

	    if (this.options.dateOnly) {
	      this.time.hide();
	    }
	  }

	  /**
	   * Refresh the time inputs.
	   *
	   * @private
	   */
	  _refreshInputs() {
	    // Set Time -- assuming 12-hour time for the input fields and ISO 24-hour time for the field
	    const hours = PrimeDate.getHourOfDay(this.date);
	    this.hourInput.setValue(hours);

	    const minutes = this.date.getMinutes();
	    this.minuteInput.setValue(("00" + minutes).slice(-2));

	    const seconds = this.date.getSeconds();
	    this.secondInput.setValue(("00" + seconds).slice(-2));

	    if (this.date.getHours() >= 12) {
	      this.ampmInput.setValue(DateTimePicker.AM_PM[1]);
	    } else {
	      this.ampmInput.setValue(DateTimePicker.AM_PM[0]);
	    }

	    this.monthInput.setValue(this.date.getMonth() + 1);
	    this.dayInput.setValue(this.date.getDate());
	    this.yearInput.setValue(this.date.getFullYear());
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      className: 'prime-date-picker',
	      closeTimeout: 200,
	      dateOnly: false
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */

	class Dismissable {
	  /**
	   * Constructs a new Dismissable object for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Element for the Dismissable widget.
	   * @param {PrimeElement|Element|EventTarget} dismissButton The Element for the Dismissable button.
	   * @constructor
	   */
	  constructor(element, dismissButton) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this.dismissButton = dismissButton;
	    this._setInitialOptions();
	  }

	  /**
	   * Closes the Dismissable by removing the open class from the element and setting a timer to remove the element from
	   * the DOM.
	   */
	  close() {
	    this.element.addClass('closed');
	    setTimeout(function() {
	      this.element.removeFromDOM();
	    }.bind(this), this.options.closeTimeout);
	  }

	  /**
	   * Destroys the widget.
	   */
	  destroy() {
	    this.dismissButton.removeEventListener('click', this._handleClick);
	  }

	  /**
	   * Initializes the Dismissable by binding the events to the dismiss button.
	   *
	   * @returns {Dismissable} This.
	   */
	  initialize() {
	    this.dismissButton.addEventListener('click', this._handleClick);
	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {Dismissable} This.
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {Dismissable} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the click event.
	   * @private
	   */
	  _handleClick(event) {
	    Utils.stopEvent(event);
	    this.close();
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      closeTimeout: 400
	    };
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class HTMLDialog {
	  /**
	   * Constructs a new dialog box from an element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the HTMLDialog widget.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this._setInitialOptions();
	    this.draggable = null;
	  }

	  /**
	   * Closes the dialog, destroys the HTML and updates or hides the overlay.
	   * @returns {HTMLDialog} This.
	   */
	  close() {
	    this.element.removeClass('open');
	    if (this.draggable !== null) {
	      this.draggable.destroy();
	      this.draggable = null;
	    }

	    setTimeout(function() {
	      this.element.hide();

	      const highestZIndex = this._determineZIndex();
	      if (highestZIndex !== 0) {
	        Overlay.instance.setZIndex(highestZIndex);
	      } else {
	        Overlay.instance.close();
	      }
	    }.bind(this), this.options.closeTimeout);

	    return this;
	  }

	  /**
	   * Destroys the dialog by calling the close function.
	   * @returns {HTMLDialog} This.
	   */
	  destroy() {
	    this.close();
	    return this;
	  }

	  /**
	   * Initializes the dialog.
	   * @returns {HTMLDialog} This.
	   */
	  initialize() {
	    this.element.hide();
	    return this;
	  }

	  /**
	   * Opens the dialog.
	   *
	   * @returns {HTMLDialog} This.
	   */
	  open() {
	    const highestZIndex = this._determineZIndex();
	    Overlay.instance.open(highestZIndex + this.options.zIndexOffset);
	    this.element.setStyle('zIndex', (highestZIndex + this.options.zIndexOffset + 10).toString());
	    this.element.show();
	    this.element.addClass('open');

	    // Call the callback before positioning to ensure all changes to the dialog have been made
	    if (this.options.callback !== null) {
	      this.options.callback(this);
	    }

	    // Position the fixed dialog in the center of the screen
	    const windowHeight = PrimeWindow.getInnerHeight();
	    const dialogHeight = this.element.getHeight();
	    this.element.setTop(((windowHeight - dialogHeight) / 2) - 20);

	    this._setupButtons();

	    if (this.draggable === null) {
	      if (this.options.draggableElementSelector !== null && this.element.queryFirst(this.options.draggableElementSelector) !== null) {
	        this.draggable = new Draggable(this.element, this.options.draggableElementSelector).initialize();
	      }
	    }

	    return this;
	  }

	  /**
	   * Updates the HTML contents of the dialog.
	   *
	   * @param html {String} The HTML.
	   * @returns {HTMLDialog} This.
	   */
	  setHTML(html) {
	    this.element.setHTML(html);
	    this._setupButtons();
	    return this;
	  }

	  /**
	   * Sets the callback that is called after the dialog has been fetched and rendered.
	   *
	   * @param callback {function} The callback function.
	   * @returns {HTMLDialog} This.
	   */
	  withCallback(callback) {
	    this.options.callback = callback;
	    return this;
	  }

	  /**
	   * Sets the class name for the dialog element.
	   *
	   * @param className {string} The class name.
	   * @returns {HTMLDialog} This.
	   */
	  withClassName(className) {
	    this.options.className = className;
	    return this;
	  }

	  /**
	   * Sets the close button element selector that is used to setup the close button in the HTML that was returned from
	   * the server.
	   *
	   * @param selector {string} The element selector.
	   * @returns {HTMLDialog} This.
	   */
	  withCloseButtonElementSelector(selector) {
	    this.options.closeButtonElementSelector = selector;
	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {HTMLDialog} This.
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Sets the draggable element selector that is used for the Draggable.
	   *
	   * @param selector {string} The element selector.
	   * @returns {HTMLDialog} This.
	   */
	  withDraggableButtonElementSelector(selector) {
	    this.options.draggableElementSelector = selector;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {HTMLDialog} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  _determineZIndex() {
	    let highestZIndex = 0;
	    PrimeDocument.query('.' + this.options.className).each(function(dialog) {
	      const zIndex = parseInt(dialog.getComputedStyle()['zIndex']);
	      if (dialog.isVisible() && zIndex > highestZIndex) {
	        highestZIndex = zIndex;
	      }
	    });
	    return highestZIndex;
	  }

	  _handleCloseClickEvent(event) {
	    Utils.stopEvent(event);
	    this.close();
	  }

	  _setupButtons() {
	    this.element.query(this.options.closeButtonElementSelector).each(function(e) {
	      e.addEventListener('click', this._handleCloseClickEvent);
	    }.bind(this));
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      callback: null,
	      className: 'prime-dialog',
	      closeButtonElementSelector: '[data-dialog-role="close-button"]',
	      closeTimeout: 200,
	      draggableElementSelector: '[data-dialog-role="draggable"]',
	      zIndexOffset: 1000
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class InProgress {
	  /**
	   * Constructs a In Progress widget that opens an overlay over an element while something is running and closes it when
	   * it finishes.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element to overlay.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this._setInitialOptions();
	    this.draggable = null;
	  }

	  /**
	   * Closes the InProgress process.
	   *
	   * @param {Function} callback (Optional) A callback function to invoke after the InProgress has been completely closed.
	   */
	  close(callback) {
	    try {
	      this.options.endFunction(this);
	    } finally {
	      const now = new Date().getTime();
	      const duration = now - this.startInstant;
	      if (duration < this.options.minimumTime) {
	        setTimeout(function() {
	          this.overlay.removeFromDOM();

	          if (callback) {
	            callback();
	          }
	        }.bind(this), this.options.minimumTime - duration);
	      } else {
	        this.overlay.removeFromDOM();

	        if (callback) {
	          callback();
	        }
	      }
	    }

	    return this;
	  }

	  /**
	   * Opens the InProgress process.
	   */
	  open() {
	    this.startInstant = new Date().getTime();
	    this.overlay = PrimeDocument.newElement('<div/>').setId('prime-in-progress-overlay').appendTo(document.body);
	    PrimeDocument.newElement('<i/>', {class: 'fa fa-spin fa-' + this.options.iconName}).appendTo(this.overlay);

	    const coords = this.element.getCoordinates();
	    const bodyCoords = PrimeDocument.bodyElement.getCoordinates();
	    this.overlay.setTop(coords.top - bodyCoords.top);
	    this.overlay.setLeft(coords.left - bodyCoords.left);
	    this.overlay.setWidth(this.element.getBorderedWidth());
	    this.overlay.setHeight(this.element.getBorderedHeight());
	    this.overlay.setStyle('zIndex', (this.element.getRelativeZIndex() + 1000).toString());

	    this.options.startFunction(this);

	    return this;
	  }

	  /**
	   * Sets the end function that is called when the InProgress process is finished.
	   *
	   * @param f {function} The function.
	   * @returns {InProgress} This.
	   */
	  withEndFunction(f) {
	    this.options.endFunction = f;
	    return this;
	  }

	  /**
	   * Sets the FontAwesome icon name to use for the overlay.
	   *
	   * @param iconName {string} The icon name.
	   * @returns {InProgress} This.
	   */
	  withIconName(iconName) {
	    this.options.iconName = iconName;
	    return this;
	  }

	  /**
	   * Sets the minimum time that the InProgress process must run.
	   *
	   * @param time {number} The time in milliseconds.
	   * @returns {InProgress} This.
	   */
	  withMinimumTime(time) {
	    this.options.minimumTime = time;
	    return this;
	  }

	  /**
	   * Sets the start function that is called when the InProgress process is started.
	   *
	   * @param f {function} The function.
	   * @returns {InProgress} This.
	   */
	  withStartFunction(f) {
	    this.options.startFunction = f;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      endFunction: function() {
	      },
	      iconName: 'refresh',
	      minimumTime: 1000,
	      startFunction: function() {
	      }
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2014-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class Searcher {
	  /**
	   * Constructs a Searcher object for the given text input.
	   *
	   * The Searcher object can be attached and used in conjunction with any other widgets in a generic manner. It
	   * provides search capabilities and manages the search results. This is useful for MultipleSelects, IntelliSense and
	   * other widgets. Here's the HTML for the search results.
	   *
	   * <pre>
	   *   &lt;input type="text" class="prime-search-result-input" value="F"/>
	   *   &lt;ul>
	   *     &lt;li>Four&lt;/li>
	   *     &lt;li>Five&lt;/li>
	   *     &lt;li>Fifteen&lt;/li>
	   *     &lt;li>Add Custom Entry: F/li>
	   *   &lt;/ul>
	   * &lt;/div>
	   * </pre>
	   *
	   * The with* methods can be used to setup the configuration for this SearchResults, but here are some defaults:
	   *
	   * <ul>
	   *   <li>closeTimeout = 200</li>
	   *   <li>customAddEnabled = true</li>
	   *   <li>customAddCallback = function(customValue){return true;}</li>
	   *   <li>customAddLabel = "Add Custom:"</li>
	   *   <li>tooManySearchResultsLabel = "Too Many Matches For:"</li>
	   *   <li>noSearchResultsLabel = "No Matches For:"</li>
	   * </ul>
	   *
	   * The callback object must conform to this interface:
	   *
	   * <pre>
	   *   CallbackObject {
	 *     void deletedBeyondSearchInput()
	 *     void doesNotContainValue()
	 *     object{results:Array, tooManyResults:boolean} search(_searchString:string),
	 *     void selectSearchResult(selectedSearchResult:string),
	 *   }
	   * </pre>
	   *
	   * @constructor
	   * @param {PrimeElement|Element|EventTarget} inputElement The input element that is used to execute the search.
	   * @param {PrimeElement|Element|EventTarget} searchResultsContainer The element that is used to store the search results.
	   * @param {*} callbackObject The object that is used to callback for searching and numerous other functions to help
	   *            communicate state and determine how to draw the input and search results.
	   */
	  constructor(inputElement, searchResultsContainer, callbackObject) {
	    Utils.bindAll(this);

	    this.searchResults = PrimeElement.wrap(searchResultsContainer);
	    this.inputElement = PrimeElement.wrap(inputElement);
	    if (this.inputElement.domElement.tagName !== 'INPUT') {
	      throw new TypeError('You can only use Prime.Widgets.SearchResults with INPUT elements');
	    }

	    this._setInitialOptions(callbackObject);
	  }

	  /**
	   * A search function that works on a select box.
	   *
	   * @param searchText {String} The search String.
	   * @param select {HTMLSelectElement|PrimeElement} The select box.
	   * @returns {{results: Array, tooManyResults: boolean}}
	   */
	  static selectSearchFunction(searchText, select) {
	    const options = PrimeElement.unwrap(select).options;
	    const selectableOptions = [];
	    for (let i = 0; i < options.length; i++) {
	      const option = new PrimeElement(options[i]);
	      if (option.isSelected()) {
	        continue;
	      }

	      const html = option.getHTML();
	      if (searchText === null || searchText === undefined || searchText === '' || html.toLowerCase().indexOf(searchText.toLowerCase()) === 0) {
	        selectableOptions.push(html);
	      }
	    }

	    // Alphabetize the options
	    if (selectableOptions.length > 0) {
	      selectableOptions.sort();
	    }

	    return {results: selectableOptions, tooManyResults: false};
	  }

	  /**
	   * Closes the search results display, unhighlights any options that are highlighted and resets the input's value to
	   * empty string.
	   */
	  closeSearchResults() {
	    this._removeAllSearchResults();
	    this.inputElement.setValue('');
	    this.searchResults.removeClass('open');
	    setTimeout(function() {
	      this.searchResults.hide();
	      this.resizeInput();
	    }.bind(this), this.options.closeTimeout);
	  }

	  /**
	   * Removes all of the event listeners from the input element.
	   */
	  destroy() {
	    this.inputElement
	        .removeEventListener('blur', this._handleBlurEvent)
	        .removeEventListener('click', this._handleClickEvent)
	        .removeEventListener('keyup', this._handleKeyUpEvent)
	        .removeEventListener('keydown', this._handleKeyDownEvent)
	        .removeEventListener('focus', this._handleFocusEvent);
	  }

	  focus() {
	    this.inputElement.focus();
	  }

	  /**
	   * @returns {PrimeElement} The highlighted search result or null.
	   */
	  getHighlightedSearchResult() {
	    return this.searchResults.queryFirst('.selected');
	  }

	  /**
	   * Highlights the next search result if one is highlighted. If there isn't a highlighted search result, this
	   * highlights the first one. This method handles wrapping.
	   *
	   * @returns {Searcher} This Searcher.
	   */
	  highlightNextSearchResult() {
	    let searchResult = this.getHighlightedSearchResult();
	    if (searchResult !== null) {
	      searchResult = searchResult.getNextSibling();
	    }

	    // Grab the first search result in the list if there isn't a next sibling
	    if (searchResult === null) {
	      searchResult = this.searchResults.queryFirst('.search-result');
	    }

	    if (searchResult !== null) {
	      this.highlightSearchResult(searchResult);
	    }

	    return this;
	  }

	  /**
	   * Highlights the previous search result if one is highlighted. If there isn't a highlighted search result, this
	   * selects the last one. This method handles wrapping.
	   *
	   * @returns {Searcher} This Searcher.
	   */
	  highlightPreviousSearchResult() {
	    let searchResult = this.getHighlightedSearchResult();
	    if (searchResult !== null) {
	      searchResult = searchResult.getPreviousSibling();
	    }

	    if (searchResult === null) {
	      searchResult = this.searchResults.queryFirst('.search-result');
	    }

	    if (searchResult !== null) {
	      this.highlightSearchResult(searchResult);
	    }

	    return this;
	  }

	  /**
	   * Highlights the given search result.
	   *
	   * @param {PrimeElement} searchResult The search result to highlight.
	   * @returns {Searcher} This Searcher.
	   */
	  highlightSearchResult(searchResult) {
	    this.searchResults.getChildren().removeClass('selected');

	    searchResult.addClass('selected');
	    const scrollTop = this.searchResults.getScrollTop();
	    const height = this.searchResults.getHeight();
	    const searchResultOffset = searchResult.getOffsetTop();
	    if (searchResultOffset + 1 >= scrollTop + height) {
	      this.searchResults.scrollTo(searchResult.getOffsetTop() - this.searchResults.getHeight() + searchResult.getOuterHeight());
	    } else if (searchResultOffset < scrollTop) {
	      this.searchResults.scrollTo(searchResultOffset);
	    }

	    return this;
	  }

	  /**
	   * Initializes the Searcher by setting up the event listeners and closing the search result element.
	   *
	   * @returns {Searcher} This.
	   */
	  initialize() {
	    this.inputElement
	        .addEventListener('blur', this._handleBlurEvent)
	        .addEventListener('click', this._handleClickEvent)
	        .addEventListener('keyup', this._handleKeyUpEvent)
	        .addEventListener('keydown', this._handleKeyDownEvent)
	        .addEventListener('focus', this._handleFocusEvent);

	    this.closeSearchResults();
	    return this;
	  }

	  /**
	   * @returns {boolean} True if the search results add custom option is being displayed currently.
	   */
	  isCustomAddVisible() {
	    return this.searchResults.queryFirst('.custom-add') !== null;
	  }

	  /**
	   * @returns {boolean} True if any search results are being displayed currently.
	   */
	  isSearchResultsVisible() {
	    return this.searchResults.hasClass('open');
	  }

	  /**
	   * Poor mans resizing of the input field as the user types into it.
	   */
	  resizeInput() {
	    const text = this.inputElement.getValue() === '' ? this.inputElement.getAttribute('placeholder') : this.inputElement.getValue();
	    const newLength = Utils.calculateTextLength(this.inputElement, text) + 35;
	    this.inputElement.setWidth(newLength);
	  }

	  /**
	   * Executes a search by optionally updating the input to the given value (if specified) and then rebuilding the search
	   * results using the input's value. This method also puts focus on the input and shows the search results (in case
	   * they are hidden for any reason).
	   *
	   * @param {string} [searchText] The text to search for (this value is also set into the input box). If this is not
	   * specified then the search is run using the input's value.
	   * @returns {Searcher} This Searcher.
	   */
	  search(searchText) {
	    // Set the search text into the input box if it is different and then lowercase it
	    if (Utils.isDefined(searchText) && this.inputElement.getValue() !== searchText) {
	      this.inputElement.setValue(searchText);
	    }

	    searchText = Utils.isDefined(searchText) ? searchText.toLowerCase() : this.inputElement.getValue();
	    this.resizeInput();

	    // Clear the search results (if there are any)
	    this._removeAllSearchResults();

	    // Call the callback
	    const searchResults = this.options.callbackObject.search(searchText);
	    if (!searchResults.hasOwnProperty('results') || !searchResults.hasOwnProperty('tooManyResults')) {
	      throw new TypeError('The callback must return an Object that contains the properties results[Array] and tooManyResults[boolean]');
	    }

	    let count = 0;
	    let matchingSearchResultElement = null;
	    for (let i = 0; i < searchResults.results.length; i++) {
	      const searchResult = searchResults.results[i];
	      const element = PrimeDocument.newElement('<li/>')
	          .addClass('search-result')
	          .setAttribute('value', searchResult)
	          .setHTML(searchResult)
	          .addEventListener('click', this._handleClickEvent)
	          .addEventListener('mouseover', this._handleMouseOverEvent)
	          .appendTo(this.searchResults);
	      if (searchResult.toLowerCase().trim() === searchText.toLowerCase().trim()) {
	        matchingSearchResultElement = element;
	      }

	      count++;
	    }

	    // Show the custom add option if necessary
	    const trimmedLength = searchText.trim().length;
	    if (this.options.customAddEnabled && trimmedLength !== 0 && matchingSearchResultElement === null
	        && (!('doesNotContainValue' in this.options.callbackObject) || this.options.callbackObject.doesNotContainValue(searchText))) {
	      matchingSearchResultElement = PrimeDocument.newElement('<li/>')
	          .addClass('custom-add')
	          .addEventListener('click', this._handleClickEvent)
	          .addEventListener('mouseover', this._handleMouseOverEvent)
	          .setHTML(this.options.customAddLabel + Utils.escapeHTML(searchText))
	          .appendTo(this.searchResults);
	      count++;
	    }

	    if (count === 0 && trimmedLength !== 0) {
	      PrimeDocument.newElement('<li/>')
	          .addClass('no-search-results')
	          .setHTML(this.options.noSearchResultsLabel + Utils.escapeHTML(searchText))
	          .appendTo(this.searchResults);
	      count++;
	    }

	    // Handle too many results
	    if (searchResults.tooManyResults) {
	      PrimeDocument.newElement('<li/>')
	          .addClass('too-many-search-results')
	          .setHTML(this.options.tooManySearchResultsLabel + Utils.escapeHTML(searchText))
	          .appendTo(this.searchResults);
	      count++;
	    }

	    if (count !== 0) {
	      this.searchResults.show();
	      this.searchResults.addClass('open');

	      if (count >= 10) {
	        this.searchResults.setHeight(this.searchResults.getChildren()[0].getOuterHeight() * 10 + 1);
	      } else {
	        this.searchResults.setHeight(this.searchResults.getChildren()[0].getOuterHeight() * count + 1);
	      }
	    } else {
	      this.closeSearchResults();
	    }

	    if (matchingSearchResultElement !== null) {
	      this.highlightSearchResult(matchingSearchResultElement);
	    }

	    return this;
	  }

	  /**
	   * Selects the highlighted search result unless there isn't one highlighted, in which case, this does nothing.
	   *
	   * @returns {Searcher} This Searcher.
	   */
	  selectHighlightedSearchResult() {
	    const searchResult = this.getHighlightedSearchResult();
	    if (searchResult === null) {
	      return this;
	    }

	    const custom = searchResult.hasClass('custom-add');
	    const value = (custom) ? this.inputElement.getValue().trim() : searchResult.getHTML();
	    if (custom) {
	      // The client of this searcher needs to warn the user.
	      if (!this.options.customAddCallback(value)) {
	        return this;
	      }
	    }

	    this.options.callbackObject.selectSearchResult(value);
	    this.closeSearchResults();

	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {Searcher} This.
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Sets whether or not this Searcher allows custom options to be added.
	   *
	   * @param {boolean} enabled The flag.
	   * @returns {Searcher} This Searcher.
	   */
	  withCustomAddEnabled(enabled) {
	    this.options.customAddEnabled = enabled;
	    return this;
	  }

	  /**
	   * Sets whether or not this Searcher allows custom options to be added.
	   *
	   * @param {Function} callback The function to call that will return true if the custom option can be added.
	   * @returns {Searcher} This Searcher.
	   */
	  withCustomAddCallback(callback) {
	    this.options.customAddCallback = callback;
	    return this;
	  }

	  /**
	   * Sets the label used when custom options are added.
	   *
	   * @param {string} customAddLabel The label.
	   * @returns {Searcher} This Searcher.
	   */
	  withCustomAddLabel(customAddLabel) {
	    this.options.customAddLabel = customAddLabel;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {Searcher} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /**
	   * Sets the label that is printed when there are no search results.
	   *
	   * @param {string} noSearchResultsLabel The label text.
	   * @returns {Searcher} This Searcher.
	   */
	  withNoSearchResultsLabel(noSearchResultsLabel) {
	    this.options.noSearchResultsLabel = noSearchResultsLabel;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Sets the label that is printed when there are too many search results.
	   *
	   * @param {string} tooManySearchResultsLabel The label text.
	   * @returns {Searcher} This Searcher.
	   */
	  withTooManySearchResultsLabel(tooManySearchResultsLabel) {
	    this.options.tooManySearchResultsLabel = tooManySearchResultsLabel;
	    return this;
	  }

	  /**
	   * Handles the blur event when the input goes out of focus.
	   *
	   * @private
	   */
	  _handleBlurEvent() {
	    window.setTimeout((function() {
	      if (document.activeElement !== this.inputElement.domElement) {
	        this.closeSearchResults();
	      }
	    }).bind(this), 300);
	  }

	  /**
	   * Handles all click events sent to the Searcher.
	   *
	   * @param {Event} event The mouse event.
	   * @private
	   */
	  _handleClickEvent(event) {
	    const target = new PrimeElement(event.currentTarget);
	    if (target.hasClass('custom-add') || target.hasClass('search-result')) {
	      this.selectHighlightedSearchResult();
	    } else if (target.domElement === this.inputElement.domElement) {
	      this.search();
	    } else {
	      console.log('Clicked something else target=[' + event.target + '] currentTarget=[' + event.currentTarget + ']');
	    }
	  }

	  /**
	   * Handles when the input field is focused by opening the search results.
	   *
	   * @private
	   */
	  _handleFocusEvent() {
	    this.search();
	  }

	  /**
	   * Handles the key down events that should not be propagated.
	   *
	   * @param {KeyboardEvent} event The keyboard event object.
	   * @private
	   */
	  _handleKeyDownEvent(event) {
	    const key = event.keyCode;
	    if (key === Events.Keys.BACKSPACE) {
	      this.previousSearchString = this.inputElement.getValue();
	    } else if (key === Events.Keys.UP_ARROW) {
	      Utils.stopEvent(event);
	      this.highlightPreviousSearchResult();
	    } else if (key === Events.Keys.DOWN_ARROW) {
	      Utils.stopEvent(event);
	      if (this.isSearchResultsVisible()) {
	        this.highlightNextSearchResult();
	      } else {
	        this.search();
	      }
	    } else if (key === Events.Keys.ENTER) {
	      Utils.stopEvent(event); // Don't bubble enter otherwise the form submits
	    }
	  }

	  /**
	   * Handles all key up events sent to the search results container.
	   *
	   * @param {KeyboardEvent} event The keyboard event object.
	   *  @private
	   */
	  _handleKeyUpEvent(event) {
	    const key = event.keyCode;
	    const value = this.inputElement.getValue();

	    if (key === Events.Keys.BACKSPACE) {
	      if (value === '' && this.previousSearchString === '') {
	        this.options.callbackObject.deletedBeyondSearchInput();
	      } else {
	        this.search();
	      }
	    } else if (key === Events.Keys.ENTER) {
	      Utils.stopEvent(event);
	      // If a search result is highlighted, add it
	      if (this.getHighlightedSearchResult() !== null) {
	        this.selectHighlightedSearchResult();
	      }
	    } else if (key === Events.Keys.ESCAPE) {
	      this.closeSearchResults();
	    } else if (key === Events.Keys.SPACE || key === Events.Keys.DELETE ||
	        (key >= 48 && key <= 90) || (key >= 96 && key <= 111) || (key >= 186 && key <= 192) || (key >= 219 && key <= 222)) {
	      this.search();
	    }
	  }

	  /**
	   * Handles mouseover events for the search results (only) by highlighting the event target.
	   *
	   * @param {Event} event The mouseover event.
	   * @private
	   */
	  _handleMouseOverEvent(event) {
	    const target = new PrimeElement(event.currentTarget);
	    this.highlightSearchResult(target);
	  }

	  /**
	   * Removes all of the search results.
	   *
	   * @private
	   */
	  _removeAllSearchResults() {
	    this.searchResults.query('li').removeAllFromDOM();
	  }

	  /* ===================================================================================================================
	   * Search function implementations.
	   * ===================================================================================================================*/

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions(callbackObject) {
	    // Defaults
	    this.options = {
	      callbackObject: callbackObject,
	      closeTimeout: 200,
	      customAddEnabled: true,
	      'customAddCallback'() {
	        return true;
	      },
	      customAddLabel: 'Add Custom: ',
	      noSearchResultsLabel: 'No Matches For: ',
	      tooManySearchResultsLabel: 'Too Many Matches For: ',
	    };

	    const userOptions = Utils.dataSetToOptions(this.inputElement);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	let count = 1;
	let AddOptionEvent = 'MultipleSelect:addOption';
	let DeselectOptionEvent = 'MultipleSelect:deselectOption';
	let SelectOptionEvent = 'MultipleSelect:selectOption';

	class MultipleSelect {
	  /**
	   * Constructs a MultipleSelect object for the given element.
	   *
	   * The MultipleSelect generates a number of different HTML elements directly after the SELECT element you pass to the
	   * constructor. A fully rendered MultipleSelect might look something like this:
	   *
	   * <pre>
	   * &lt;select id="foo">
	   *   &lt;option value="one">One&lt;/option>
	   *   &lt;option value="two">Two&lt;/option>
	   *   &lt;option value="three">Three&lt;/option>
	   * &lt;/select>
	   * &lt;div id="foo-display" class="prime-multiple-select">
	   *   &lt;ul id="foo-option-list" class="option-list">
	   *     &lt;li id="foo-option-one">&lt;span>One&lt;/span>&lt;a href="#">X&lt;/a>&lt;/li>
	   *     &lt;li id="foo-option-two">&lt;span>Two&lt;/span>&lt;a href="#">X&lt;/a>&lt;/li>
	   *     &lt;li id="foo-option-three">&lt;span>Three&lt;/span>&lt;a href="#">X&lt;/a>&lt;/li>
	   *     &lt;li>&lt;input type="text" value="aaa"/>&lt;/li>
	   *   &lt;/ul>
	   *   &lt;ul class="search-results">
	   *     &lt;li>One&lt;/li>
	   *     &lt;li>Two&lt;/li>
	   *     &lt;li>Three&lt;/li>
	   *     &lt;li>Add Custom Entry: aaa/li>
	   *   &lt;/ul>
	   * &lt;/div>
	   * </pore>
	   *
	   * The with* methods can be used to setup the configuration for this MultipleSelect, but here are some defaults:
	   *
	   * <ul>
	   *   <li>placeholder = "Choose"</li>
	   *   <li>customAddEnabled = true</li>
	   *   <li>customAddLabel = "Add Custom Value:"</li>
	   *   <li>noSearchResultsLabel = "No Matches For:"</li>
	   * </ul>
	   *
	   * @constructor
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the MultipleSelect.
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    if (this.element.domElement.tagName !== 'SELECT') {
	      throw new TypeError('You can only use MultipleSelect with select elements');
	    }

	    if (this.element.getAttribute('multiple') !== 'multiple') {
	      throw new TypeError('The select box you are attempting to convert to a MultipleSelect must have the multiple="multiple" attribute set');
	    }

	    this._setInitialOptions();
	  }

	  /*
	   * Statics
	   */
	  /**
	   * @returns {number}
	   */
	  static get count() {
	    return count;
	  }

	  /**
	   * @param {number} value
	   */
	  static set count(value) {
	    count = value;
	  }

	  /**
	   * @returns {string}
	   */
	  static get AddOptionEvent() {
	    return AddOptionEvent;
	  }

	  /**
	   * @param {string} value
	   */
	  static set AddOptionEvent(value) {
	    AddOptionEvent = value;
	  }

	  /**
	   * @returns {string}
	   */
	  static get DeselectOptionEvent() {
	    return DeselectOptionEvent;
	  }

	  /**
	   * @param {string} value
	   */
	  static set DeselectOptionEvent(value) {
	    DeselectOptionEvent = value;
	  }

	  /**
	   * @returns {string}
	   */
	  static get SelectOptionEvent() {
	    return SelectOptionEvent;
	  }

	  /**
	   * @param {string} value
	   */
	  static set SelectOptionEvent(value) {
	    SelectOptionEvent = value;
	  }

	  /**
	   * Finds the HTMLSelectOption with the given id and returns it wrapped in a PrimeElement.
	   *
	   * @param {String} id
	   * @returns {PrimeElement}
	   */
	  static findOptionWithId(id) {
	    return PrimeDocument.queryFirst('[data-option-id="' + id + '"]');
	  }

	  /**
	   * Pass through to add event listeners to This. The custom events that this MultipleSelect fires are:
	   *
	   *  'MultipleSelect:deselectOption'
	   *  'MultipleSelect:selectOption'
	   *  'MultipleSelect:addOption'
	   *
	   * @param {string} event The name of the event.
	   * @param {Function} listener The listener function.
	   * @returns {MultipleSelect} This.
	   */
	  addEventListener(event, listener) {
	    this.element.addEventListener(event, listener);
	    return this;
	  }

	  /**
	   * Determines if this MultipleSelect contains an option with the given value.
	   *
	   * @param {String} value The value to look for.
	   */
	  containsOptionWithValue(value) {
	    return this.findOptionWithValue(value) !== null;
	  }

	  /**
	   * Adds the given option to this select. The option will not be selected.
	   *
	   * @param {String} value The value for the option.
	   * @param {String} display The display text for the option.
	   * @param {?String} [id] The id of the element. (Defaults to null)
	   * @returns {MultipleSelect} This.
	   */
	  addOption(value, display, id) {
	    if ((id === null || id === undefined) && this.containsOptionWithValue(value)) {
	      return this;
	    }

	    let element = PrimeDocument.newElement('<option/>')
	        .setValue(value)
	        .setHTML(display)
	        .appendTo(this.element);

	    if (id) {
	      element.setDataAttribute("optionId", id);
	    }

	    // Fire the custom event
	    this.element.fireEvent(MultipleSelect.AddOptionEvent, value, this);

	    return this;
	  }

	  /**
	   * Deselects the option with the given value by removing the selected attribute from the option in the select box and
	   * removing the option from the display container. If the MultipleSelect doesn't contain an option for the given value,
	   * this method throws an exception.
	   *
	   * @param {String} value The value to look for.
	   * @returns {MultipleSelect} This.
	   */
	  deselectOptionWithValue(value) {
	    const option = this.findOptionWithValue(value);
	    if (option === null) {
	      throw new Error('MultipleSelect doesn\'t contain an option with the value [' + value + ']');
	    }

	    this.deselectOption(option);

	    return this;
	  }

	  /**
	   * Deselects the option with the given value by removing the selected attribute from the option in the select box and
	   * removing the option from the display container.
	   *
	   * @param {PrimeElement} option The option to deselect.
	   * @returns {MultipleSelect} This.
	   */
	  deselectOption(option) {
	    option.setSelected(false);

	    const id = option.getDataAttribute('optionId') || this._makeOptionID(option);
	    const displayOption = PrimeDocument.queryById(id);
	    if (displayOption !== null) {
	      displayOption.removeFromDOM();
	    }

	    // If there are no selected options left, add back the placeholder attribute to the input and resize it
	    if (this.optionList.query('li').length === 1) {
	      this.input.setAttribute('placeholder', this.options.placeholder);
	      this.searcher.resizeInput();
	    }

	    // Fire the custom event
	    this.element.fireEvent(MultipleSelect.DeselectOptionEvent, option.getValue(), this);

	    return this;
	  }

	  /**
	   * Destroys the widget completely.
	   */
	  destroy() {
	    this.element.show();
	    this.displayContainer.removeFromDOM();
	  }

	  /**
	   * Finds the HTMLSelectOption with the given text and returns it wrapped in a PrimeElement.
	   *
	   * @param {String} text The text to look for.
	   * @returns {PrimeElement} The option element or null.
	   */
	  findOptionWithText(text) {
	    const options = this.element.getOptions();
	    for (let i = 0; i < options.length; i++) {
	      if (options[i].getTextContent() === text) {
	        return options[i];
	      }
	    }

	    return null;
	  }

	  deselectOptionWithId(id) {
	    const option = MultipleSelect.findOptionWithId(id);
	    if (option === null) {
	      throw new Error('MultipleSelect doesn\'t contain an option with the id [' + id + ']');
	    }

	    this.deselectOption(option);

	    return this;
	  }

	  /**
	   * Finds the HTMLSelectOption with the given value and returns it wrapped in a PrimeElement.
	   *
	   * @param {String} value The value to look for.
	   * @returns {PrimeElement} The option element or null.
	   */
	  findOptionWithValue(value) {
	    for (let i = 0; i < this.element.domElement.length; i++) {
	      const cur = this.element.domElement.options[i];
	      if (cur.value === value) {
	        return new PrimeElement(cur);
	      }
	    }

	    return null;
	  }

	  /**
	   * @returns {string[]} The currently selected options values.
	   */
	  getSelectedValues() {
	    return this.element.getSelectedValues();
	  }

	  /**
	   * Determines if the MultipleSelect contains an option with the given value.
	   *
	   * @param {string} value The value.
	   * @returns {boolean} True if the MultipleSelect contains an option with the given value, false otherwise.
	   */
	  hasOptionWithValue(value) {
	    return this.findOptionWithValue(value) !== null;
	  }

	  /**
	   * Highlights the final selected option (if there is one) to indicate that it will be unselected if the user clicks
	   * the delete key again.
	   *
	   * @returns {MultipleSelect} This.
	   */
	  highlightOptionForUnselect() {
	    const options = this.optionList.getChildren();
	    if (options.length > 1) {
	      options[options.length - 2].addClass('selected');
	    }

	    return this;
	  }

	  /**
	   * Initializes the display from the underlying select element. All of the current display options (li elements) are
	   * removed. New display options are added for each selected option in the select box.
	   *
	   * @returns {MultipleSelect} This.
	   */
	  initialize() {
	    this.element.hide();

	    let id = this.element.getId();
	    if (id === null || id === '') {
	      id = 'prime-multiple-select' + MultipleSelect.count++;
	      this.element.setId(id);
	    }

	    this.displayContainer = PrimeDocument.queryById(id + '-display');
	    this.input = null;
	    if (this.displayContainer === null) {
	      this.displayContainer = PrimeDocument.newElement('<div/>')
	          .setId(id + '-display')
	          .addClass(this.options.className)
	          .addEventListener('click', this._handleClickEvent)
	          .addEventListener('keyup', this._handleKeyUpEvent)
	          .insertAfter(this.element);

	      this.optionList = PrimeDocument.newElement('<ul/>')
	          .addClass('option-list')
	          .appendTo(this.displayContainer);

	      this.searchResults = PrimeDocument.newElement('<ul/>')
	          .addClass('search-results')
	          .hide()
	          .appendTo(this.displayContainer);
	    } else {
	      this.displayContainer
	          .removeAllEventListeners()
	          .addEventListener('click', this._handleClickEvent)
	          .addEventListener('keyup', this._handleKeyUpEvent);
	      this.optionList = this.displayContainer.queryFirst('.option-list');
	      this.searchResults = this.displayContainer.queryFirst('.search-results');
	    }

	    PrimeDocument.queryFirst('html').addEventListener('click', this._handleGlobalClickEvent);

	    // Close the search
	    this.searchResults.hide();

	    this._redraw();

	    return this;
	  }

	  /**
	   * @returns {boolean} True if the last option is highlighted for unselect.
	   */
	  isLastOptionHighlightedForUnselect() {
	    const options = this.optionList.getChildren();
	    return options.length > 1 && options[options.length - 2].hasClass('selected');
	  }

	  /**
	   * Removes all of the options from the MultipleSelect.
	   *
	   * @returns {MultipleSelect} This.
	   */
	  removeAllOptions() {
	    // Remove in reverse order because the options array is dynamically updated when elements are deleted from the DOM
	    const options = this.element.domElement.options;
	    for (let i = options.length - 1; i >= 0; i--) {
	      this.removeOption(new PrimeElement(options[i]));
	    }

	    return this;
	  }

	  /**
	   * Removes the highlighted option.
	   */
	  removeHighlightedOption() {
	    const options = this.optionList.getChildren();
	    if (this.options.allowDuplicates) {
	      this.deselectOptionWithId(options[options.length - 2].getId());
	    } else {
	      this.deselectOptionWithValue(options[options.length - 2].getAttribute('value'));
	    }
	    this.search();
	  }

	  /**
	   * Removes the given option from the MultipleSelect by removing the option in the select box and the option in the
	   * display container.
	   *
	   * @param {PrimeElement} option The option to remove.
	   * @returns {MultipleSelect} This.
	   */
	  removeOption(option) {
	    if (!(option instanceof PrimeElement)) {
	      throw new TypeError('MultipleSelect#removeOption only takes PrimeElement instances');
	    }

	    option.removeFromDOM();

	    let id, displayOption;
	    if (this.options.allowDuplicates) {
	      // The ids are random so we need to get the data attribute.
	      id = option.getDataAttribute('optionId');
	      displayOption = PrimeDocument.queryById(id);
	    } else {
	      // The ids aren't random and can be reproducably created.
	      id = this._makeOptionID(option);
	      displayOption = PrimeDocument.queryById(id);
	    }

	    // Check if the option has already been selected
	    if (displayOption !== null) {
	      displayOption.removeFromDOM();
	    }

	    return this;
	  }

	  /**
	   * Removes the option with the given value from the MultipleSelect by removing the option in the select box and the
	   * option in the display container. If the MultipleSelect doesn't contain an option with the given value, this throws
	   * an exception.
	   *
	   * @param {string} value The value of the option to remove.
	   * @returns {MultipleSelect} This.
	   */
	  removeOptionWithValue(value) {
	    const option = this.findOptionWithValue(value);
	    if (option === null) {
	      throw new Error('MultipleSelect doesn\'t contain an option with the value [' + value + ']');
	    }

	    this.removeOption(option);

	    return this;
	  }

	  /**
	   * Selects the given option by setting the selected attribute on the option in the select box (the object passed in is
	   * the option from the select box wrapped in a PrimeElement) and adding it to the display container. If the
	   * option is already in the display container, that step is skipped.
	   *
	   * @param {PrimeElement} option The option object from the select box wrapped in a PrimeElement instance.
	   * @returns {MultipleSelect} This.
	   */
	  selectOption(option) {
	    if (!(option instanceof PrimeElement)) {
	      throw new TypeError('MultipleSelect#selectOption only takes PrimeElement instances');
	    }

	    const id = this._makeOptionID(option);

	    // Check if the option has already been selected
	    if (PrimeDocument.queryById(id) === null) {
	      /*
	      If we allow dupes, always duplicate the option and append it to the end or the order will be a problem. The default multiselect doesn't support order)
	       */
	      if (this.options.allowDuplicates) {
	        this.addOption(option.getTextContent(), option.getTextContent(), id);
	        option = MultipleSelect.findOptionWithId(id);
	      }
	      option.setSelected(true);

	      const li = PrimeDocument.newElement('<li/>')
	          .setAttribute('value', option.getValue())
	          .setId(id)
	          .insertBefore(this.inputOption);
	      PrimeDocument.newElement('<span/>')
	          .setHTML(option.getHTML())
	          .setAttribute('value', option.getValue())
	          .appendTo(li);
	      PrimeDocument.newElement('<a/>')
	          .setAttribute('href', '#')
	          .setAttribute('value', option.getValue())
	          .setHTML(this.options.removeIcon)
	          .addEventListener('click', this._handleClickEvent)
	          .appendTo(li);
	    }

	    // Remove the placeholder attribute on the input and resize it
	    this.input.removeAttribute('placeholder');

	    // Close the search results and resize the input
	    this.searcher.closeSearchResults();

	    // Scroll the display to the bottom
	    this.optionList.scrollToBottom();

	    // Fire the custom event
	    this.element.fireEvent(MultipleSelect.SelectOptionEvent, option.getValue(), this);

	    return this;
	  }

	  /**
	   * Selects the option with the given value by setting the selected attribute on the option in the select box (the
	   * object passed in is the option from the select box wrapped in a PrimeElement) and adding it to the display
	   * container. If the option is already in the display container, that step is skipped.
	   * <p/>
	   * If there isn't an option with the given value, this throws an exception.
	   *
	   * @param {String} value The value of the option to select.
	   * @returns {MultipleSelect} This.
	   */
	  selectOptionWithValue(value) {
	    const option = this.findOptionWithValue(value);
	    if (option === null) {
	      throw new Error('MultipleSelect doesn\'t contain an option with the value [' + value + ']');
	    }

	    this.selectOption(option);

	    return this;
	  }

	  /**
	   * Sets the selected options. This mimics the function on Element to provide consistency.
	   *
	   * @param {string[]} [arguments] The list of options to select based on their values.
	   * @returns {MultipleSelect} This.
	   */
	  setSelectedValues() {
	    this.element.setSelectedValues.apply(this.element, arguments);
	    this._redraw();
	    return this;
	  }

	  /**
	   * Unhighlights the last option if it is highlighted.
	   *
	   * @returns {MultipleSelect} This.
	   */
	  unhighlightOptionForUnselect() {
	    this.optionList.getChildren().each(function(element) {
	      element.removeClass('selected');
	    });
	    return this;
	  }

	  withAllowDuplicates(value) {
	    this.options.allowDuplicates = value;
	    return this;
	  }

	  /**
	   * Sets the class name for the MultipleSelect element.
	   *
	   * @param className {string} The class name.
	   * @returns {MultipleSelect} This.
	   */
	  withClassName(className) {
	    this.options.className = className;
	    return this;
	  }

	  /**
	   * Sets the timeout used in the close method to allow for transitions.
	   *
	   * @param timeout {int} The timeout.
	   * @returns {MultipleSelect} This.
	   */
	  withCloseTimeout(timeout) {
	    this.options.closeTimeout = timeout;
	    return this;
	  }

	  /**
	   * Sets whether or not this MultipleSelect allows custom options to be added.
	   *
	   * @param {boolean} enabled The flag.
	   * @returns {MultipleSelect} This.
	   */
	  withCustomAddEnabled(enabled) {
	    this.options.customAddEnabled = enabled;
	    return this;
	  }

	  /**
	   * Sets the label used when custom options are added.
	   *
	   * @param {string} customAddLabel The label.
	   * @returns {MultipleSelect} This.
	   */
	  withCustomAddLabel(customAddLabel) {
	    this.options.customAddLabel = customAddLabel;
	    return this;
	  }

	  /**
	   * Enable error class handling. When this option is used, if the specified error class is found on any element
	   * in the tab content the same error class will be added to the tab to identify the tab contains errors.
	   *
	   * @returns {MultipleSelect} This.
	   */
	  withErrorClassHandling(errorClass) {
	    this.options.errorClass = errorClass;
	    return this;
	  }

	  /**
	   * Sets the label that is printed when there are no search results. This must be called before render is called.
	   *
	   * @param {string} noSearchResultsLabel The label text.
	   * @returns {MultipleSelect} This.
	   */
	  withNoSearchResultsLabel(noSearchResultsLabel) {
	    this.options.noSearchResultsLabel = noSearchResultsLabel;
	    return this;
	  }

	  /**
	   * Sets the placeholder text for This. This must be called before render is called.
	   *
	   * @param {string} placeholder The placeholder text.
	   * @returns {MultipleSelect} This.
	   */
	  withPlaceholder(placeholder) {
	    this.options.placeholder = placeholder;
	    return this;
	  }

	  /**
	   * Sets the remove icon value. This overrides the default value.
	   *
	   * @param {string} removeIcon The remove icon text.
	   * @returns {MultipleSelect} This.
	   */
	  withRemoveIcon(removeIcon) {
	    this.options.removeIcon = removeIcon;
	    return this;
	  }

	  /**
	   * Sets the search function that can be used to search other sources besides the select box that backs this widget.
	   *
	   * @param searchFunction {Function} The search function.
	   * @returns {MultipleSelect} This.
	   */
	  withSearchFunction(searchFunction) {
	    this.options.searchFunction = searchFunction;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Searcher's callback interface methods.
	   * ===================================================================================================================*/

	  /**
	   * Called when the Searcher gets a keyboard event that deletes beyond the search input. This highlights the last word
	   * in the phrase for removal.
	   */
	  deletedBeyondSearchInput() {
	    if (this.isLastOptionHighlightedForUnselect()) {
	      this.removeHighlightedOption();
	    }

	    this.highlightOptionForUnselect();
	  }

	  /**
	   * Called when the search needs to determine if the custom add option should be displayed. As long as this
	   * MultipleSelect does not contain the given value, the custom add option should be displayed.
	   *
	   * @param {string} value The value.
	   * @returns {boolean} True if this MultipleSelect does not contain the value, false otherwise.
	   */
	  doesNotContainValue(value) {
	    return !this.containsOptionWithValue(value);
	  }

	  /**
	   * Called when the Searcher is executing a search. This executes a search via the callback and returns the results.
	   *
	   * @param {string} [searchText] The text to search for.
	   * @returns {Object} The SearchResults.
	   */
	  search(searchText) {
	    this.unhighlightOptionForUnselect();
	    return this.options.searchFunction.call(null, searchText, this.element);
	  }

	  /**
	   * Called when the Searcher gets an event that causes a search result to be selected. This adds the word.
	   */
	  selectSearchResult(value) {
	    // Add the custom option if there is one
	    let option = this.findOptionWithText(value);
	    if (option === null) {
	      this.addOption(value, value);
	      option = this.findOptionWithText(value);
	    }

	    this.selectOption(option);
	  }


	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the blur event when the input goes out of focus.
	   *
	   * @private
	   */
	  _handleBlurEvent() {
	    window.setTimeout((function() {
	      if (document.activeElement !== this.input.domElement) {
	        this.searcher.closeSearchResults();
	      }
	    }).bind(this), 300);
	    this.displayContainer.removeClass('focus');
	  }

	  /**
	   * Handles all click events sent to the MultipleSelect.
	   *
	   * @param {Event} event The mouse event.
	   * @private
	   */
	  _handleClickEvent(event) {
	    Utils.stopEvent(event);
	    const target = new PrimeElement(event.target);
	    if (target.is('a')) {
	      if (this.options.allowDuplicates) {
	        const id = target.getParent().getId();
	        this.removeOption(MultipleSelect.findOptionWithId(id));
	      } else {
	        this.removeOptionWithValue(target.getAttribute('value'));
	      }
	    } else if (target.is('span')) {
	      target.selectElementContents();
	    } else {
	      this.input.focus();
	    }
	  }

	  /**
	   * Handles the blur event when the input goes out of focus.
	   *
	   * @private
	   */
	  _handleFocusEvent() {
	    this.displayContainer.addClass('focus');
	  }

	  /**
	   * Handles mouse clicks outside of This. If they clicked anything that is not within this MultipleSelect,
	   * it closes the search results.
	   *
	   * @param {Event} event The event.
	   * @returns {boolean} Always true so the event is bubbled.
	   * @private
	   */
	  _handleGlobalClickEvent(event) {
	    const target = new PrimeElement(event.target);
	    if (this.displayContainer.domElement !== target.domElement && !target.isChildOf(this.displayContainer)) {
	      this.searcher.closeSearchResults();
	    }
	  }

	  /**
	   * Handles all key up events sent to the display container.
	   *
	   * @param {Event} event The browser event object.
	   * @returns {boolean} True if the search display is not open, false otherwise. This will prevent the event from continuing.
	   * @private
	   */
	  _handleKeyUpEvent(event) {
	    const key = event.keyCode;
	    if (key === Events.Keys.ESCAPE) {
	      this.unhighlightOptionForUnselect();
	    }
	  }

	  /**
	   * Makes an ID for the option.
	   *
	   * @param {PrimeElement} option The option to make the ID for.
	   * @private
	   */
	  _makeOptionID(option) {
	    if (this.options.allowDuplicates === true) {
	      let d = new Date().getTime();
	      // UUID ish
	      return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
	        let r = (d + Math.random() * 16) % 16 | 0;
	        d = Math.floor(d / 16);
	        return (c === 'x' ? r : (r & 0x3 | 0x8)).toString(16);
	      });
	    }
	    return this.element.getId() + '-option-' + option.getValue().replace(' ', '-');
	  }

	  /**
	   * Redraws the widget.
	   * @private
	   */
	  _redraw() {
	    // Remove the currently displayed options
	    this.optionList.getChildren().each(function(option) {
	      option.removeFromDOM();
	    });

	    // Add the input option since the select options are inserted before it
	    this.inputOption = PrimeDocument.newElement('<li/>')
	        .appendTo(this.optionList);
	    this.input = PrimeDocument.newElement('<input/>')
	        .addEventListener('click', this._handleClickEvent)
	        .addEventListener('blur', this._handleBlurEvent)
	        .addEventListener('focus', this._handleFocusEvent)
	        .setAttribute('type', 'text')
	        .appendTo(this.inputOption);
	    this.searcher = new Searcher(this.input, this.searchResults, this)
	        .withOptions(this.options)
	        .initialize();

	    // Add the selected options
	    let hasSelectedOptions = false;
	    const options = this.element.getOptions();
	    for (let i = 0; i < options.length; i++) {
	      const option = options[i];
	      if (option.isSelected()) {
	        this.selectOption(option);
	        hasSelectedOptions = true;
	      }
	    }

	    // Put the placeholder attribute in if the MultipleSelect has no selected options
	    if (!hasSelectedOptions) {
	      this.input.setAttribute('placeholder', this.options.placeholder);
	    }

	    this.searcher.resizeInput();

	    // If error class handling was enabled and the select box has the error class, add it to the display
	    if (this.options.errorClass && this.element.hasClass(this.options.errorClass)) {
	      this.displayContainer.addClass(this.options.errorClass);
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      allowDuplicates: false,
	      className: 'prime-multiple-select',
	      closeTimeout: 200,
	      customAddEnabled: true,
	      customAddLabel: 'Add Custom Value: ',
	      errorClass: null,
	      noSearchResultsLabel: 'No Matches For: ',
	      placeholder: 'Choose',
	      removeIcon: 'X',
	      searchFunction: Searcher.selectSearchFunction
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2016-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	//Some externs to make intellij linter shutup. :p
	/**
	 * @external TouchEvent
	 */

	/**
	 * @property {Array} changedTouches
	 * @name TouchEvent#changedTouches
	 */

	/**
	 * @class Touchable
	 */
	class Touchable {
	  /**
	   * Constructs a new Touchable object for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the Touchable widget.
	   * @param {Function} [eventPropagationHandler] A Function that handles how the event is handled for the touchstart,
	   * touchend, touchmove, and touchcancel events. This Function takes the event object as its only parameter.
	   * @constructor
	   */
	  constructor(element, eventPropagationHandler) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this.eventPropagationHandler = eventPropagationHandler;
	  }

	  /**
	   * Destroys the Touchable Widget
	   */
	  destroy() {
	    this.element
	        .removeEventListener('touchstart', this._handleTouchStart)
	        .removeEventListener('touchmove', this._handleTouchMove)
	        .removeEventListener('touchcancel', this._handleTouchCancel)
	        .removeEventListener('touchend', this._handleTouchEnd)
	        .removeEventListenersByPattern(/Prime\.Widgets\.Touchable:.+/);
	  }

	  /**
	   * Initializes the widget by attaching all of the event listeners to the element.
	   *
	   * @returns {Touchable} This.
	   */
	  initialize() {
	    this.element
	        .addEventListener('touchstart', this._handleTouchStart)
	        .addEventListener('touchmove', this._handleTouchMove)
	        .addEventListener('touchcancel', this._handleTouchCancel)
	        .addEventListener('touchend', this._handleTouchEnd);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a long press is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withLongPressHandler(handler) {
	    this.element.addEventListener('Touchable:longPress', handler);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a move event is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withMoveHandler(handler) {
	    this.element.addEventListener('Touchable:move', handler);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a long press is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withSwipeDownHandler(handler) {
	    this.element.addEventListener('Touchable:swipeDown', handler);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a swipe left event is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withSwipeLeftHandler(handler) {
	    this.element.addEventListener('Touchable:swipeLeft', handler);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a swipe right event is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withSwipeRightHandler(handler) {
	    this.element.addEventListener('Touchable:swipeRight', handler);
	    return this;
	  }

	  /**
	   * Provide a handler that will be called when a swipe up event is detected.
	   *
	   * @param {Function} handler The event handler.
	   * @returns {Touchable} This
	   */
	  withSwipeUpHandler(handler) {
	    this.element.addEventListener('Touchable:swipeUp', handler);
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Collects all of the touch data at the end of the touch and calculates the distances and times.
	   *
	   * @param {TouchEvent} event The TouchEvent.
	   * @private
	   */
	  _collectTouchData(event) {
	    const touchPoints = event.changedTouches.length;
	    if (touchPoints > 1) {
	      return;
	    }

	    const touch = event.changedTouches[0];
	    this.elapsedTime = new Date().getTime() - this.touchStarted;
	    this.touchEndX = touch.pageX;
	    this.touchEndY = touch.pageY;
	    this.touchX = this.touchStartX - this.touchEndX;
	    this.touchY = this.touchStartY - this.touchEndY;
	  }

	  /**
	   * Called when all processing is finished and the handlers are called based on direction and time of the touches.
	   *
	   * @private
	   */
	  _finished() {
	    // Make sure this was a swipe
	    const event = {
	      elapsedTime: this.elapsedTime,
	      touchStartX: this.touchStartX,
	      touchStartY: this.touchStartY,
	      touchEndX: this.touchEndX,
	      touchEndY: this.touchEndY,
	      touchX: this.touchX,
	      touchY: this.touchY,
	      element: this.element,
	      target: this.element.domElement
	    };
	    event.swipe = Math.abs(event.touchX) > 50 || Math.abs(event.touchY) > 50;
	    event.swipeX = event.swipe && Math.abs(event.touchX) > Math.abs(event.touchY);
	    event.swipeY = event.swipe && !event.swipeX;
	    event.longPress = !event.swipe && event.elapsedTime > 500;

	    if (event.longPress) {
	      this.element.fireCustomEvent('Touchable:longPress', event);
	    } else if (event.swipeX && event.touchX > 0) {
	      this.element.fireCustomEvent('Touchable:swipeLeft', event);
	    } else if (event.swipeX) {
	      this.element.fireCustomEvent('Touchable:swipeRight', event);
	    } else if (event.swipeY && event.touchY > 0) {
	      this.element.fireCustomEvent('Touchable:swipeUp', event);
	    } else if (event.swipeY) {
	      this.element.fireCustomEvent('Touchable:swipeDown', event);
	    }
	  }

	  /**
	   * Handle the touch cancel event.
	   *
	   * @param {TouchEvent} event The touch event.
	   * @private
	   */
	  _handleTouchCancel(event) {
	    this._collectTouchData(event);
	    this._finished();
	    if (Utils.isDefined(this.eventPropagationHandler)) {
	      this.eventPropagationHandler(event);
	    }
	  }

	  /**
	   * Handle the touch end event.
	   *
	   * @param {TouchEvent} event The touch event.
	   * @private
	   */
	  _handleTouchEnd(event) {
	    this._collectTouchData(event);
	    this._finished();
	    if (Utils.isDefined(this.eventPropagationHandler)) {
	      this.eventPropagationHandler(event);
	    }
	  }

	  /**
	   * Handle the touch move event.
	   *
	   * @param {TouchEvent} event The touch event.
	   * @private
	   */
	  _handleTouchMove(event) {
	    this.element.fireEvent('Touchable:move', event);
	    if (Utils.isDefined(this.eventPropagationHandler)) {
	      this.eventPropagationHandler(event);
	    }
	  }

	  /**
	   * Handle the touch start event.
	   *
	   * @param {TouchEvent} event The touch event.
	   * @private
	   */
	  _handleTouchStart(event) {
	    const touchPoints = event.changedTouches.length;
	    if (touchPoints > 1) {
	      if (Utils.isDefined(this.eventPropagationHandler)) {
	        this.eventPropagationHandler(event);
	      }

	      return;
	    }

	    const touch = event.changedTouches[0];
	    this.touchStarted = new Date().getTime();
	    this.touchStartX = touch.pageX;
	    this.touchStartY = touch.pageY;
	    if (Utils.isDefined(this.eventPropagationHandler)) {
	      this.eventPropagationHandler(event);
	    }
	  }
	}

	/*
	 * Copyright (c) 2015-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class SideMenu {
	  /**
	   * Constructs the side menu.
	   *
	   * @param {PrimeElement|Element} button The button element that is used to open the side menu.
	   * @param {PrimeElement|Element} sideMenuElement The side menu element that will be "hidden" and "show".
	   * @constructor
	   */
	  constructor(button, sideMenuElement) {
	    Utils.bindAll(this);
	    this.sideMenu = PrimeElement.wrap(sideMenuElement);
	    this.button = PrimeElement.wrap(button);
	    this._setInitialOptions();
	  }

	  /**
	   * Closes the side menu.
	   *
	   * @returns {SideMenu} This.
	   */
	  close() {
	    if (!PrimeDocument.bodyElement.hasClass(this.options.closedClass)) {
	      PrimeDocument.bodyElement.addClass(this.options.closedClass);
	    }

	    if (PrimeDocument.bodyElement.hasClass(this.options.openClass)) {
	      PrimeDocument.bodyElement.removeClass(this.options.openClass);
	    }

	    return this;
	  }

	  /**
	   * Initializes the widget by attaching the event listener to the menu button.
	   *
	   * @returns {SideMenu}
	   */
	  initialize() {
	    this.button.addEventListener('click', this._handleClickEvent);
	    return this;
	  }

	  /**
	   * @returns {boolean} True if the side menu is currently open.
	   */
	  isOpen() {
	    return this.sideMenu.getLeft() >= 0;
	    // return PrimeDocument.bodyElement.hasClass('prime-side-menu-open') || !PrimeDocument.bodyElement.hasClass('prime-side-menu-closed');
	  }

	  /**
	   * Opens the mobile nav.
	   * @returns {SideMenu} This.
	   */
	  open() {
	    if (PrimeDocument.bodyElement.hasClass(this.options.closedClass)) {
	      PrimeDocument.bodyElement.removeClass(this.options.closedClass);
	    }

	    if (!PrimeDocument.bodyElement.hasClass(this.options.openClass)) {
	      PrimeDocument.bodyElement.addClass(this.options.openClass);
	    }

	    this.touchable = new Touchable(PrimeDocument.bodyElement).withSwipeLeftHandler(this._handleSwipeLeft);
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {SideMenu} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the click event on the side menu button and calls either the open or close function.
	   *
	   * @private
	   */
	  _handleClickEvent(event) {
	    Utils.stopEvent(event);
	    if (this.isOpen()) {
	      this.close();
	    } else {
	      this.open();
	    }
	  }

	  /**
	   * Handles the synthetic swipe left event that Prime.js provides.
	   *
	   * @private
	   */
	  _handleSwipeLeft() {
	    if (this.isOpen()) {
	      this.close();
	    }

	    if (Utils.isDefined(this.touchable)) {
	      this.touchable.destroy();
	      this.touchable = null;
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      closedClass: 'prime-side-menu-closed',
	      openClass: 'prime-side-menu-open'
	    };

	    const userOptions = Utils.dataSetToOptions(this.sideMenu);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const PrimeStorage = {
	  /**
	   * True if local storage is supported.
	   * @type {boolean} true if local storage is supported. Local in this case being used to indicate either type 'local' or 'session'.
	   */
	  supported: typeof(Storage) !== 'undefined',

	  /**
	   * Set an object into session storage.
	   * @param key {string} the key to store the object.
	   * @param object {object} the object to store.
	   */
	  setSessionObject: function(key, object) {
	    PrimeStorage._setObject(sessionStorage, key, object);
	  },

	  /**
	   * Retrieve an object from session storage.
	   * @param key {string} the key that was used to store the object.
	   * @return {object} the stored object or null if it does not exist or local storage is not supported.
	   */
	  getSessionObject: function(key) {
	    return PrimeStorage._getObject(sessionStorage, key);
	  },

	  /**
	   * Set an object into local storage storage.
	   * @param key {string} the key to store the object.
	   * @param object {object} the object to store.
	   */
	  setLocalObject: function(key, object) {
	    PrimeStorage._setObject(localStorage, key, object);
	  },

	  /**
	   * Retrieve an object from local storage.
	   * @param key {string} the key that was used to store the object.
	   * @return {object} the stored object or null if it does not exist or local storage is not supported.
	   */
	  getLocalObject: function(key) {
	    return PrimeStorage._getObject(localStorage, key);
	  },

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  _getObject: function(storage, key) {
	    if (PrimeStorage.supported) {
	      const item = storage.getItem(key);
	      if (item !== null) {
	        return JSON.parse(item);
	      }
	    }

	    return null;
	  },

	  _setObject: function(storage, key, object) {
	    if (PrimeStorage.supported) {
	      storage.setItem(key, JSON.stringify(object));
	    }
	  }
	};

	/*
	 * Copyright (c) 2014-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class SplitButton {
	  /**
	   * Constructs a new SplitButton object for the given element.
	   *
	   * The markup must be a specific structure in order for the SplitButton to remember the last action the user took
	   * as well as show and hide the menu of items.
	   *
	   * Here's the format for the elements.
	   *
	   * <pre>
	   *   &lt;div class="split-button"&gt;
	   *     &lt;a href="#"&gt;Loading... &lt;/a&gt;
	   *     &lt;button type="button"&gt;&lt;/button&gt;
	   *     &lt;div type="menu"&gt;
	   *       &lt;a class="item" href="/admin/foo/add/"&gt;Add&lt;/a&gt;
	   *       &lt;a class="item" href="/admin/foo/delete/"&gt;Delete&lt;/a&gt;
	   *     &lt;/div&gt;
	   *   &lt;/ul&gt;
	   * </pre>
	   *
	   * Also, it is important to understand how to attach event listeners to the SplitButton. You cannot attach them to
	   * the individual menu items since those will never be clicked. Instead, you must attach them to the div at the top
	   * and handle all the events from inside. This is due to how the split button changes the main button based on the
	   * action the user took last.
	   *
	   * @param element {PrimeElement|Element|EventTarget} The element to transform into a split button.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this.currentAction = this.element.queryFirst('a');
	    if (this.currentAction === null) {
	      throw new TypeError('The SplitButton element must contain an <a> that is the currently selected action.');
	    }
	    this.loading = this.currentAction;

	    this.button = this.element.queryFirst('button');
	    if (this.button === null) {
	      throw new TypeError('The SplitButton element must contain a <button> that is used to open it.');
	    }

	    this.menu = this.element.queryFirst('.menu');
	    if (this.menu === null) {
	      throw new TypeError('The SplitButton element must contain a <div class="menu"> that contains the menu items.');
	    }

	    this.items = this.menu.query('.item');
	    if (this.items.length === 0) {
	      throw new TypeError('The SplitButton element must contain at least one item.');
	    }

	    this.actionIndex = null;
	    this.options = {};
	    this._setInitialOptions();
	  }

	  /**
	   * Removes the event listeners.
	   */
	  destroy() {
	    this.button.removeEventListener('click', this._handleButtonClick);
	  }

	  /**
	   * Initializes the SplitButton by setting up all the event listeners and managing the default action or the last action
	   * that the user took.
	   */
	  initialize() {
	    this.button.addEventListener('click', this._handleButtonClick);
	    this.items.each(item => item.addEventListener('click', this._handleItemClick));
	    this.redraw();
	    return this;
	  }

	  redraw() {
	    // Load the last action (if any)
	    let actionId = null;
	    if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	      actionId = PrimeStorage.getSessionObject(this.options.localStorageKey);
	      if (actionId !== null) {
	        this.selectAction(actionId);
	      }
	    }

	    // If no action is selected from local storage, select the default
	    if (actionId === null) {
	      this.selectDefaultAction();
	    }

	    return this;
	  }

	  selectAction(actionId) {
	    const item = PrimeDocument.queryById(actionId);
	    if (item !== null) {
	      let classes = '';
	      if (this.loading !== null) {
	        classes = this.loading.getAttribute('class');
	        this.loading.removeFromDOM();
	        this.loading = null;
	      } else {
	        classes = this.currentAction.getAttribute('class');
	        this.currentAction.setAttribute('class', this.currentAction.getAttribute('data-original-class'));
	        this.currentAction.removeFromDOM();
	        this.currentAction.prependTo(this.menu);
	      }

	      item.setAttribute('data-original-class', item.getAttribute('class'));
	      item.setAttribute('class', classes);
	      item.removeFromDOM();
	      item.prependTo(this.element);
	      this.currentAction = item;
	    } else {
	      // This shouldn't infinitely recurse because that method always finds something
	      this.selectDefaultAction();
	    }

	    // this.currentAction.setAttribute('href', item.getAttribute('href'));
	    // this.currentAction.setHTML(item.getHTML());
	    return this;
	  }

	  selectDefaultAction() {
	    // Find the default
	    let actionId = null;
	    for (let i = 0; i < this.items.length; i++) {
	      if (this.items[i].hasClass('default')) {
	        actionId = this.items[i].getId();
	        break;
	      }
	    }

	    // Fall back if it wasn't found
	    if (actionId === null) {
	      actionId = this.items[0].getId();
	    }

	    // Select the default (or first item)
	    if (actionId !== null) {
	      this._saveLastAction(actionId);
	      this.selectAction(actionId);
	    }

	    return this;
	  }

	  withLocalStorageKey(localStorageKey) {
	    this.options.localStorageKey = localStorageKey;
	    return this;
	  }

	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }

	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the button click event.
	   * @private
	   */
	  _handleButtonClick(event) {
	    Utils.stopEvent(event);
	    if (this.menu.isVisible()) {
	      this.menu.hide();
	    } else {
	      this.menu.show();
	    }
	  }

	  /**
	   * Handles the item click event to update the last action in local storage (if enabled).
	   * @private
	   */
	  _handleItemClick(event) {
	    for (let i = 0; i < this.items.length; i++) {
	      if (this.items[i].domElement === event.target) {
	        let actionId = this.items[i].getId();
	        this._saveLastAction(actionId);
	        this.selectAction(actionId);
	        this.menu.hide();
	        break;
	      }
	    }
	  }

	  _saveLastAction(actionId) {
	    if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	      PrimeStorage.setSessionObject(this.options.localStorageKey, actionId);
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      localStorageKey: null
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class Table {
	  /**
	   * Constructs a new Table object for the given table element.
	   *
	   * @param {PrimeElement|Element} element The table element.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.column = null;
	    this.columnIndex = 0;
	    this.sortAscending = true;

	    this.element = PrimeElement.wrap(element);
	    this.thead = this.element.queryFirst('thead');
	    this.tbody = this.element.queryFirst('tbody');

	    if (!this.element.is('table')) {
	      throw new TypeError('The element you passed in is not a table element.');
	    }

	    this._setInitialOptions();
	  }

	  /**
	   * Initializes the table widget.
	   *
	   * @returns {Table} This.
	   */
	  initialize() {
	    // Sortable by default unless it is disabled
	    if (this.element.getDataAttribute('sortable') !== 'false') {
	      this._initializeSort();
	    }

	    // Initialize the checkbox handling
	    this.selectAll = this.element.queryFirst('thead > tr > th input[type="checkbox"]');
	    if (this.selectAll !== null) {
	      this.selectAll.addEventListener('change', this._handleSelectAllChange);
	    }

	    this.element.query('tbody > tr > td input[type="checkbox"]').addEventListener('click', this._handleCheckboxEvent);
	    this.checkedCount = 0;
	    this.numberofCheckboxes = this.element.query('tbody td input[type="checkbox"]').length;

	    return this;
	  }

	  /**
	   * Sort the table.
	   */
	  sort() {
	    this._clearSortIndicators();

	    if (this.column.hasClass('sort-up')) {
	      this.column.removeClass('sort-up').addClass('sort-down');
	      this.sortAscending = false;
	    } else if (this.column.hasClass('sort-down')) {
	      this.column.removeClass('sort-down').addClass('sort-up');
	      this.sortAscending = true;
	    } else {
	      this.column.addClass('sort-up');
	      this.sortAscending = true;
	    }

	    // Collect the values to sort
	    const rows = [];
	    this.tbody.query('tr').each(function(element) {
	      rows.push(element);
	    });

	    rows.sort(this._comparator);
	    let i = 0;
	    const length = rows.length;
	    if (this.sortAscending) {
	      for (i = 0; i <  length; i++) {
	        this.tbody.appendElement(rows[i]);
	      }
	    } else {
	      for (i = length; i > 0; i--) {
	        this.tbody.appendElement(rows[i - 1]);
	      }
	    }

	    // Save current sorted column state in local storage.
	    if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	      const data = {
	        columnIndex: this.columnIndex,
	        sortAscending: this.sortAscending
	      };
	      PrimeStorage.setSessionObject(this.options.localStorageKey, data);
	    }
	  }

	  /**
	   * Sets a callback on a checkbox event.
	   *
	   * @param {function} callback The callback function
	   * @returns {Table} This.
	   */
	  withCheckEventCallback(callback) {
	    this.options.checkEventCallback = callback;
	    return this;
	  }

	  /**
	   * Enables local storage of the sorted column. This key is required to enable local storage of the sorted column.
	   *
	   * @param {String} key The local storage key.
	   * @returns {Table} This.
	   */
	  withLocalStorageKey(key) {
	    this.options.localStorageKey = key;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {Table} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Remove the ascending and descending sort classes on every column except the current column being sorted.
	   * @private
	   */
	  _clearSortIndicators() {
	    this.thead.query('th').each((function(element) {
	      if (element.domElement !== this.column.domElement) {
	        element.removeClass('sort-up sort-down');
	      }
	    }).bind(this));
	  }

	  /**
	   * Sort function to be used by the .sort() method.
	   * @param a The first row to compare
	   * @param b The second row to compare
	   * @returns {number}
	   * @private
	   */
	  _comparator(a, b) {
	    const sortType = this.thead.query('th')[this.columnIndex].getDataAttribute('sortType') || 'string';
	    if (sortType !== 'string' && sortType !== 'number') {
	      throw new Error('Unsupported sort type. [string] or [number] are the two supported sort types.');
	    }

	    const cell1 = a.query('td')[this.columnIndex];
	    const cell2 = b.query('td')[this.columnIndex];

	    const sortValue1 = cell1.getDataAttribute('sortValue');
	    const sortValue2 = cell2.getDataAttribute('sortValue');

	    // Prefer the data-sort-value if provided
	    let value1 = sortValue1 || cell1.getTextContent().toLowerCase();
	    let value2 = sortValue2 || cell2.getTextContent().toLowerCase();

	    if (sortType === 'string') {
	      return value1.localeCompare(value2);
	    } else {
	      value1 = this._toNumber(value1);
	      value2 = this._toNumber(value2);

	      if (value1 < value2) {
	        return -1;
	      }

	      if (value1 > value2) {
	        return 1;
	      }

	      return 0;
	    }
	  }

	  // noinspection JSMethodCanBeStatic
	  /**
	   * Return the column index where 0 is the first column in the table.
	   * @param column {PrimeElement} the column to determine the index of
	   * @returns {number} a positive integer representing the index of the column in the table.
	   * @private
	   */
	  _getColumnIndex(column) {
	    let columnIndex = 0;
	    let current = column;
	    let previous = column;
	    while (previous !== null) {
	      previous = current.getPreviousSibling();
	      current = previous;
	      columnIndex++;
	    }

	    return columnIndex - 1;
	  }

	  _handleCheckboxEvent(event) {
	    const target = new PrimeElement(event.currentTarget);
	    const currentCheckedCount = this.checkedCount;
	    this.checkedCount = this.checkedCount + (target.isChecked() ? 1 : -1);

	    if (this.selectAll !== null) {
	      if (currentCheckedCount === this.numberofCheckboxes && this.numberofCheckboxes !== this.checkedCount) {
	        this.selectAll.setChecked(false);
	      } else if (currentCheckedCount !== this.numberofCheckboxes && this.numberofCheckboxes === this.checkedCount) {
	        this.selectAll.setChecked(true);
	      }
	    }

	    if (this.options.checkEventCallback !== null) {
	      this.options.checkEventCallback({
	        checkedCount: this.checkedCount
	      });
	    }
	  }

	  _handleSelectAllChange() {
	    if (this.selectAll.isChecked()) {
	      this.element.query('tbody tr > td input[type="checkbox"]').each(function(e) {
	        if (!e.isChecked()) {
	          e.setChecked(true);
	          this.checkedCount++;
	        }
	      }.bind(this));
	    } else {
	      this.element.query('tbody tr > td input[type="checkbox"]').each(function(e) {
	        if (e.isChecked()) {
	          e.setChecked(false);
	          this.checkedCount--;
	        }
	      }.bind(this));
	    }

	    if (this.options.checkEventCallback !== null) {
	      this.options.checkEventCallback({
	        checkedCount: this.checkedCount
	      });
	    }
	  }

	  /**
	   * Handle the click event on the sortable column.
	   * @param event {MouseEvent} the click event
	   * @private
	   */
	  _handleSortableColumnClick(event) {
	    Utils.stopEvent(event);
	    const target = new PrimeElement(event.currentTarget);
	    this.column = target;
	    this.columnIndex = this._getColumnIndex(target);

	    this.sort();
	  }

	  /**
	   * Add the click event listener to the column unless it matches the ignore selector.
	   * @param column {PrimeElement} the column element to initialize.
	   * @private
	   */
	  _initializeColumn(column) {
	    if (!column.is('[data-sortable="false"]') && column.queryFirst('input[type="checkbox"]') === null) {
	      column.addClass('sortable').addEventListener('click', this._handleSortableColumnClick);
	    }
	  }

	  _initializeSort() {
	    this.thead.query('th').each(this._initializeColumn);

	    if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	      const state = PrimeStorage.getSessionObject(this.options.localStorageKey);
	      if (state !== null) {
	        this.columnIndex = state.columnIndex;
	        this.sortAscending = state.sortAscending;

	        this.column = this.thead.query('th')[this.columnIndex];
	        if (this.sortAscending) {
	          this.column.addClass('sort-down');
	        } else {
	          this.column.addClass('sort-up');
	        }

	        this.sort();
	      }
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      localStorageKey: null,
	      checkEventCallback: null
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }

	  // noinspection JSMethodCanBeStatic
	  _toNumber(value) {
	    const number = Number(value);
	    if (isNaN(value)) {
	      console.error(new Error('Expected value [' + value + '] to be a number.'));
	      return value;
	    }
	    return number;
	  }
	}

	/*
	 * Copyright (c) 2015-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class Tabs {
	  /**
	   * Constructs a new Tabs object for the given ul element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The ul element to build the tab widget from
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    if (this.element.getTagName().toLowerCase() === 'ul') {
	      this.tabsContainer = this.element;
	    } else {
	      this.tabsContainer = this.element.queryFirst('ul');
	    }

	    if (this.tabsContainer === null) {
	      throw new TypeError('Tabs requires a ul element. The passed element does not contain a ul element');
	    }

	    this._setInitialOptions();
	    this.tabContents = {};
	    this.tabs = {};
	    this.tabArray = [];
	    this.selectedTab = null;
	  }

	  /**
	   * Destroys the Tabs widget
	   */
	  destroy() {
	    this.tabsContainer.query('a').each(function(a) {
	      a.removeEventListener('click', this._handleClick);
	    }.bind(this));

	    for (let i = 0; i < this.tabs.length; i++) {
	      this.tabs[i].removeClass(this.options.tabContentClass);
	    }
	  }

	  /**
	   * Hides the tab for the given Id.
	   *
	   * @param id The Id of the tab to hide.
	   */
	  hideTab(id) {
	    const tab = this.tabs[id];
	    tab.hide();
	    this.redraw();
	  }

	  /**
	   * Initializes the Tabs widget. Call this after you have set all the initial options.
	   *
	   * @returns {Tabs} This.
	   */
	  initialize() {
	    this.tabsContainer.query('li:not(.disabled)').each(function(tab) {
	      const a = tab.queryFirst('a').addEventListener('click', this._handleClick);
	      const dataSet = tab.getDataSet();

	      const href = a.getAttribute('href');
	      const isAnchor = href.charAt(0) === '#';
	      if (isAnchor) {
	        dataSet.tabId = href.substring(1);
	        dataSet.tabUrl = '';
	      } else {
	        dataSet.tabId = href;
	        dataSet.tabUrl = href;
	      }

	      this.tabs[dataSet.tabId] = tab;
	      this.tabArray.push(tab);

	      let content = PrimeDocument.queryById(dataSet.tabId);
	      if (content === null && isAnchor) {
	        throw new Error('A div is required with the following ID [' + dataSet.tabId + ']');
	      } else if (content === null) {
	        content = PrimeDocument.newElement('<div>').insertAfter(this.element).setAttribute('id', href);
	      }

	      content.hide();

	      content.addClass(this.options.tabContentClass);
	      this.tabContents[dataSet.tabId] = content;
	    }.bind(this));

	    if (this.options.deepLinkingEnabled) {
	      const tabId = window.location.hash.replace(/^#/, '');
	      if (Utils.isDefined(tabId) && Utils.isDefined(this.tabs[tabId])) {
	        this.selectTab(tabId);
	      }
	    }

	    this.redraw();
	    return this;
	  }

	  /**
	   * Re-applies the first-child, last-child, and active classes based on the current state of the tabs. If there
	   * is no tab that is active, this also selects the first tab that is visible.
	   */
	  redraw() {
	    let firstVisible = null;
	    let lastVisible = null;
	    let selectNew = false;
	    let noneActive = true;
	    for (let i = 0; i < this.tabArray.length; i++) {
	      if (this.tabArray[i].isVisible()) {
	        if (firstVisible === null) {
	          firstVisible = this.tabArray[i];
	        }

	        lastVisible = this.tabArray[i];

	        if (this.tabArray[i].hasClass('selected')) {
	          noneActive = false;
	        }
	      } else if (this.tabArray[i].hasClass('selected')) {
	        selectNew = true;
	      }

	      this.tabArray[i].removeClass('first-visible-tab');
	      this.tabArray[i].removeClass('last-visible-tab');
	    }

	    firstVisible.addClass('first-visible-tab');
	    lastVisible.addClass('last-visible-tab');

	    let tabId = null;
	    if (selectNew || noneActive) {
	      if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	        const state = PrimeStorage.getSessionObject(this.options.localStorageKey);
	        if (state !== null) {
	          tabId = state.tabId;
	        }
	      }

	      // If no tabId was found or the tab is not currently visible, select the first visible
	      if (tabId === null || !this.tabs[tabId] || !this.tabs[tabId].isVisible()) {
	        tabId = firstVisible.getDataSet().tabId;
	      }

	      this.selectTab(tabId);
	    }

	    // If error class handling was enabled, add the error class to the tab and set focus
	    if (this.options.errorClass) {
	      for (tabId in this.tabContents) {
	        if (this.tabContents.hasOwnProperty(tabId)) {
	          const errorElement = this.tabContents[tabId].queryFirst('.' + this.options.errorClass);
	          if (errorElement !== null) {
	            this.tabs[tabId].queryFirst('a').addClass(this.options.errorClass);
	            this.selectTab(tabId);
	          }
	        }
	      }
	    }
	  }

	  /**
	   * Select the active tab. Sets the active class on the li and shows only the corresponding tab content.
	   *
	   * @param id The Id of the tab to select.
	   */
	  selectTab(id) {
	    if (this.selectedTab !== null && this.selectedTab.getDataSet().tabId === id) {
	      return;
	    }

	    for (const tabId in this.tabs) {
	      if (this.tabs.hasOwnProperty(tabId)) {
	        this.tabs[tabId].removeClass('selected');
	      }
	    }

	    this.tabs[id].addClass('selected');
	    this.selectedTab = this.tabs[id];
	    for (const tabId in this.tabContents) {
	      if (this.tabContents.hasOwnProperty(tabId) && tabId === id) {
	        this.tabContents[tabId].show('block');
	        if (this.options.selectCallback) {
	          this.options.selectCallback(this.tabs[tabId], this.tabContents[tabId]);
	        }
	      } else {
	        this.tabContents[tabId].hide();
	      }
	    }

	    // Save current selected tab state in local storage. The JSON object isn't necessary at the moment,
	    // but we can tack on other properties as needed for additional state in the future.
	    if (PrimeStorage.supported && this.options.localStorageKey !== null) {
	      const data = {
	        tabId: id
	      };
	      PrimeStorage.setSessionObject(this.options.localStorageKey, data);
	    }

	    const ajaxURL = this.selectedTab.getDataSet().tabUrl;
	    if (ajaxURL !== '') {
	      this.selectedTab.addClass('loading');
	      this.tabContents[id].setHTML('');
	      this.tabContents[id].addClass('loading');
	      new PrimeRequest(ajaxURL, 'GET')
	          .withSuccessHandler(this._handleAJAXResponse)
	          .withErrorHandler(this._handleAJAXResponse)
	          .go();
	    }
	  }

	  /**
	   * Shows the tab for the given Id.
	   *
	   * @param {String} id The Id of the tab to hide.
	   */
	  showTab(id) {
	    this.tabs[id].show();
	    this.redraw();
	  }

	  /**
	   * Adds a callback for AJAX calls. This is invoked after the AJAX load completes and the HTML is inserted into the
	   * DOM. The function is passed the container for the tab that was selected.
	   *
	   * @param {Function} callback The callback function.
	   * @returns {Tabs} This Tabs.
	   */
	  withAJAXCallback(callback) {
	    this.options.ajaxCallback = callback;
	    return this;
	  }

	  /**
	   * Disable the default behavior of allowing a deep link provided on the URL to set the default tab during render.
	   *
	   * @returns {Tabs} This Tabs.
	   */
	  withDeepLinkingDisabled() {
	    this.options.deepLinkingEnabled = false;
	    return this;
	  }

	  /**
	   * Enable error class handling. When this option is used, if the specified error class is found on any element
	   * in the tab content the same error class will be added to the tab to identify the tab contains errors.
	   *
	   * @returns {Tabs} This Tabs.
	   */
	  withErrorClassHandling(errorClass) {
	    this.options.errorClass = errorClass;
	    return this;
	  }

	  /**
	   * Enables local storage of the currently selected tab. If the user navigates away from the page and back, the same
	   * tab will be selected. This key is how the selected tab is stored in local storage and by setting a key you also
	   * enable this feature.
	   *
	   * @param {?String} key The local storage key.
	   * @returns {Tabs} This Tabs.
	   */
	  withLocalStorageKey(key) {
	    this.options.localStorageKey = key;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {Tabs} This Tabs.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /**
	   * Specifies a callback function that is called whenever tabs are changed.
	   *
	   * @param {?Function} callback The callback function.
	   * @returns {Tabs} This Tabs.
	   */
	  withSelectCallback(callback) {
	    this.options.selectCallback = callback;
	    return this;
	  }

	  /**
	   * Sets the class name for the tab content elements.
	   *
	   * @param className {String} The class name.
	   * @returns {Tabs}
	   */
	  withTabContentClass(className) {
	    this.options.tabContentClass = className;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the AJAX response.
	   *
	   * @param {XMLHttpRequest} xhr The AJAX response.
	   * @private
	   */
	  _handleAJAXResponse(xhr) {
	    this.selectedTab.removeClass('loading');
	    const container = this.tabContents[this.selectedTab.getDataSet().tabId];
	    container.removeClass('loading');
	    container.setHTML(xhr.responseText);

	    if (this.options.ajaxCallback !== null) {
	      this.options.ajaxCallback(container);
	    }
	  }

	  /**
	   * Handle the tab click by showing the corresponding panel and hiding the others.
	   *
	   * @param {MouseEvent} event The click event on the anchor tag.
	   * @private
	   */
	  _handleClick(event) {
	    Utils.stopEvent(event);
	    const a = new PrimeElement(event.currentTarget);
	    if (!a.hasClass('disabled')) {
	      const href = a.getAttribute('href');
	      if (href.charAt(0) === '#') {
	        this.selectTab(href.substring(1));
	      } else {
	        this.selectTab(href);
	      }
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      ajaxCallback: null,
	      errorClass: null,
	      deepLinkingEnabled: true,
	      localStorageKey: null,
	      selectCallback: null,
	      tabContentClass: 'prime-tab-content'
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	const open = [];

	class Tooltip {
	  /**
	   * Constructs a new Tooltip object for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the Tooltip widget.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = PrimeElement.wrap(element);
	    this._setInitialOptions();
	  }

	  static get open() {
	    return open;
	  }

	  /**
	   * Hides the tooltip.
	   *
	   * @returns {Tooltip} This.
	   */
	  hide() {
	    if (Tooltip.open.length > 0) {
	      Tooltip.open.forEach(function(t) {
	        t.removeFromDOM();
	      });
	    }
	    return this;
	  }

	  /**
	   * Initializes the widget by attaching event listeners to the element.
	   *
	   * @returns {Tooltip} This.
	   */
	  initialize() {
	    this.element.addEventListener('mouseenter', this._handleMouseEnter).addEventListener('mouseleave', this._handleMouseLeave);
	    PrimeDocument.addEventListener('scroll', this._handleMouseLeave);
	    return this;
	  }

	  /**
	   * Shows the tooltip.
	   *
	   * @returns {Tooltip} This.
	   */
	  show() {
	    const text = this.element.getDataSet()[this.options.dataName];
	    const zIndex = this.element.getRelativeZIndex();
	    let classNames = this.options.className + ' ' + this.element.getTagName().toLocaleLowerCase();
	    if (this.options.additionalClasses !== null) {
	      classNames = classNames + ' ' + this.options.additionalClasses;
	    }
	    const tooltip = PrimeDocument.newElement('<span>')
	        .appendTo(PrimeDocument.bodyElement)
	        .addClass(classNames)
	        .setHTML(text)
	        .setStyle('zIndex', zIndex + 10);

	    const left = this.element.getLeft();
	    const top = this.element.getTop();
	    const width = this.element.getWidth();
	    const tooltipWidth = tooltip.getWidth();
	    const tooltipHeight = tooltip.getHeight();

	    tooltip.setLeft(left - (tooltipWidth / 2) + (width / 2));
	    tooltip.setTop(top - tooltipHeight - 8);

	    // If the tooltip is too close to the top of the screen invert it and move it under the element
	    if ((top - tooltipHeight - 8) < 0) {
	      tooltip.setTop(top + this.element.getHeight() + 8).addClass('inverted');
	    }

	    Tooltip.open.push(tooltip);
	    return this;
	  }

	  /**
	   * Sets the class name to use when creating the tooltip.
	   *
	   * @param additionalClasses {?String} The class name or spaces separated names.
	   * @returns {Tooltip} This.
	   */
	  withAdditionalClassNames(additionalClasses) {
	    this.options.additionalClasses = additionalClasses;
	    return this;
	  }

	  /**
	   * Sets the class name to use when creating the tooltip.
	   *
	   * @param className {String} The class name.
	   * @returns {Tooltip} This.
	   */
	  withClassName(className) {
	    this.options.className = className;
	    return this;
	  }

	  /**
	   * Set data-set name to pull the tooltip text from.
	   *
	   * @param {string} name The data-set name.
	   * @returns {Tooltip} This.
	   */
	  withDataName(name) {
	    this.options.dataName = name;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {Tooltip} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the mouse enter event to show the tooltip.
	   *
	   * @private
	   */
	  _handleMouseEnter() {
	    this.show();
	  }

	  /**
	   * Handles the mouse exit event to hide the tooltip.
	   *
	   * @private
	   */
	  _handleMouseLeave() {
	    this.hide();
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      additionalClasses: null,
	      className: 'prime-tooltip',
	      dataName: 'tooltip'
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	PrimeDocument.onReady(function() {
	  // Fix browser issues with tooltips sticking around on back-button navigation
	  PrimeWindow.addEventListener('beforeunload', function() {
	    if (Tooltip.open.length > 0) {
	      Tooltip.open.forEach(function(t) {
	        t.removeFromDOM();
	      });
	    }
	  });
	});

	/*
	 * Copyright (c) 2016-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class TreeView {
	  /**
	   * Constructs a new TreeView object for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element for the TreeView widget.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);
	    this.element = PrimeElement.wrap(element);
	    this._setInitialOptions();
	  }

	  /**
	   * Initializes the element by traverse its children to find all of the anchor tags with the folder-toggle class (or
	   * whatever you set the class to).
	   *
	   * @returns {TreeView} This.
	   */
	  initialize() {
	    this.element.query('a.' + this.options.folderToggleClassName).each(function(e) {
	      e.addEventListener('click', this._handleClick);
	    }.bind(this));
	    return this;
	  }

	  /**
	   * Sets the folder toggle class name.
	   *
	   * @param className {String} The class name.
	   * @returns {TreeView} This.
	   */
	  withFolderToggleClassName(className) {
	    this.options.folderToggleClassName = className;
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  // noinspection JSMethodCanBeStatic
	  /**
	   * Handles the click event.
	   * @private
	   */
	  _handleClick(event) {
	    Utils.stopEvent(event);
	    const a = PrimeElement.wrap(event.target);
	    const li = a.getParent();
	    if (a.hasClass('open')) {
	      a.removeClass('open');
	      li.removeClass('open');
	    } else {
	      a.addClass('open');
	      li.addClass('open');
	    }
	  }

	  /**
	   * Set the initial options for this widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      folderToggleClassName: 'prime-folder-toggle'
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (const option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */




	var Widgets = Object.freeze({
		AJAXDialog: AJAXDialog,
		DateTimePicker: DateTimePicker,
		Dismissable: Dismissable,
		Draggable: Draggable,
		HTMLDialog: HTMLDialog,
		InProgress: InProgress,
		MultipleSelect: MultipleSelect,
		Overlay: Overlay,
		Searcher: Searcher,
		SideMenu: SideMenu,
		SplitButton: SplitButton,
		Table: Table,
		Tabs: Tabs,
		Tooltip: Tooltip,
		Touchable: Touchable,
		TreeView: TreeView
	});

	/*
	 * Copyright (c) 2012-2018, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class BaseTransition {
	  /**
	   * Constructs a BaseTransition for the given element.
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element the effect will be applied to.
	   * @param {number} endValue The end value for the transition.
	   * @constructor
	   */
	  constructor(element, endValue) {
	    Utils.bindAll(this);
	    this.element = PrimeElement.wrap(element);
	    this.duration = 1000;
	    this.endFunction = null;
	    this.endValue = endValue;
	    this.iterations = 20;
	  }

	  /**
	   * Sets the function that is called when the effect has completed.
	   *
	   * @param {Function} endFunction The function that is called when the effect is completed.
	   * @returns {BaseTransition} This Effect.
	   */
	  withEndFunction(endFunction) {
	    this.endFunction = endFunction;
	    return this;
	  }

	  /**
	   * Sets the duration of the fade-out effect.
	   *
	   * @param {number} duration The duration in milliseconds.
	   * @returns {BaseTransition} This Effect.
	   */
	  withDuration(duration) {
	    if (duration < 100) {
	      throw new TypeError('Duration should be greater than 100 milliseconds or it won\'t really be noticeable');
	    }

	    this.duration = duration;
	    return this;
	  }

	  /*
	   * Protected functions
	   */

	  /**
	   * Changes an integer style property of the Element iteratively over a given period of time from one value to another
	   * value.
	   *
	   * @protected
	   * @param {Function} getFunction The function on the element to call to get the current value for the transition.
	   * @param {Function} setFunction The function on the element to call to set the new value for the transition.
	   */
	  _changeNumberStyleIteratively(getFunction, setFunction) {
	    let currentValue = getFunction.call(this.element);
	    const step = Math.abs(this.endValue - currentValue) / this.iterations;

	    // Close around ourselves
	    const self = this;
	    const stepFunction = function(last) {
	      if (last) {
	        currentValue = self.endValue;
	      } else {
	        if (currentValue < self.endValue) {
	          currentValue += step;
	        } else {
	          currentValue -= step;
	        }
	      }

	      setFunction.call(self.element, currentValue);
	    };

	    Utils.callIteratively(this.duration, this.iterations, stepFunction, this._internalEndFunction);
	  }

	  /* ===================================================================================================================
	   * Private Methods
	   * ===================================================================================================================*/

	  /**
	   * Handles the call back at the end.
	   *
	   * @private
	   */
	  _internalEndFunction() {
	    this._subclassEndFunction(this);

	    if (this.endFunction) {
	      this.endFunction(this);
	    }
	  }

	  /**
	   * Virtual function stub
	   *
	   * @private
	   */
	  _subclassEndFunction() {
	  }
	}

	class Fade extends BaseTransition {
	  /**
	   * Constructs a new Fade for the given element. The fade effect uses the CSS opacity style and supports the IE alpha
	   * style. The duration defaults to 1000 milliseconds (1 second). This changes the opacity over the duration from 1.0 to
	   * 0.0. At the end, this hides the element so that it doesn't take up any space.
	   *
	   * @constructor
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element to fade out.
	   */
	  constructor(element) {
	    super(element, 0.0);
	  }

	  /**
	   * Internal call back at the end of the transition. This hides the element so it doesn't take up space.
	   *
	   * @override
	   * @private
	   */
	  _subclassEndFunction() {
	    this.element.hide();
	  }

	  /**
	   * Executes the fade effect on the element using the opacity style.
	   */
	  go() {
	    this._changeNumberStyleIteratively(this.element.getOpacity, this.element.setOpacity);
	  }
	}

	class Appear extends BaseTransition {
	  /**
	   * Constructs a new Appear for the given element. The appear effect uses the CSS opacity style and supports the IE
	   * alpha style. The duration defaults to 1000 milliseconds (1 second). This first sets the opacity to 0, then it shows
	   * the element and finally it raises the opacity.
	   *
	   * @constructor
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element to appear.
	   * @param {number} [opacity=1.0] The final opacity to reach when the effect is complete. Defaults to 1.0.
	   */
	  constructor(element, opacity) {
	    if (!Utils.isDefined(opacity)) {
	      opacity = 1.0;
	    }
	    super(element, opacity);
	  }

	  /**
	   * Executes the appear effect on the element using the opacity style.
	   */
	  go() {
	    this.element.setOpacity(0.0);
	    this.element.show();
	    this._changeNumberStyleIteratively(this.element.getOpacity, this.element.setOpacity);
	  }
	}

	class ScrollTo extends BaseTransition {
	  /**
	   * Constructs a new ScrollTo for the given element. The duration defaults to 1000 milliseconds (1 second).
	   *
	   * @param {PrimeElement|Element|EventTarget} element The Prime Element to scroll.
	   * @param {number} position The position to scroll the element to.
	   * @constructor
	   */
	  constructor(element, position) {
	    super(element, position);

	    this.axis = 'vertical';
	  }

	  /**
	   * Set the scroll axis, either 'horizontal' or 'vertical'. Default is 'vertical'.
	   *
	   * @param {string} axis The axis to scroll.
	   * @returns {ScrollTo}
	   */
	  withAxis(axis) {
	    this.axis = axis || 'vertical';
	    return this;
	  }

	  /**
	   * Executes the scroll effect on the element.
	   */
	  go() {
	    if (this.axis === 'vertical') {
	      this._changeNumberStyleIteratively(this.element.getScrollTop, this.element.scrollTo);
	    } else {
	      this._changeNumberStyleIteratively(this.element.getScrollLeft, this.element.scrollLeftTo);
	    }
	  }
	}

	class SlideOpen {
	  /**
	   * Creates a SlideOpen effect on the given element.
	   *
	   * @param {PrimeElement} element The element.
	   * @constructor
	   */
	  constructor(element) {
	    Utils.bindAll(this);

	    this.element = element;
	    if (this.isOpen()) {
	      element.domElement.primeVisibleHeight = element.getHeight();
	    } else {
	      element.setStyle('height', 'auto');
	      element.domElement.primeVisibleHeight = element.getHeight();
	      element.setStyle('height', '0');
	    }

	    this.isTransitioning = false;

	    this._setInitialOptions();
	  }

	  close() {
	    if (!this.isOpen()) {
	      return;
	    }

	    // Set a fixed height instead of auto so that the transition runs, but only if the element is "open"
	    this.element.setHeight(this.element.domElement.primeVisibleHeight);
	    this.isTransitioning = true;

	    // This timeout is needed since the height change takes time to run
	    setTimeout(function() {
	      this.element.setHeight(0);
	      this.element.removeClass('open');
	      this.isTransitioning = false;

	      if (this.options.closeCallback !== null) {
	        this.options.closeCallback();
	      }
	    }.bind(this), this.options.timeout);
	  }

	  isOpen() {
	    return this.element.getHeight() !== 0 || this.element.hasClass('open');
	  }

	  open() {
	    if (this.isOpen()) {
	      return;
	    }

	    this.element.setHeight(this.element.domElement.primeVisibleHeight);
	    this.isTransitioning = true;

	    setTimeout(function() {
	      this.element.setHeight('auto');
	      this.element.addClass('open');
	      this.isTransitioning = false;

	      if (this.options.openCallback !== null) {
	        this.options.openCallback();
	      }
	    }.bind(this), this.options.timeout);
	  }

	  toggle() {
	    if (this.isOpen()) {
	      this.close();
	    } else {
	      this.open();
	    }
	  }

	  /**
	   * Set the close callback function.
	   *
	   * @param {?Function} callback The close callback
	   * @returns {SlideOpen} This.
	   */
	  withCloseCallback(callback) {
	    this.options.closeCallback = callback;
	    return this;
	  }

	  /**
	   * Set the open callback function.
	   *
	   * @param {?Function} callback The open callback
	   * @returns {SlideOpen} This.
	   */
	  withOpenCallback(callback) {
	    this.options.openCallback = callback;
	    return this;
	  }

	  /**
	   * Set more than one option at a time by providing a map of key value pairs. This is considered an advanced
	   * method to set options on the widget. The caller needs to know what properties are valid in the options object.
	   *
	   * @param {Object} options Key value pair of configuration options.
	   * @returns {SlideOpen} This.
	   */
	  withOptions(options) {
	    if (!Utils.isDefined(options)) {
	      return this;
	    }

	    for (let option in options) {
	      if (options.hasOwnProperty(option)) {
	        this.options[option] = options[option];
	      }
	    }
	    return this;
	  }

	  /* ===================================================================================================================
	   * Private methods
	   * ===================================================================================================================*/

	  /**
	   * Set the initial options of the widget.
	   * @private
	   */
	  _setInitialOptions() {
	    // Defaults
	    this.options = {
	      openCallback: null,
	      closeCallback: null,
	      timeout: 310
	    };

	    const userOptions = Utils.dataSetToOptions(this.element);
	    for (let option in userOptions) {
	      if (userOptions.hasOwnProperty(option)) {
	        this.options[option] = userOptions[option];
	      }
	    }
	  }
	}




	var Effects = Object.freeze({
		BaseTransition: BaseTransition,
		Fade: Fade,
		Appear: Appear,
		ScrollTo: ScrollTo,
		SlideOpen: SlideOpen
	});

	/*
	 * Copyright (c) 2016-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class DataQueue {
	  /**
	   * First-In-First-Out Queue implementation modeled after java.util.Deque interface.
	   *
	   * @constructor
	   */
	  constructor() {
	    Utils.bindAll(this);
	    this._elements = {};
	    this._head = 0;
	    this._tail = 0;
	  }

	  /**
	   * Add the element to the head of the queue.
	   *
	   * @param {Object} element An object to store in the queue.
	   * @returns {DataQueue} This Element.
	   */
	  add(element) {
	    this._elements[this._head] = element;
	    this._head++;

	    return this;
	  }

	  /**
	   * Return true if the queue is empty.
	   *
	   * @returns {boolean} True if the queue is empty, false if not.
	   */
	  isEmpty() {
	    return this._head === this._tail;
	  }

	  /**
	   * Return but do not remove the tail of the queue. This is the oldest element in the queue.
	   *
	   * @returns {Object} The object at the tail of the queue, or null if empty.
	   */
	  peek() {
	    if (this.isEmpty()) {
	      return null;
	    }

	    return this._elements[this._tail];
	  }

	  /**
	   * Return and remove the tail of the queue. This is the oldest element in the queue.
	   *
	   * @returns {Object} the object at the tail of the queue, or null if the queue is empty.
	   */
	  poll() {
	    if (this.isEmpty()) {
	      return null;
	    }

	    const object = this._elements[this._tail];
	    delete this._elements[this._tail];
	    this._tail++;

	    // The cursor should not go off the end of the queue
	    if (this._cursor < this._tail) {
	      this._cursor = this._tail;
	    }

	    return object;
	  }

	  /**
	   * Return the size of the queue.
	   *
	   * @returns {Number} The size of the queue.
	   */
	  size() {
	    return this._head - this._tail;
	  }

	}

	/*
	 * Copyright (c) 2013-2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */
	class Template {
	  /**
	   * A Javascript Object that can serve to generate Prime.Document.Element from a source string and optional parameters.
	   *
	   * @constructor
	   * @param {string} template The String that defines the source of the template.
	   */
	  constructor(template) {
	    Utils.bindAll(this);
	    this.init(template);
	  }

	  init(template) {
	    this.template = template;
	  }

	  /**
	   * Generates a String from the given parameterHash.  Provide a hash of String keys to values.
	   * Keys can be regular text strings, in which case it will look for and replace #{key} as with the value.  You can
	   * also make the key a String "/key/", which will be converted to a Regex and run.
	   *
	   * For the value you can provide a straight up String, int, etc, or you can provide a function which will be called
	   * to provide the value
	   *
	   * @param {Object} parameters An object that contains the parameters for the template to replace.
	   * @returns {string} The result of executing the template.
	   */
	  generate(parameters) {
	    parameters = Utils.isDefined(parameters) ? parameters : {};
	    let templateCopy = String(this.template);
	    for (let key in parameters) {
	      if (parameters.hasOwnProperty(key)) {
	        const value = parameters[key];
	        let expressedValue;
	        if (typeof(value) === 'function') {
	          expressedValue = value();
	        } else {
	          expressedValue = value;
	        }

	        if (key.indexOf('/') === 0 && key.lastIndexOf('/') === key.length - 1) {
	          templateCopy = templateCopy.replace(new RegExp(key.substring(1, key.length - 1), "g"), expressedValue);
	        } else {
	          const expressedKey = "#{" + key + "}";
	          while (templateCopy.indexOf(expressedKey) !== -1) {
	            templateCopy = templateCopy.replace(expressedKey, expressedValue);
	          }
	        }
	      }
	    }

	    return templateCopy;
	  }

	  /**
	   * Calls to generate and then appends the resulting value to the inner HTML of the provided primeElement.
	   *
	   * @param {PrimeElement} primeElement The prime Element instance to append the result of executing the template to.
	   * @param {Object} parameters An object that contains the parameters for the template to replace.
	   */
	  appendTo(primeElement, parameters) {
	    if (Utils.isDefined(primeElement)) {
	      primeElement.setHTML(primeElement.getHTML() + this.generate(parameters));
	    } else {
	      throw new TypeError('Please supply an element to append to');
	    }
	  }

	  /**
	   * Calls to generate and then inserts the resulting elements into the dom before the primeElement
	   *
	   * @param {PrimeElement} primeElement The prime Element instance to insert the result of executing the template before.
	   * @param {Object} parameters An object that contains the parameters for the template to replace.
	   */
	  insertBefore(primeElement, parameters) {
	    if (Utils.isDefined(primeElement)) {
	      const holder = document.createElement('div');
	      holder.innerHTML = this.generate(parameters);
	      new PrimeElement(holder.children[0]).insertBefore(primeElement);
	    } else {
	      throw new TypeError('Please supply an element to append to');
	    }
	  }

	  /**
	   * Calls to generate and then inserts the resulting elements into the dom after the primeElement
	   *
	   * @param {PrimeElement} primeElement The prime Element instance to insert the result of executing the template after.
	   * @param {Object} parameters An object that contains the parameters for the template to replace.
	   */
	  insertAfter(primeElement, parameters) {
	    if (Utils.isDefined(primeElement)) {
	      const holder = document.createElement('div');
	      holder.innerHTML = this.generate(parameters);
	      new PrimeElement(holder.children[0]).insertAfter(primeElement);
	    } else {
	      throw new TypeError('Please supply an element to append to');
	    }
	  }
	}

	/*
	 * Copyright (c) 2017, Inversoft Inc., All Rights Reserved
	 *
	 * Licensed under the Apache License, Version 2.0 (the "License");
	 * you may not use this file except in compliance with the License.
	 * You may obtain a copy of the License at
	 *
	 *   http://www.apache.org/licenses/LICENSE-2.0
	 *
	 * Unless required by applicable law or agreed to in writing,
	 * software distributed under the License is distributed on an
	 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
	 * either express or implied. See the License for the specific
	 * language governing permissions and limitations under the License.
	 */

	// Do any polyfill imports here for backwards compatibility
	const Ajax = {
	  Request: PrimeRequest
	};

	const Data = {
	  Queue: DataQueue
	};

	exports.Effects = Effects;
	exports.Widgets = Widgets;
	exports.Ajax = Ajax;
	exports.Data = Data;
	exports.Browser = Browser;
	exports.Events = Events;
	exports.Date = PrimeDate;
	exports.Document = PrimeDocument;
	exports.Storage = PrimeStorage;
	exports.Window = PrimeWindow;
	exports.Template = Template;
	exports.Utils = Utils;

	return exports;

}({}));
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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