(ns leiningen.new.elbenwald-template
  (:require
    [clojure.core.cache :refer [ttl-cache-factory]]
    [clojure.java.io :as io]
    [clojure.string :as s]
    [leiningen.core.main :as main]
    [leiningen.core.utils :as utils]
    [leiningen.new.templates :refer [->files *dir* name-to-path year date renderer sanitize-ns]]
    [me.raynes.fs :as fs]
    [stencil.loader :refer [set-cache]])
  (:import (org.springframework.core.io.support PathMatchingResourcePatternResolver)))

(def render (renderer "elbenwald-template"))

;TODO testing purpose, delete after testing
; (def sub-dir-to-copy "leiningen/new/elbenwald_template/app/resources/public/")
; (def p (PathMatchingResourcePatternResolver.))
; (def list-public-resources (.getResources p (str "classpath:" sub-dir-to-copy "**/*.*")))



(defn format-features [features]
  (apply str (interpose ", " features)))

;TODO: use partials to reuse code parts
;TODO: remove argument "data" at render function usages where it is not necessary
(defn generate-project-lib
  "Create a new Elbenwald project (library)"
  [data]
  (main/info "Generating fresh 'lein new' elbenwald project (lib).")
  (->files data
           [".gitignore" (render "all/gitignore" data)]
           ["CHANGELOG.md" (render "all/CHANGELOG.md" data)]
           ["LICENSE" (render "all/LICENSE" data)]
           ["README.md" (render "all/README.md" data)]
           ["project.clj" (render "all/project.clj" data)]
           ["doc/intro.md" (render "all/doc/intro.md" data)]
           ["resources/.gitkeep" (render "lib/resources/gitkeep" data)]
           ["env/dev/clj/user.clj" (render "lib/env/dev/clj/user.clj" data)]
           ["env/dev/resources/logback.xml" (render "lib/env/dev/resources/logback.xml" data)]
           ["env/test/resources/logback.xml" (render "lib/env/test/resources/logback.xml" data)]
           ["src/{{sanitized}}/core.clj" (render "lib/src/core.clj" data)]
           ["src/{{sanitized}}/incubator.clj" (render "lib/src/incubator.clj" data)]
           ["src/{{sanitized}}/protocols.clj" (render "lib/src/protocols.clj" data)]
           ["src/{{sanitized}}/utils.clj" (render "lib/src/utils.clj" data)]
           ["test/{{sanitized}}/core_test.clj" (render "all/test/core_test.clj" data)]
           ["test/{{sanitized}}/incubator_test.clj" (render "all/test/incubator_test.clj" data)]
           ["test/{{sanitized}}/test_utils.clj" (render "all/test/test_utils.clj" data)]
           ["test/{{sanitized}}/utils_test.clj" (render "all/test/utils_test.clj" data)]))

(defn generate-project-cli
  "Create a new Elbenwald project (command-line-interface)"
  [data]
  (main/info "Generating fresh 'lein new' elbenwald project (cli).")
  (->files data

           [".gitignore" (render "cli/gitignore" data)]
           ["CHANGELOG.md" (render "all/CHANGELOG.md" data)]
           ["dev-config.edn" (render "app/dev-config.edn" data)]
           ["LICENSE" (render "all/LICENSE" data)]
           ["profiles.clj" (render "app/profiles.clj" data)]
           ["project.clj" (render "cli/project.clj" data)]
           ["README.md" (render "all/README.md" data)]
           ["test-config.edn" (render "app/test-config.edn" data)]

           ["doc/intro.md" (render "cli/doc/intro.md" data)]

           ["env/dev/clj/{{sanitized}}/env.clj" (render "cli/env/dev/clj/env.clj" (assoc data :profile "dev"))]
           ["env/dev/clj/user.clj" (render "cli/env/dev/clj/user.clj" data)]
           ["env/dev/resources/logback.xml" (render "cli/env/dev/resources/logback.xml" data)]
           ["env/prod/clj/{{sanitized}}/env.clj" (render "cli/env/dev/clj/env.clj" (assoc data :profile "prod"))]
           ["env/test/resources/logback.xml" (render "cli/env/test/resources/logback.xml" data)]

           ["resources/.gitkeep" (render "cli/resources/gitkeep" data)]

           ["src/{{sanitized}}/app.clj" (render "cli/src/app.clj" data)]
           ["src/{{sanitized}}/config.clj" (render "cli/src/config.clj" data)]
           ["src/{{sanitized}}/core.clj" (render "cli/src/core.clj" data)]
           ["src/{{sanitized}}/global.clj" (render "cli/src/global.clj" data)]
           ["src/{{sanitized}}/incubator.clj" (render "cli/src/incubator.clj" data)]
           ["src/{{sanitized}}/protocols.clj" (render "cli/src/protocols.clj" data)]
           ["src/{{sanitized}}/utils.clj" (render "cli/src/utils.clj" data)]

           ["test/{{sanitized}}/core_test.clj" (render "cli/test/core_test.clj" data)]
           ["test/{{sanitized}}/incubator_test.clj" (render "cli/test/incubator_test.clj" data)]
           ["test/{{sanitized}}/test_utils.clj" (render "cli/test/test_utils.clj" data)]
           ["test/{{sanitized}}/utils_test.clj" (render "cli/test/utils_test.clj" data)]))

(defn generate-project-app
  "Create a new Elbenwald project (luminus-web-app)"
  [data]
  (main/info "Generating fresh 'lein new' elbenwald project (app).")
  (->files data

           ; files

           [".gitignore" (render "app/gitignore" data)]
           ["Capstanfile" (render "app/Capstanfile" data)]
           ["CHANGELOG.md" (render "all/CHANGELOG.md" data)]
           ["dev-config.edn" (render "app/dev-config.edn" data)]
           ["Dockerfile" (render "app/Dockerfile" data)]
           ["LICENSE" (render "all/LICENSE" data)]
           ["Procfile" (render "app/Procfile" data)]
           ["profiles.clj" (render "app/profiles.clj" data)]
           ["project.clj" (render "app/project.clj" data)]
           ["README.md" (render "all/README.md" data)]
           ["test-config.edn" (render "app/test-config.edn" data)]

           ["doc/intro.md" (render "app/doc/intro.md" data)]

           ["env/dev/clj/{{sanitized}}/env_middleware.clj" (render "app/env/dev/clj/env_middleware.clj" data)]
           ["env/dev/clj/{{sanitized}}/env.clj" (render "app/env/dev/clj/env.clj" data)]
           ["env/dev/clj/{{sanitized}}/figwheel.clj" (render "app/env/dev/clj/figwheel.clj" data)]
           ["env/dev/clj/user.clj" (render "app/env/dev/clj/user.clj" data)]
           ["env/dev/cljs/{{sanitized}}/app.cljs" (render "app/env/dev/cljs/app.cljs" data)]
           ["env/dev/resources/config.edn" (render "app/env/dev/resources/config.edn" data)]
           ["env/dev/resources/logback.xml" (render "app/env/dev/resources/logback.xml" data)]
           ["env/prod/clj/{{sanitized}}/env.clj" (render "app/env/prod/clj/env.clj" data)]
           ["env/prod/cljs/{{sanitized}}/app.cljs" (render "app/env/prod/cljs/app.cljs" data)]
           ["env/prod/resources/config.edn" (render "app/env/prod/resources/config.edn" data)]
           ["env/prod/resources/logback.xml" (render "app/env/prod/resources/logback.xml" data)]
           ["env/test/resources/config.edn" (render "app/env/test/resources/config.edn" data)]
           ["env/test/resources/logback.xml" (render "app/env/test/resources/logback.xml" data)]

           ["externs/jquery-3.2.js" (render "app/externs/jquery-3.2.js")]

           ["resources/docs/docs.md" (render "app/resources/docs/docs.md" data)]
           ["resources/graphql/schema.edn" (render "app/resources/graphql/schema.edn")]
           ["resources/html/auth.html" (render "app/resources/html/auth.html")]
           ["resources/html/error.html" (render "app/resources/html/error.html")]
           ["resources/html/graphiql.html" (render "app/resources/html/graphiql.html")]
           ["resources/html/home.html" (render "app/resources/html/home.html" data)]
           ["resources/html/login.html" (render "app/resources/html/login.html")]

           ["src/clj/{{sanitized}}/app.clj" (render "app/src/clj/app.clj" data)]
           ["src/clj/{{sanitized}}/config.clj" (render "app/src/clj/config.clj" data)]
           ["src/clj/{{sanitized}}/core.clj" (render "app/src/clj/core.clj" data)]
           ["src/clj/{{sanitized}}/global.clj" (render "app/src/clj/global.clj" data)]
           ["src/clj/{{sanitized}}/handler.clj" (render "app/src/clj/handler.clj" data)]
           ["src/clj/{{sanitized}}/incubator.clj" (render "app/src/clj/incubator.clj" data)]
           ["src/clj/{{sanitized}}/layout.clj" (render "app/src/clj/layout.clj" data)]
           ["src/clj/{{sanitized}}/middleware.clj" (render "app/src/clj/middleware.clj" data)]
           ["src/clj/{{sanitized}}/middleware/exception.clj" (render "app/src/clj/middleware/exception.clj" data)]
           ["src/clj/{{sanitized}}/middleware/formats.clj" (render "app/src/clj/middleware/formats.clj" data)]
           ["src/clj/{{sanitized}}/nrepl.clj" (render "app/src/clj/nrepl.clj" data)]
           ["src/clj/{{sanitized}}/protocols.clj" (render "app/src/clj/protocols.clj" data)]
           ["src/clj/{{sanitized}}/routes/home.clj" (render "app/src/clj/routes/home.clj" data)]
           ["src/clj/{{sanitized}}/routes/services.clj" (render "app/src/clj/routes/services.clj" data)]
           ["src/clj/{{sanitized}}/routes/services/graphql.clj" (render "app/src/clj/routes/services/graphql.clj" data)]
           ["src/clj/{{sanitized}}/scheduling.clj" (render "app/src/clj/scheduling.clj" data)]
           ["src/clj/{{sanitized}}/utils.clj" (render "app/src/clj/utils.clj" data)]
           ["src/cljc/{{sanitized}}/i18n.cljc" (render "app/src/cljc/i18n.cljc" data)]
           ["src/cljc/{{sanitized}}/i18n/de.cljc" (render "app/src/cljc/i18n/de.cljc" data)]
           ["src/cljc/{{sanitized}}/i18n/en.cljc" (render "app/src/cljc/i18n/en.cljc" data)]
           ["src/cljc/{{sanitized}}/queries.cljc" (render "app/src/cljc/queries.cljc" data)]
           ["src/cljc/{{sanitized}}/utilsc.cljc" (render "app/src/cljc/utilsc.cljc" data)]
           ["src/cljc/{{sanitized}}/validation.cljc" (render "app/src/cljc/validation.cljc" data)]
           ["src/cljs/{{sanitized}}/ajax.cljs" (render "app/src/cljs/ajax.cljs" data)]
           ["src/cljs/{{sanitized}}/core.cljs" (render "app/src/cljs/core.cljs" data)]
           ["src/cljs/{{sanitized}}/events.cljs" (render "app/src/cljs/events.cljs" data)]
           ["src/cljs/{{sanitized}}/pages/data.cljs" (render "app/src/cljs/pages/data.cljs" data)]
           ["src/cljs/{{sanitized}}/utils.cljs" (render "app/src/cljs/utils.cljs" data)]

           ["test/clj/{{sanitized}}/core_test.clj" (render "app/test/core_test.clj" data)]
           ["test/clj/{{sanitized}}/handler_test.clj" (render "app/test/clj/handler_test.clj" data)]
           ["test/clj/{{sanitized}}/incubator_test.clj" (render "app/test/clj/incubator_test.clj" data)]
           ["test/clj/{{sanitized}}/test_utils.clj" (render "app/test/clj/test_utils.clj" data)]
           ["test/clj/{{sanitized}}/utils_test.clj" (render "app/test/utils_test.clj" data)]
           ["test/cljs/{{sanitized}}/core_test.cljs" (render "app/test/cljs/core_test.cljs" data)]
           ["test/cljs/{{sanitized}}/doo_runner.cljs" (render "app/test/cljs/doo_runner.cljs" data)])

  ;WARNING: lein new template --to-dir xxx => xxx must equal short project name
  (let [p (PathMatchingResourcePatternResolver.)
        resources-path "resources/public/"
        list-public-resources (.getResources p (str "classpath:leiningen/new/elbenwald_template/app/" resources-path "**/*.*"))
        resources-prefix-length (-> (first list-public-resources)
                                  .getPath
                                  (s/last-index-of resources-path))
        target-path (System/getProperty "leiningen.original.pwd")]
    (doseq [resource list-public-resources]
      (let [resource-path (.getPath resource)
            target-file (fs/file target-path
                                 (data :name-short)
                                 (subs resource-path resources-prefix-length))
            target-parent (fs/parent target-file)]
        (when-not (fs/exists? target-parent)
          (fs/mkdirs target-parent))
        (with-open [input-stream (.getInputStream resource)]
          (io/copy input-stream target-file))))))

(defn generate-project-test
  "Create a new Elbenwald project"
  [data]
  (main/info "Generating fresh 'lein new' elbenwald project (test).")
  (->files data
           ["whole.clj" (render "test/from_parts.clj" data)])) ;constructing file from parts using mustache partials


;only one of the features +lib, +cli, +app should be picked
(defn elbenwald-template
  "Generates an elbenwald lein project."
  [name & feature-params]
  (let [supported-features #{;;project types
                             "+cli" "+app" "+lib" "+test"}
                             ;;misc
                             ;"+cljs"}

        ;ttl cache is needed to enable mustache partial-functionality with stencil
        ;https://github.com/davidsantiago/stencil#manual-cache-management
        cache (set-cache (ttl-cache-factory {}))

        unsupported (-> (set feature-params)
                      (clojure.set/difference supported-features)
                      (not-empty))
        data {:date          (date)
              :features      (set feature-params)
              :name          name
              :name-short    (-> name s/reverse (s/split #"\.") first s/reverse)
              :ns-name       (sanitize-ns name)
              :project-name  (-> name s/reverse (s/replace-first \. \/) s/reverse)
              :sanitized     (name-to-path name)
              :year          (year)}]
    (cond
      unsupported
      (main/info "Unrecognized options:" (format-features unsupported)
                 "\nSupported options are:" (format-features supported-features))

      (contains? (:features data) "+lib")
      (generate-project-lib (assoc data :lib true))

      (contains? (:features data) "+cli")
      (generate-project-cli (assoc data :cli true))

      (contains? (:features data) "+app")
      (generate-project-app (assoc data :app true))

      (contains? (:features data) "+test")
      (generate-project-test data))))


(comment
  (ns leiningen.new.elbenwald-template)

  (let [p (PathMatchingResourcePatternResolver.)
        resources-path "resources/public/"
        list-public-resources (.getResources p (str "classpath:leiningen/new/elbenwald_template/app/" resources-path "**/*.*"))
        resources-prefix-length (-> (first list-public-resources)
                                  .getPath
                                  (s/last-index-of resources-path))
        target-path (System/getProperty "leiningen.original.pwd")]
    (doseq [resource (take 1 list-public-resources)]
      (let [resource-path (.getPath resource)
            target-file (fs/file target-path
                                 "(data :name-short)"
                                 (subs resource-path resources-prefix-length))
            target-parent (fs/parent target-file)]
        (prn resources-prefix-length resource-path target-file))))
        ; (when (-> target-path fs/directory? not)
        ;   (utils/mkdirs target-path))
        ; (with-open [xin (.getInputStream res)]
        ;   (io/copy xin (io/file target-file)))))))

  (let [p (PathMatchingResourcePatternResolver.)
        list-public-resources (.getResources p "classpath:leiningen/new/elbenwald_template/app/resources/public/**/*.*")
        sub-str "resources/public/"
        index-from (-> (first list-public-resources)
                     .getPath
                     (s/last-index-of sub-str))]
    (doseq [res (take 1  list-public-resources)]
      (let [full-path (.getPath res)
            target-file (fs/file (System/getProperty "leiningen.original.pwd")
                                 "(data :name-short)"
                                 (subs full-path index-from))
            target-parent (fs/parent target-file)]
        (prn index-from full-path target-file)))))
        ; (when-not (fs/exists? target-parent)
        ;   (fs/mkdirs target-parent))
        ; (with-open [xin (.getInputStream res)]
        ;   (io/copy xin target-file))))))
