(ns embang.ctlmh
  "Control LMH"
  (:refer-clojure :exclude [rand rand-int rand-nth])
  (:use [embang 
         [state :only [set-log-weight]]
         inference
         lmh]))

(derive ::algorithm ::embang.lmh/algorithm)

(defmethod infer :ctlmh [_ prog value & {}]
  (letfn
    [(sample-seq [state]
       (lazy-seq
         (let [;; Choose uniformly a random choice to resample.
               entry (rand-nth (get-trace state))
               ;; Compute next state from the resampled choice.
               next-state (next-state state entry)
               ;; Reconstruct the current state through transition back
               ;; from the next state; the rdb will be different.
               prev-state (prev-state state next-state entry)
               ;; Apply Metropolis-Hastings acceptance rule to select
               ;; either the new or the current state.
               state (if (or 
                           ;; Always accept proposed noise.
                           (:noise (meta (first (:choice-id entry))))
                           ;; Otherwise accept with MH propability.
                           (> (- (utility next-state)
                                 (utility prev-state))
                              (Math/log (rand))))
                       next-state
                       state)]
           ;; Include the selected state into the sequence of samples,
           ;; setting the weight to the unit weight.
           (cons (set-log-weight state 0.) (sample-seq state)))))]

    (let [state (:state (exec ::algorithm prog value initial-state))]
      (if (seq (get-trace state))
        (sample-seq state)
        ;; No randomness in the program.
        (repeat (set-log-weight state 0.))))))
