(ns embelyon.codex.core
  "This module should be the only curio required to elucidate the many
  mysteries of the CloudFormation. Require it, then delve as deep as your
  curiosity will allow."
  (:require [embelyon.codex.spec :as cs]
            [embelyon.codex.impl.specification :as sp]
            [embelyon.codex.impl.specification.spec :as sps]
            [embelyon.codex.impl.data :as data]
            [clojure.spec.alpha :as s]))

;;; Core exploration API

(defn specification-data
  "Return a CloudFormation specification as a map"
  ([opts]
   (sp/specification-data opts))
  ([]
   (sp/specification-data)))

(s/fdef specification-data
  :args (s/cat :opts (s/? ::cs/options))
  :ret  ::sps/cloud-formation-spec)

(defn update-specification!
  "Force an update of the specification data for an AWS region"
  [region]
  (data/update-specification! region))

(s/fdef update-specification!
  :args (s/cat :region ::sps/region)
  :ret  nil?)

(defn resource-specification-version
  "Returns the specification version for a region"
  ([opts]
   (:ResourceSpecificationVersion (specification-data opts)))
  ([]
   (resource-specification-version {})))

(s/fdef resource-specification-version
  :args (s/cat :opts (s/? ::cs/options))
  :ret  string?)

(defn property-types
  "Returns the specification version for a region"
  ([opts]
   (:PropertyTypes (specification-data opts)))
  ([]
   (property-types {})))

(s/fdef property-types
  :args (s/cat :opts (s/? ::cs/options))
  :ret  ::sps/PropertyTypes)

(defn resource-types
  "Return resource types available to an AWS region"
  ([opts]
   (sp/resource-types opts))
  ([]
   (sp/resource-types)))

(s/fdef resource-types
  :args (s/cat :opts (s/? ::cs/options))
  :ret  ::sps/ResourceTypes)

(defn service-names
  "List AWS service names for the given AWS Region"
  ([opts]
   (sp/service-names opts))
  ([]
   (sp/service-names)))

(s/fdef service-names
  :args (s/cat :opts (s/? ::cs/options))
  :ret  (s/coll-of string?))

(defn valid-service?
  "Is the given service name a valid cloud formation service?"
  ([service-name opts]
   (sp/valid-service? service-name opts))
  ([service-name]
   (sp/valid-service? service-name)))

(s/fdef valid-service?
  :args (s/cat :service-name string?)
  :ret  boolean?)

(defn service
  "Get a full blown service specification for a service identified by name. The given service must be supported by the target AWS Region"
  ([service-name opts]
   (sp/service service-name opts))
  ([service-name]
   (sp/service service-name)))

(s/fdef service
  :args (s/cat :service-name string? :opts (s/? ::cs/options))
  :ret  (s/nilable ::sps/service))

(defn resource-ids
  "Get all resources for a cloud formation service"
  [service]
  (sp/resource-ids service))

(s/fdef resource-ids
  :args (s/cat :service ::sps/service)
  :ret  (s/coll-of keyword?))

(defn resource
  "Get the specification for a specific AWS resource. ID will be added
  as meta so we stop oursevles from tampering with the shape of the
  AWS specification"
  ([id opts]
   (sp/resource id opts))
  ([id]
   (sp/resource id)))

(s/fdef resource
  :args (s/cat :id ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::sps/resource))

(defn property-ids
  "Get all properties of a given resource"
  [resource]
  (sp/property-ids resource))

(s/fdef property-ids
  :args (s/cat :resource ::sps/resource)
  :ret  (s/coll-of keyword?))

(defn property-type
  "Returns a property type for the AWS region"
  ([property-type-id opts]
   (sp/property-type property-type-id opts))
  ([property-type-id]
   (sp/property-type property-type-id)))

(s/fdef property-type
  :args (s/cat :type-id ::cs/id :opts (s/? ::cs/options))
  :ret  (s/nilable ::sps/property-type))

(defn item-property-type
  "Given a resource or property type, return detailed information about a property
  of a resource or property type"
  ([type property opts]
   (sp/item-property-type type property opts))
  ([type property]
   (sp/item-property-type type property)))

(s/fdef item-property-type
  :args (s/cat
          :type
          (s/or
            :property-type ::sps/property-type
            :resource ::sps/resource)
          :property ::sps/property
          :opts (s/? ::cs/options))
  :ret (s/nilable ::sps/property-type))

(defn property
  "Returns the named property of the given resource or property type"
  [item key]
  (get-in item [:Properties key]))

(s/fdef property
  :args (s/cat
          :item
          (s/or :resource ::sps/resource :property-type ::sps/property-type)
          :key keyword?)
  :ret (s/nilable ::sps/property))

(defn resources
  "Returns a sequence of resources for the given AWS service"
  ([service-name opts]
   (if-some [service (service service-name opts)]
     (some->>
       (resource-ids service)
       (map #(resource % opts)))
     '()))
  ([service-name]
   (resources service-name {})))

(s/fdef resources
  :args (s/cat :service-name ::cs/id :opts (s/? ::cs/options))
  :ret  (s/coll-of ::sps/resource))
