(ns embelyon.codex.impl.data
  "The data module contains functions for fetching and reading region
  based cloud formation specifications"
  (:require [clojure.java.io :as io]
            [org.httpkit.client :as client]
            [cheshire.core :as json])
  (:import (java.io File)))

;;; @todo Find CloudFormation specs for regions in China
;;; not listed here at time of writing: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/cfn-resource-specification.html
(def regions
  {:us-east-1      "https://d1uauaxba7bl26.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :us-east-2      "https://dnwj8swjjbsbt.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :us-west-1      "https://d68hl49wbnanq.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :us-west-2      "https://d201a2mn26r7lk.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-south-1     "https://d2senuesg1djtx.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-northeast-1 "https://d33vqc0rt9ld30.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-northeast-2 "https://d1ane3fvebulky.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-northeast-3 "https://d2zq80gdmjim8k.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-southeast-1 "https://doigdx0kgq9el.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ap-southeast-2 "https://d2stg8d246z9di.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :ca-central-1   "https://d2s8ygphhesbe7.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :eu-central-1   "https://d1mta8qj7i28i2.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :eu-west-1      "https://d3teyb21fexa9r.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :eu-west-2      "https://d1742qcu2c1ncx.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :eu-west-3      "https://d2d0mfegowb3wk.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :eu-north-1     "https://diy8iv58sj6ba.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"
   :sa-east-1      "https://d3c9jyj3w509b0.cloudfront.net/latest/gzip/CloudFormationResourceSpecification.json"})

(defn- exists?
  [path]
  (.exists (io/as-file path)))

(defn- url
  [region]
  (get regions region))

(defn- get-file-path
  [region]
  (-> (System/getProperty "user.home")
      (str File/separator)
      (str ".")
      (str (name region))
      (str ".scout.json")))

(defn- get-file
  [region]
  (-> (get-file-path region)
      (io/as-file)))

(defn- copy
  [region file]
  (some->
    (url region)
    (client/get {:as :byte-array})
    (deref)
    (:body)
    (io/copy file)))

(defn- read-data'
  [path]
  (let [reader (io/reader path)]
    (json/parse-stream reader true)))

(def read-data'' (memoize read-data'))

(defn- env
  [var-name default]
  (or (System/getenv var-name) default))

(defn region-from-opts
  "Handles getting a region from an options map. This function
  should be responsible for handling the default region"
  [opts]
  (keyword (or (:region opts) (env "AWS_REGION" :us-east-2))))

(defn read-data
  "Read json data - inferring which specification to use from
  the given options. json reading is memoized"
  [opts]
  (let [region (region-from-opts opts)
        path   (get-file-path region)]
    (read-data'' path)))

(defn update-specification!
  "Force an update of a region specification file"
  ([region file]
   (when-not (contains? regions region)
     (throw
       (ex-info "CloudFormation specification not found"
         {:region region})))
   (copy region file))
  ([region]
   (update-specification! region (get-file region))))

(defn download!
  "Fetches the specification file if it does not exist"
  [region]
  (let [file (get-file region)]
    (when-not (exists? file)
      (update-specification! region file))
    (get-file-path region)))


