(ns embelyon.codex.impl.specification
  "Defines core functions for working with cloud formation specification data"
  (:require [embelyon.codex.impl.data :as data]
            [clojure.string :as string]
            [clojure.spec.alpha :as s]))

(defmacro defcodexfn
  "Creates a lore function. A lore function handles options
  in the tail end of the argument list and ensures a specification
  is downloaded and ready for use in the function body. Use
  read-data with opts to get the json for the region specification"
  [name comment args & body]
  (let [no-opts (vec (take (- (count args) 1) args))]
    `(defn ~name
       ~comment
       (~args
        (let [opts#   (last ~args)
              region# (data/region-from-opts opts#)]
          (data/download! region#)
          ~@body))
       (~no-opts
        (~name ~@no-opts {})))))

(defcodexfn specification-data
  "Returns cloud formation specification data as a map. Infers region from opts or AWS_REGION env
  var. Defaults to :us-east-2"
  [opts]
  (data/read-data opts))

(defn resource-types
  "Returns the specification version for a region"
  ([opts]
   (:ResourceTypes (specification-data opts)))
  ([]
   (resource-types {})))

(defn- get* [key map] (get map key))

;;; Services

(defn- service-name
  "Extract the service name from a resource type name"
  [resource-type-name]
  (-> resource-type-name
      name
      (string/split #"::")
      drop-last
      (as-> service (string/join "::" service))))

(defn- resources-by-service
  "Return resources grouped by the services they belong to"
  ([opts]
   (->> opts
        resource-types
        (group-by (fn [[k v]] (service-name k)))
        (reduce-kv (fn [m k v] (assoc m k (into {} v))) {})))
  ([]
   (resources-by-service {})))

(defn service-names
  ([opts]
   (->> opts
        resource-types
        keys
        (map service-name)
        set))
  ([]
   (service-names {})))

(defn valid-service?
  "Is the given service name a valid cloud formation service?"
  ([service-name opts]
   (-> (service-names opts)
       (contains? service-name)))
  ([service-name]
   (valid-service? service-name {})))

(defn service
  "Get the specification for an AWS service by name"
  ([service-name opts]
   {:pre [(valid-service? service-name opts)]}
   (->> (resources-by-service opts)
        (get* service-name)
        (assoc {:name service-name} :resources)))
  ([service-name]
   (service service-name {})))

(defn resource-ids
  "Get all resources for a cloud formation service"
  [service]
  (keys (:resources service)))

(defn property-ids
  "Get all properties of a given resource"
  [resource]
  (keys (:Properties resource)))

(defn- type-with-meta
  "Return a type with it's id attached as meta"
  [type-key id opts]
  (some->
    (specification-data opts)
    (get type-key)
    (get (keyword id))
    (with-meta {:id   id
                :type (if (= :ResourceTypes type-key)
                        :resource
                        :property-type)})))

(defn resource
  "Get information about a specific AWS resource. ID will be added
  as meta to prevent tampering with the shape of the AWS specification"
  ([id opts]
   (type-with-meta :ResourceTypes id opts))
  ([id]
   (resource id {})))

(defn property-type
  "Return a property type from data"
  ([property-type-id opts]
   (type-with-meta :PropertyTypes property-type-id opts))
  ([property-type-id]
   (property-type property-type-id {})))

(defn prefix
  "Prefix an id with a resource name. This is useful for properties keyed under resources - such as 
  AWS::S3::Bucket.BucketEncryption"
  [type id]
  (some-> type
          meta
          :id
          name
          (string/replace #"[.].+", "")
          (str "." (name id))
          keyword))

(defn- first-entry
  [ks m]
  (reduce
    (fn [r k]
      (or r (get m k))) nil ks))

(defn item-property-type
  "Get a property type for a resource or property type's property"
  ([type property opts]
   (some->>
     property
     (first-entry [:ItemType :Type])
     (prefix type)
     (property-type)))
  ([type property]
   (item-property-type type property {})))
