(ns er-cassandra.model.delete
  (:require
   [clojure.set :as set]
   [clojure.core.match :refer [match]]
   [manifold.deferred :as d]
   [cats.core :as m :refer [with-monad mlet return]]
   [cats.monad.deferred :as dm :refer [deferred-monad]]
   [qbits.alia :as alia]
   [qbits.alia.manifold :as aliam]
   [qbits.hayt :as h]
   [er-cassandra.key :as k]
   [er-cassandra.record :as r]
   [er-cassandra.model.types :as t]
   [er-cassandra.model.util :refer [combine-responses create-lookup-record]]
   [er-cassandra.model.unique-key :as unique-key]
   [er-cassandra.model.upsert :as upsert]
   [er-cassandra.model.select :as select]
   [er-cassandra.model.util :as util])
  (:import [er_cassandra.model.types Model]))

(defn delete
  "delete a single instance, removing primary, secondary unique-key and
   lookup records "

  ([session ^Model model key record-or-key-value]

   (with-monad deferred-monad
     (mlet [record (select/select-one session
                                      model
                                      key
                                      record-or-key-value)

            primary-response (if record
                               (upsert/delete-record
                                session
                                model
                                (:primary-table model)
                                (t/extract-uber-key-value
                                 model
                                 record))
                               (return nil))

            unique-responses (if record
                               (unique-key/release-stale-unique-keys
                                session
                                model
                                record
                                nil)
                               (return nil))

            secondary-responses (if record
                                  (upsert/delete-stale-secondaries
                                   session
                                   model
                                   record
                                   nil)
                                  (return nil))

            lookup-responses (if record
                               (upsert/delete-stale-lookups
                                session
                                model
                                record
                                nil)
                               (return nil))]
       (return
        [:ok record :deleted])))))

(defn delete-many
  "issue one delete query for each record and combine the responses"
  [session ^Model model key records]
  (->> records
       (map (fn [record]
              (delete session model key record)))
       util/combine-responses))
