(ns er-model.groups.model
  (:require
   [clojure.string :as str]
   [plumbing.core :refer :all]
   [schema.core :as s]
   [clj-uuid :as uuid]
   [clojure.set :as set]
   [cats.core :refer [mlet return]]
   [cats.context :refer [with-context]]
   [cats.labs.manifold :refer [deferred-context]]
   [er-cassandra.model :as model]
   [er-cassandra.model.types :as t]
   [er-cassandra.model.util :as util]
   [er-cassandra.model.callbacks.search-key-callback :as skey]
   [er-cassandra.model.callbacks.sort-name-callback :as sortnm]
   [er-model.users.model :as u]
   [er-model.orgs.model :as o]
   [er-model.groups.schema :as gs]))

(model/defmodel RawGroups
  {:callbacks {:before-save [(t/create-updated-at-callback)
                             (skey/create-search-keys-callback
                              :search_key
                              :name)
                             (sortnm/create-sort-name-callback
                              :sort_name
                              :name)]}
   :primary-table {:name :groups
                   :key [:org_id :id]}
   :lookup-key-tables [{:name :groups_by_visibility_sort_name
                        :key [[:org_id :visibility] :sort_name :id]}
                       {:name :groups_by_visibility_search_key
                        :key [[:org_id :visibility] :search_key :id]
                        :collections {:search_key :set}}]})

(def Groups
  (assoc-in RawGroups
            [:callbacks :after-load]
            [(t/create-filter-view-callback
              [:search_key])]))

(model/defmodel UserGroups
  {:callbacks {:before-save [(skey/create-search-keys-callback
                              :group_search_key
                              :group_name)
                             (sortnm/create-sort-name-callback
                              :group_sort_name
                              :group_name)]}
   :primary-table {:name :user_groups
                   :key [[:org_id :user_id] :group_id]}
   :lookup-key-tables [{:name :group_users
                        :key [[:org_id :group_id] :user_id]}
                       {:name :user_groups_by_group_sort_name
                        :key [[:org_id :user_id] :group_sort_name :group_id]}
                       {:name :user_groups_by_group_search_key
                        :key [[:org_id :user_id] :group_search_key :group_id]
                        :collections {:group_search_key :set}}]})

(defnk find-many-groups
  [session org ids]
  (let [org-id (t/extract-uber-key-value o/Orgs org)]
    (model/select-many
     session
     Groups
     [:org_id :id]
     (map (fn [gid] [org-id gid])
          ids))))

(defnk find-public-for-org
  [session org]
  (with-context deferred-context
    (mlet [org-id (return (last (t/extract-uber-key-value o/Orgs org)))]
      (model/select session
                    Groups
                    [:org_id :visibility]
                    [org-id "public"]))))

(defnk find-public-groups-for-org
  "find public groups in an org.
   load too many then filter down because of repetition in index table"
  [session org {q ""} {limit 20}]
  (let [org-id (last (t/extract-uber-key-value o/Orgs org))
        q (str/trim (or q ""))
        q2 (str q "\uffff")]
    (with-context deferred-context
      (if (empty? q)
        (model/select session
                      Groups
                      [:org_id :visibility]
                      [org-id "public"]
                      {:from :groups_by_visibility_sort_name
                       :limit limit})
        (mlet [uf (model/select session
                                Groups
                                [:org_id :visibility]
                                [org-id "public"]
                                {:from :groups_by_visibility_search_key
                                 :where [[:>= :search_key q]
                                         [:<= :search_key q2]]
                                 :limit (* 2 limit)})
               f (return (distinct-by :id uf))]
          (return (take limit f)))))))

(defnk find-groups-for-org-and-user
  "find groups in an org a user belongs to"
  [session org user {q ""} {limit 20}]
  (let [org-id (last (t/extract-uber-key-value o/Orgs org))
        user-id (last (t/extract-uber-key-value u/Users user))
        q (str/trim (or q ""))
        q2 (str q "\uffff")]
    (with-context deferred-context
      (if (empty? q)
        (mlet [ugs (model/select session
                                 UserGroups
                                 [:org_id :user_id]
                                 [org-id user-id]
                                 {:from :user_groups_by_group_sort_name
                                  :limit limit})
               gids (return (map :group_id ugs))]
          (find-many-groups {:session session
                             :org org-id
                             :ids gids}))
        (mlet [ugs (model/select session
                                 UserGroups
                                 [:org_id :user_id]
                                 [org-id user-id]
                                 {:from :user_groups_by_group_search_key
                                  :where [[:>= :group_search_key q]
                                          [:<= :group_search_key q2]]
                                  :limit (* 2 limit)})
               dugs (return (take limit (distinct-by :group_id ugs)))
               gids (return (map :group_id dugs))]
          (find-many-groups {:session session
                             :org org-id
                             :ids gids}))))))

(defnk find-groups
  "find public and private groups"
  [session org user {q ""} {limit 20} :as params]
  (prn ["FIND-GROUPS" params])
  (with-context deferred-context
    (mlet [pgs (find-public-groups-for-org params)
           ugs (find-groups-for-org-and-user params)]
      (return
       (->> (into pgs ugs)
            (distinct-by :id)
            (sort-by #(some-> % :sort_name)))))))

(defnk find-users-for-group
  [session group]
  (with-context deferred-context
    (mlet [[org-id group-id] (return
                                (t/extract-uber-key-value Groups group))
             user-groups (model/select session
                                       UserGroups
                                       [:org_id :group_id]
                                       [org-id group-id])
             user-ids (return (map (fn [gu]
                                       (-> gu
                                           (dissoc :group_id)
                                           (dissoc :org_id)
                                           (set/rename-keys {:user_id :id})))
                                     user-groups))]
      (model/select-many session
                         u/Users
                         :id
                         user-ids))))

(defnk find-user-groups-for-group
  [session group]
  (with-context deferred-context
    (mlet [[org-id group-id] (return
                                (t/extract-uber-key-value Groups group))]
      (model/select session
                    UserGroups
                    [:org_id :group_id]
                    [org-id group-id]))))

(defnk add-user-to-group
  [session group user]
  (let [[org-id group-id] (t/extract-uber-key-value Groups group)
        user-id (last (t/extract-uber-key-value u/Users user))]
    (model/upsert session UserGroups {:org_id org-id
                                      :group_id group-id
                                      :user_id user-id
                                      :group_name (:name group)})))

(defnk upsert-group-with-user-ids
  [session group-with-user-ids]
  (s/validate gs/ClientGroupWithUserIdsSchema group-with-user-ids)
  (let [user-ids (:user_ids group-with-user-ids)
        group (dissoc group-with-user-ids :user_ids)

        org-id (:org_id group)
        group-id (or (:id group) (uuid/v1))
        group (assoc group :id group-id)

        user-groups (map (fn [uid]
                           {:org_id org-id
                            :group_id group-id
                            :user_id uid
                            :group_name (:name group)})
                         user-ids)]
    (when (nil? org-id)
      (throw (ex-info "no org-id" {:group-with-user-ids group-with-user-ids})))

    (with-context deferred-context
      (mlet [ur (model/upsert session
                              Groups
                              group)

             cgu (model/select session
                               UserGroups
                               [:org_id :group_id]
                               [org-id group-id])

             ugu (model/upsert-many session
                                    UserGroups
                                    user-groups)

             dids (return
                   (set/difference
                    (set user-ids)
                    (set (map :user_id cgu))))

             dgus (return (map (fn [gid]
                                [org-id gid])
                              dids))]

        (model/delete-many session
                           UserGroups
                           [:org_id :group_id]
                           dgus)
        (return (first ur))))))
