(ns er-model.system
  (:require [clojure.string :as str]
            [environ.core :refer [env]]
            [nrepl.embed :as nrepl]
            [yoyo]
            [yoyo.system :as ys]
            [yoyo.closeable :as yyc]

            [er-cassandra.session.alia :as alia-session]))

;; yoyo management of a basic system

(defonce system nil)
(defonce system-ready-promise-atom (atom (promise))) ;; need a promise at load

(defn environment
  []
  (or (env :env) "development"))

(def config
  {:environment (environment)

   :alia-opts {:contact-points (or (some->
                                    (env :cassandra-contact-points)
                                    (str/split #","))
                                   ["localhost"])
               :keyspace (or (env :cassandra-keyspace) "er")}})

(defn create-alia-session
  [{{:keys [contact-points keyspace] :as alia-opts} :alia-opts}]
  (alia-session/create-session alia-opts))

(defn resolve-system
  []
  @(ns-resolve 'er-model.system 'system))

(defn wait-system
  [ready-promise-atom-sym]
  "wait for the system to be constructed then return it"
  (-> (ns-resolve (symbol (namespace ready-promise-atom-sym))
                  (symbol (name ready-promise-atom-sym)))
      deref ; atom
      deref ; promise
      deref) ; val
  (resolve-system))

(defn with-ready
  [system ready-promise-atom-sym]
  (fn [latch]
    (system
     (fn [started-system]
       (let [ready-promise-atom-var (ns-resolve
                                     (symbol (namespace ready-promise-atom-sym))
                                     (symbol (name ready-promise-atom-sym)))
             ready-promise-atom @ready-promise-atom-var
             ready-promise @ready-promise-atom]

         (deliver ready-promise true)
         (latch started-system))))))

(defn model-system-map
  []
  "a limited system for use when running migrations"
  {:config config

   :cassandra (-> (yyc/with-closeable
                    "cassandra"
                    create-alia-session)
                  (ys/using {:alia-opts [:config :alia-opts]}))})

(def model-system
  (-> (ys/make-system model-system-map)
      (ys/with-system-put-to 'er-model.system/system)
      (with-ready 'er-model.system/system-ready-promise-atom)))

(defn start!
  ([] (start! 'er-model.system/model-system))
  ([system-fn]
   (yoyo/set-system-fn! system-fn)
   (yoyo/start!)))

(defn start-if-wait!
  ([] (start-if-wait! 'er-model.system/model-system))
  ([system-fn]
   (locking system-ready-promise-atom
     (when-not @(ns-resolve 'er-model.system 'system)
       (start! system-fn))
     (wait-system 'er-model.system/system-ready-promise-atom))))

(defn stop! []
  (reset! system-ready-promise-atom (promise))
  (yoyo/stop!))

(defn reload! []
  (reset! system-ready-promise-atom (promise))
  (yoyo/reload!))

(defn -main []
  (nrepl/start-nrepl! {:port (or (env :nrepl-port) 7889)})
  (start!))
