(ns er-model.users.model
  (:refer-clojure :exclude [find])
  (:require
   [plumbing.core :refer :all]

   [clj-uuid :as uuid]
   [cats.core :refer [return mlet]]
   [cats.context :refer [with-context]]
   [cats.labs.manifold :refer [deferred-context]]

   [buddy.hashers :as hashers]
   [er-cassandra.model :as model]
   [er-cassandra.model.types :as t]))

(defn create-encrypt-password-callback
  ([] (create-encrypt-password-callback
       :encrypted_password
       {:algorithm :pbkdf2+sha256}))

  ([encrypted-password-col encrypt-opts]
   (fn [r]
     (if-let [pw (:password r)]
       (-> r
           (dissoc :password)
           (assoc encrypted-password-col (hashers/encrypt pw encrypt-opts)))
       r))))

(model/defmodel RawUsers
  {:callbacks {:before-save [(create-encrypt-password-callback)
                             (t/create-protect-columns-callback
                              :update_password
                              :encrypted_password)
                             (t/create-updated-at-callback)]}
   :primary-table {:name :users
                   :key [:id]}
   :unique-key-tables [{:name :users_by_email
                        :key [:email]
                        :collections {:email :list}}
                       {:name :users_by_phone_no
                        :key [:phone_no]
                        :collections {:phone_no :list}}
                       {:name :users_by_device_id
                        :key [:device_id]
                        :collections {:device_id :set}}]
   :lookup-key-tables [{:name :users_by_confirmation_token
                        :key [:confirmation_token]}
                       {:name :users_by_login_token
                        :key [:login_token]}
                       {:name :users_by_reset_password_token
                        :key [:reset_password_token]}
                       ]})

(def Users
  (assoc-in RawUsers
            [:callbacks :after-load]
            [(t/create-select-view-callback
              [:id :updated_at])]))

(defn check-password
  [password user-record]
  (hashers/check password (:encrypted_password user-record)))

(defnk find-for-key
  [session key value]
  (model/select-one session Users key value))

(defnk find-and-check-password
  [session key key-value password]
  (with-context deferred-context
    (mlet [user (model/select-one session RawUsers key key-value)
           user-view (first (t/run-callbacks Users :after-load [user]))]
      (return
       (if (check-password password user)
         user-view
         nil)))))

(defnk find-many
  [session ids]
  (model/select-many session Users :id ids))
